/**
 * @file ProtocolConnect.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Implementation of generic protocol connect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic protocol connect state machine.
 */

#ifndef _PROTOCOL_CONNECT_H_
#define _PROTOCOL_CONNECT_H_

#include "BtStackInternalTypes.h"
#include "IProtocolConnect.h"
#include "ProtocolConnectSm.h"

namespace btstackif {

/**
 * Protocol connect state machine class.
 */
class ProtocolConnect : public IProtocolConnect, public ProtocolConnectSm
{
public:
   /**
    * Default constructor.
    */
   ProtocolConnect();

   /**
    * Destructor.
    */
   virtual ~ProtocolConnect();

   //===================================================================================================================

   virtual void setProtocolManager(IN IProtocolManager* manager);

   virtual void setDeviceManager(IN IDeviceManager* manager);

   virtual void setServiceSearch(IN IServiceSearch* manager);

   virtual void setProtocolManagerRequest(IN IProtocolManagerRequest* manager);

   virtual void setProtocolSmHelper(IN IProtocolSmHelper* helper);

   virtual void resetStateMachine(void);

   virtual void sendStartEvent(void);

   virtual void sendConnectEvent(void);

   virtual void sendAcceptEvent(void);

   virtual void sendRejectEvent(void);

   virtual void sendCancelEvent(void);

   virtual void sendWaitingForServiceSuccessEvent(void);

   virtual void sendWaitingForServiceFailedEvent(void);

   virtual void sendCreateSuccessEvent(void);

   virtual void sendCreateFailedEvent(void);

   virtual void sendSearchSuccessEvent(void);

   virtual void sendSearchFailedEvent(const BTSRequestResult result);

   virtual void sendWaitingForProtocolSuccessEvent(void);

   virtual void sendWaitingForProtocolFailedEvent(void);

   virtual void sendSettingSuccessEvent(void);

   virtual void sendSettingFailedEvent(void);

   virtual void sendBusyEvent(void);

   virtual void sendRemoteConnectEvent(void);

   virtual void sendConnectingEvent(void);

   virtual void sendDisconnectingEvent(void);

   virtual void sendCanceledEvent(void);

   virtual void sendConnectResultEvent(const BTSRequestResult result);

   virtual void sendFinalStateEvent(const BTSRequestResult result);

   virtual void sendCancelResultEvent(const BTSRequestResult result);

   virtual void sendGuardPeriodExpiredEvent(void);

   virtual void sendCharacterDeviceAvailableEvent(void);

   virtual void sendSetPageTimeoutSuccessEvent(void);

   virtual void setConnectionData(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid, IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId, IN const bool deviceConnected);

   virtual void setPageTimeout(IN const BTSTimeValue pageTimeout);

   virtual void setDeviceConnectionStatus(IN const bool deviceConnected);

   virtual void setMasInstanceNameAndId(IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId);

   virtual void setRestrictedConnecting(void);

   virtual bool doConnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* user = 0, IN const BTSSessionHandle handle = 0);

   virtual bool isConnectingState(OUT bool& localConnect) const;

   virtual bool isDisconnectedState(void) const;

   virtual void setTestDisableSearching(IN const bool enable);

   virtual void setTestDisableSetting(IN const bool enable);

   //===================================================================================================================

   virtual int checkCreating();

   virtual int checkSearching();

   virtual int checkSetting();

   virtual int checkSettingPageTimeout();

   virtual int checkWaitingForProtocol();

   virtual int checkWaitingForService();

   virtual int entryBusy();

   virtual int entryCheckForCharacterDevice();

   virtual int entryCheckForRetry();

   virtual int entryDisconnected();

   virtual int exitBusy();

   virtual int handleBusyCanceled(const BTSRequestResult result);

   virtual int exitConnecting();

   virtual int handleCancel();

   virtual int handleCancelDuringBusy();

   virtual int handleCancelDuringConnecting();

   virtual int handleCancelDuringSearching();

   virtual int handleCharacterDeviceAvailable();

   virtual int handleConnectCompleted(const BTSRequestResult result);

   virtual int handleConnectFinalState(const BTSRequestResult result);

   virtual int handleConnecting();

   virtual int handleConnectingCanceled();

   virtual int handleContinueCancel();

   virtual int handleCreateCanceled();

   virtual int handleCreateFailed();

   virtual int handleDisconnectedReached(const BTSRequestResult result);

   virtual int handleDisconnecting();

   virtual int handleExitNoRetry();

   virtual int handleFinalStateDuringGuardPeriod(const BTSRequestResult result);

   virtual int handleFinalStateDuringWaitForCharacterDevice(const BTSRequestResult result);

   virtual int handleGuardPeriodExpired();

   virtual int handleRemoteConnect();

   virtual int handleRemoteConnectDuringCanceled();

   virtual int handleRemoteConnectDuringConnecting();

   virtual int handleRemoteConnectDuringSearching();

   virtual int handleRemoteConnectDuringSetting();

   virtual int handleRemoteConnectDuringWaiting();

   virtual int handleRemoteConnectDuringWaitingForProtocol();

   virtual int handleRemoteConnecting();

   virtual int handleSearchCanceled();

   virtual int handleSearchFailed(const BTSRequestResult result);

   virtual int handleSetPageTimeoutCanceled();

   virtual int handleSettingCanceled();

   virtual int handleSettingFailed();

   virtual int handleWaitingForProtocolCanceled();

   virtual int handleWaitingForProtocolFailed();

   virtual int handleWaitingForServiceCanceled();

   virtual int handleWaitingForServiceFailed();

   virtual int initSm();

   virtual int isCanceled();

   virtual int isFinalStateDisconnectedReached(const BTSRequestResult result);

   virtual int isFinalStateReached(const BTSRequestResult result);

   virtual int sendAccept();

   virtual int sendConnect();

   virtual int sendDirectAccept();

   virtual int sendReject();

   virtual int storeFinalState(const BTSRequestResult result);

   //===================================================================================================================

private:
   IProtocolManager* _protocolManager; /**< protocol manager interface */
   IDeviceManager* _deviceManager; /**< device manager interface */
   IServiceSearch* _serviceSearch; /**< service search interface */
   IProtocolManagerRequest* _protocolManagerRequest; /**< protocol manager request interface */
   IProtocolSmHelper* _protocolSmHelper; /**< protocol SM helper interface */
   BTSBDAddress _address; /**< device address */
   BTSProtocolId _protocol; /**< protocol */
   BTSSppInstanceId _sppInstance; /**< SPP instance id */
   BTSUuid _outUuid; /**< SPP uuid */
   BTSUuid _inUuid; /**< SPP uuid */
   BTSMasInstanceName _masInstanceName; /**< MAS instance name */
   BTSMasInstanceId _masInstanceId; /**< MAS instance id */
   bool _cancel; /**< cancel requested */
   bool _final; /**< final state received */
   bool _finalDisconnected; /**< final state disconnected received */
   bool _connecting; /**< connecting state set */
   bool _disconnecting; /**< disconnecting state set */
   bool _localConnectOngoing; /**< local connect ongoing */
   bool _localConnectSent; /**< local connect sent */
   BTSRequestResult _result; /**< connect result */
   bool _remoteConnectForwarded; /**< flag indicating that remote connect request was forwarded to application */
   bool _remoteConnectReceived; /**< flag indicating that remote connect request was received from stack */
   bool _acceptReceived; /**< flag indicating that application has accepted remote connect */
   bool _rejectReceived; /**< flag indicating that application has rejected remote connect */
   bool _deviceConnected; /**< flag indicating that device is/was connected */
   bool _noAclFailed; /**< flag indicating that failed connect was not caused by ACL connect failed */
   bool _restricted; /**< flag indicating that connect is restricted */
   bool _smProcessingActive; /**< flag indicating that SM processing is active */
   bool _testDisableSearching; /**< flag to test connect without searching */
   bool _testDisableSetting; /**< flag to test connect without setting UUID */
   BTSTimeValue _pageTimeout; /**< page timeout */

   void handleConnectFinished(void);

   void handleConnectFinishedNew(void);

   void sendExitNoRetryEvent(void);

   void sendRetryConnectEvent(void);

   void sendContinueCancelEvent(void);

   bool isRetryProtocol(IN const BTSProtocolId protocol) const;
};

} //btstackif

#endif //_PROTOCOL_CONNECT_H_
