/**
 * @file IProtocolSmHelper.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Interface definition for protocol state machine helper.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for protocol state machine helper. For internal usage only.
 */

#ifndef _I_PROTOCOL_SM_HELPER_H_
#define _I_PROTOCOL_SM_HELPER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 * Interface definition.
 */
class IProtocolSmHelper
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolSmHelper() {}

   /**
    * Create remote connect request message for application.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void createRemoteConnectRequest(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Convert connect result code into disconnect reason.
    *
    * @param[in] result: request result
    *
    * @return = disconnect reason
    */
   virtual BTSDisconnectReason convertConnectResult2DisconnectReason(IN const BTSRequestResult result) const = 0;

   /**
    * Get disconnect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    *
    * @return = disconnect reason
    */
   virtual BTSDisconnectReason getDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const = 0;

   /**
    * Set connection status (within internal protocol list).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] connected: connected flag
    * @param[in] reason: disconnect reason
    */
   virtual void setConnectionStatus(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connected, IN const BTSDisconnectReason reason) = 0;

   /**
    * Update connection status to application.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] sendStatusToAll: flag
    */
   virtual void updateConnectionStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool sendStatusToAll) const = 0;

   /**
    * Update connection result to application.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] isConnectRequest: connect/disconnect flag
    * @param[in] result: result
    */
   virtual void updateConnectionResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool isConnectRequest, IN const BTSRequestResult result) const = 0;

   /**
    * Handle changed protocol connection status.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] stackReason: stack disconnect reason
    */
   virtual void handleChangedProtocolConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDisconnectReason stackReason = BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL) = 0;

   /**
    * Check if a retry is allowed.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    *
    * @return = true: retry is allowed,
    * @return = false: retry is not allowed
    */
   virtual bool isRetryAllowed(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const = 0;

   /**
    * Start retry timer.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void startRetryTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Stop retry timer.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void stopRetryTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Increase retry counter.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void increaseRetryCounter(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Set SPP UUID.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] instance: SPP instance
    * @param[in] outgoingUuid: SPP outgoing UUID
    * @param[in] incomingUuid: SPP incoming UUID
    */
   virtual void setSppUuid(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSSppInstanceId instance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid) = 0;

   /**
    * Check if any protocol is connected.
    *
    * @param[in] address: device address
    *
    * @return = true: any protocol is connected,
    * @return = false: no protocol is connected
    */
   virtual bool isAnyProtocolConnected(IN const BTSBDAddress& address) const = 0;

   /**
    * Check whether character device (RFCOMM) was already reported by udev monitor.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    *
    * @return = true: character device was reported,
    * @return = false: character device was not reported
    */
   virtual bool wasCharacterDeviceReported(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const = 0;

   /**
    * Start timer to wait for character device (RFCOMM).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void startWaitForCharacterDeviceTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Stop timer to wait for character device (RFCOMM).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void stopWaitForCharacterDeviceTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) = 0;

   /**
    * Check access rights for created character device (RFCOMM).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    */
   virtual void checkAccessRightsForCharacterDevice(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance) const = 0;

   /**
    * Configure page timeout.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] pageTimeout: page timeout
    *
    * @return = true: configuration was started,
    * @return = false: configuration was not started, e.g. not necessary or not possible
    */
   virtual bool configurePageTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSTimeValue pageTimeout) = 0;

   /**
    * Handle end of page timeout configuration.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] result: result
    * @param[in] pageTimeout: page timeout
    */
   virtual void handleEndOfPageTimeoutConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result, IN const BTSTimeValue pageTimeout) = 0;

   /**
    * Start timer to measure the time needed for local connect.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] connectAction: indicator for connect/disconnect action
    */
   virtual void startConnectTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction) = 0;

   /**
    * Stop timer to measure the time needed for local connect.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] connectAction: indicator for connect/disconnect action
    */
   virtual void stopConnectTimer(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction) = 0;

   /**
    * Check if connect timer is active.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] connectAction: indicator for connect/disconnect action
    *
    * @return = true: connect timer is active,
    * @return = false: connect timer is inactive
    */
   virtual bool isConnectTimerActive(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const bool connectAction) const = 0;
};

} //btstackif

#endif //_I_PROTOCOL_SM_HELPER_H_
