/**
 * @file IProtocolObserver.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Interface definition for protocol observer.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for protocol observer.
 */

#ifndef _I_PROTOCOL_OBSERVER_H_
#define _I_PROTOCOL_OBSERVER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IProtocolObserver
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolObserver() {}

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    */
   virtual void protocolAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) = 0;

   /**
    * Signal for protocol removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    */
   virtual void protocolRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance) = 0;

   /**
    * Signal for protocol available.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    * @param[in] level2: flag for level 1/2
    */
   virtual void protocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2) = 0;

   /**
    * Signal for protocol unavailable.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance name (MAP only)
    * @param[in] level2: flag for level 1/2
    */
   virtual void protocolUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2) = 0;
};

} //btstackif

#endif //_I_PROTOCOL_OBSERVER_H_
