/**
 * @file IProtocolManagerCallback.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Callback interface definition for protocol manager state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific protocol manager state machine.
 *          Callback interface must be provided by generic protocol manager state machine.
 */

#ifndef _I_PROTOCOL_MANAGER_CALLBACK_H_
#define _I_PROTOCOL_MANAGER_CALLBACK_H_

#include "BtStackInternalTypes.h"
#include "ProtocolManagerData.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_BaseMessage;

/**
 * Interface definition.
 */
class IProtocolManagerCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolManagerCallback() {}

   /**
    * Get configuration for connecting PAN via ConnMan.
    *
    * @return = true: use ConnMan interface to establish PAN connection,
    * @return = false: use proprietary BT stack interface to establish PAN connection
    */
   virtual bool isPanViaConnMan(void) const = 0;

   /**
    * Result for waiting for service.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] result: result
    */
   virtual void waitingForServiceResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result) = 0;

   /**
    * Result for waiting for protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] result: result
    */
   virtual void waitingForProtocolResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result) = 0;

   /**
    * Connect indication (optional message).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] result: result
    */
   virtual void connectIndication(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result) = 0;

   /**
    * Result for connecting protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] result: result
    *
    * @return = true: connect failed on ACL level (page timeout),
    * @return = false: connect successful or connect failed with other reason than page timeout
    */
   virtual bool connectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result) = 0;

   /**
    * Indicate connecting.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void connecting(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Disconnect indication (optional message).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] result: result
    */
   virtual void disconnectIndication(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result) = 0;

   /**
    * Result for disconnecting protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] result: result
    */
   virtual void disconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSRequestResult result) = 0;

   /**
    * Indicate disconnecting.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void disconnecting(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Update protocol connection status (for connected as well as disconnected state). Can be caused by connect or disconnect request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] status: status
    * @param[in] reason: disconnect reason
    *
    * @return = true: connect failed on ACL level (page timeout),
    * @return = false: connect successful or connect failed with other reason than page timeout
    */
   virtual bool updateStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason) = 0;

   /**
    * Update failed connect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] reason: disconnect reason
    */
   virtual void updateFailedConnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason) = 0;

   /**
    * Update disconnect reason.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] reason: disconnect reason
    */
   virtual void updateDisconnectReason(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDisconnectReason reason) = 0;

   /**
    * Update virtual device name (e.g. RFCOMM device name).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] name: virtual device name
    */
   virtual void updateVirtualDeviceName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const BTSDeviceName& name) = 0;

   /**
    * Inform about added/removed character device (TTY monitor).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] name: virtual device name
    * @param[in] added: added/removed flag
    */
   virtual void characterDeviceAddedRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDeviceName& name, IN const bool added) = 0;

   /**
    * Indicate remote connect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void remoteConnectRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Indicate canceled remote connect.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void remoteConnectCanceled(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Signal for protocol added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void protocolAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Signal for protocol removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    */
   virtual void protocolRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Signal for protocol available.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] protocolAvailableInfo: info for level 1/2
    */
   virtual void protocolAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ProtocolManagerData::ProtocolAvailableInfo& protocolAvailableInfo) = 0;

   /**
    * Signal for protocol unavailable.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    * @param[in] protocolAvailableInfo: info for level 1/2
    */
   virtual void protocolUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName, IN const ProtocolManagerData::ProtocolAvailableInfo& protocolAvailableInfo) = 0;

   /**
    * Get App2Bts message.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    *
    * @return = App2Bts message if found else NULL
    */
   virtual App2Bts_BaseMessage* getApp2BtsWorkingMessage(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Get protocol entry.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] uuid: SPP UUID (SPP only)
    * @param[in] masInstance: MAS instance id (MAP only)
    * @param[in] masName: MAS instance name (MAP only)
    *
    * @return = protocol entry
    */
   virtual ProtocolManagerData& getEntry(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& uuid, IN const BTSMasInstanceId masInstance, IN const BTSMasInstanceName& masName) = 0;

   /**
    * Get protocol list.
    *
    * @return = protocol list
    */
   virtual const ::std::map< BTSProtocolBaseEntry, ProtocolManagerData >& getProtocolList(void) const = 0;

   /**
    * Check if protocol is available (ready to use).
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppUuid: SPP UUID
    * @param[in] masInstance: MAS instance name
    * @param[in] level2: flag for level 1/2
    *
    * @return = true: protocol is available,
    * @return = false: protocol is not available
    */
   virtual bool isProtocolAvailableInternal(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool level2 = true) const = 0;

   /**
    * Get MAS instance name (only valid for MAP, will be handled internally).
    *
    * @param[out] instanceName: MAS instance name
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] instanceId: MAS instance id
    */
   virtual void getMasInstanceNameInternal(OUT BTSMasInstanceName& instanceName, IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSMasInstanceId instanceId) const = 0;
};

} //btstackif

#endif //_I_PROTOCOL_MANAGER_CALLBACK_H_
