/**
 * @file IProtocolConnect.h
 *
 * @par SW-Component
 * State machine for protocol manager
 *
 * @brief Interface definition for generic protocol connect state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for protocol connect state machine.
 */

#ifndef _I_PROTOCOL_CONNECT_H_
#define _I_PROTOCOL_CONNECT_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IProtocolManager;
class IDeviceManager;
class IServiceSearch;
class IProtocolManagerRequest;
class IProtocolSmHelper;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IProtocolConnect
{
public:
   /**
    * Destructor.
    */
   virtual ~IProtocolConnect() {}

   /**
    * Set protocol manager interface.
    *
    * @param[in] manager: protocol manager interface
    */
   virtual void setProtocolManager(IN IProtocolManager* manager) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] manager: device manager interface
    */
   virtual void setDeviceManager(IN IDeviceManager* manager) = 0;

   /**
    * Set service search interface.
    *
    * @param[in] manager: service search interface
    */
   virtual void setServiceSearch(IN IServiceSearch* manager) = 0;

   /**
    * Set protocol manager request interface.
    *
    * @param[in] manager: protocol manager request interface
    */
   virtual void setProtocolManagerRequest(IN IProtocolManagerRequest* manager) = 0;

   /**
    * Set protocol SM helper.
    *
    * @param[in] helper: protocol SM helper interface
    */
   virtual void setProtocolSmHelper(IN IProtocolSmHelper* helper) = 0;

   /**
    * Reset state machine.
    */
   virtual void resetStateMachine(void) = 0;

   /**
    * Send start event.
    */
   virtual void sendStartEvent(void) = 0;

   /**
    * Send connect event.
    */
   virtual void sendConnectEvent(void) = 0;

   /**
    * Send accept event.
    */
   virtual void sendAcceptEvent(void) = 0;

   /**
    * Send reject event.
    */
   virtual void sendRejectEvent(void) = 0;

   /**
    * Send cancel event.
    */
   virtual void sendCancelEvent(void) = 0;

   /**
    * Send waiting for service success event.
    */
   virtual void sendWaitingForServiceSuccessEvent(void) = 0;

   /**
    * Send waiting for service failed event.
    */
   virtual void sendWaitingForServiceFailedEvent(void) = 0;

   /**
    * Send create success event.
    */
   virtual void sendCreateSuccessEvent(void) = 0;

   /**
    * Send create failed event.
    */
   virtual void sendCreateFailedEvent(void) = 0;

   /**
    * Send search success event.
    */
   virtual void sendSearchSuccessEvent(void) = 0;

   /**
    * Send search failed event.
    *
    * @param[in] result: result
    */
   virtual void sendSearchFailedEvent(const BTSRequestResult result) = 0;

   /**
    * Send waiting for protocol success event.
    */
   virtual void sendWaitingForProtocolSuccessEvent(void) = 0;

   /**
    * Send waiting for protocol failed event.
    */
   virtual void sendWaitingForProtocolFailedEvent(void) = 0;

   /**
    * Send setting success event.
    */
   virtual void sendSettingSuccessEvent(void) = 0;

   /**
    * Send setting failed event.
    */
   virtual void sendSettingFailedEvent(void) = 0;

   /**
    * Send busy event.
    */
   virtual void sendBusyEvent(void) = 0;

   /**
    * Send remote connect event.
    */
   virtual void sendRemoteConnectEvent(void) = 0;

   /**
    * Send connecting event.
    */
   virtual void sendConnectingEvent(void) = 0;

   /**
    * Send disconnecting event.
    */
   virtual void sendDisconnectingEvent(void) = 0;

   /**
    * Send canceled event.
    */
   virtual void sendCanceledEvent(void) = 0;

   /**
    * Send connect result event.
    *
    * @param[in] result: result
    */
   virtual void sendConnectResultEvent(const BTSRequestResult result) = 0;

   /**
    * Send final state event.
    *
    * @param[in] result: result
    */
   virtual void sendFinalStateEvent(const BTSRequestResult result) = 0;

   /**
    * Send cancel result event.
    *
    * @param[in] result: result
    */
   virtual void sendCancelResultEvent(const BTSRequestResult result) = 0;

   /**
    * Send guard period expired event.
    */
   virtual void sendGuardPeriodExpiredEvent(void) = 0;

   /**
    * Send character device available event.
    */
   virtual void sendCharacterDeviceAvailableEvent(void) = 0;

   /**
    * Send set page timeout success event.
    */
   virtual void sendSetPageTimeoutSuccessEvent(void) = 0;

   /**
    * Set connection data.
    *
    * @param[in] address: device address
    * @param[in] protocol: protocol id
    * @param[in] sppInstance: SPP instance id (SPP only)
    * @param[in] outgoingUuid: outgoing SPP UUID (SPP only)
    * @param[in] incomingUuid: outgoing SPP UUID (SPP only)
    * @param[in] masInstanceName: MAS instance name (MAP only)
    * @param[in] masInstanceId: MAS instance id (MAP only)
    * @param[in] deviceConnected: device connection status
    */
   virtual void setConnectionData(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSSppInstanceId sppInstance, IN const BTSUuid& outgoingUuid, IN const BTSUuid& incomingUuid, IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId, IN const bool deviceConnected) = 0;

   /**
    * Set page timeout.
    *
    * @param[in] pageTimeout: page timeout
    */
   virtual void setPageTimeout(IN const BTSTimeValue pageTimeout) = 0;

   /**
    * Set device connection status.
    *
    * @param[in] deviceConnected: device connection status
    */
   virtual void setDeviceConnectionStatus(IN const bool deviceConnected) = 0;

   /**
    * Set MAS instance name and id.
    *
    * @param[in] masInstanceName: MAS instance name (MAP only)
    * @param[in] masInstanceId: MAS instance id (MAP only)
    */
   virtual void setMasInstanceNameAndId(IN const BTSMasInstanceName& masInstanceName, IN const BTSMasInstanceId masInstanceId) = 0;

   /**
    * Set restricted connecting.
    */
   virtual void setRestrictedConnecting(void) = 0;

   /**
    * Do state machine processing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] user: user reference
    * @param[in] handle: session handle
    *
    * @return = true: SM finished,
    * @return = false: SM not finished
    */
   virtual bool doConnectSmProcessing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN BtStackIfCallback* user = 0, IN const BTSSessionHandle handle = 0) = 0;

   /**
    * Check for local/remote connect during connecting state.
    *
    * @param[out] localConnect: flag indicating local/remote connect
    *
    * @return = true: SM is in connecting state,
    * @return = false: SM is not in connecting state
    */
   virtual bool isConnectingState(OUT bool& localConnect) const = 0;

   /**
    * Check for disconnected state.
    *
    * @return = true: SM is in disconnected state,
    * @return = false: SM is not in disconnected state
    */
   virtual bool isDisconnectedState(void) const = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestDisableSearching(IN const bool enable) = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestDisableSetting(IN const bool enable) = 0;
};

} //btstackif

#endif //_I_PROTOCOL_CONNECT_H_
