/**
 * @file Pairing.cpp
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Implementation of generic pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic pairing state machine.
 */

#include "Pairing.h"
#include "IPairingRequest.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "IRestrictedPairingConnecting.h"
#include "IConfiguration.h"
#include "IInquiry.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Pairing.cpp.trc.h"
#endif
#endif

namespace btstackif {

Pairing::Pairing() :
_requestIf(0),
_controlIf(0),
_deviceManager(0),
_restrictionIf(0),
_configurationIf(0),
_inquiryIf(0),
_deviceList(),
_testTriggerIgnoreReportedLinkKey(false)
{
}

Pairing::~Pairing()
{
   _requestIf = 0;
   _controlIf = 0;
   _deviceManager = 0;
   _restrictionIf = 0;
   _configurationIf = 0;
   _inquiryIf = 0;
}

void Pairing::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _deviceManager
   // keep _restrictionIf
   // keep _configurationIf
   // keep _inquiryIf
   _deviceList.clear();
   _testTriggerIgnoreReportedLinkKey = false;

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void Pairing::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current state/action
   for(::std::map< BTSBDAddress, PairingData >::const_iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      const PairingData& entry = it->second;

      // check for pending (pending devices are not forwarded to application => therefore handle only non-pending devices)
      if(false == entry.info.getBit(PairingData::PENDING))
      {
         const BTSBDAddress& workingAddress = it->first;
         const BTSLinkKey linkKey;
         const BTSDLinkKey dLinkKey;
         const BTSMajorServiceClass serviceClass;
         const BTSDeviceName name;

         // check for ongoing pairing => if entry is in list then pairing is ongoing
         createPairingFinishedMsg(bts2AppMsgList, 0, 0, true, workingAddress, BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);

         // check for pending start pairing result
         if(App2BtsOC_StartPairing == entry.requestItem.item.opCode)
         {
            // send result as failed
            createStartPairingResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, BTS_REQ_PAIRING_UNKNOWN);
         }
      }
   }

   // reset control data
   reset();
}

void Pairing::setInstance(IN IPairingRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void Pairing::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void Pairing::setDeviceManager(IN IDeviceManager* deviceManager)
{
   _deviceManager = deviceManager;

   FW_IF_NULL_PTR_RETURN(_deviceManager);

   _deviceManager->registerObserver(this);
}

void Pairing::setRestrictionIf(IN IRestrictedPairingConnecting* restrictionIf)
{
   _restrictionIf = restrictionIf;

   FW_NORMAL_ASSERT(0 != _restrictionIf);
}

void Pairing::setConfigurationIf(IN IConfiguration* configurationIf)
{
   _configurationIf = configurationIf;

   FW_NORMAL_ASSERT(0 != _configurationIf);
}

void Pairing::setInquiryIf(IN IInquiry* inquiryIf)
{
   _inquiryIf = inquiryIf;

   FW_NORMAL_ASSERT(0 != _inquiryIf);
}

IStateMachine* Pairing::getSmEntryInterface(void)
{
   return this;
}

void Pairing::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(resultCode);
   (void)(statusCode);

   const BTSLinkKey linkKey;
   const BTSDLinkKey dLinkKey;
   const BTSMajorServiceClass serviceClass;
   const BTSDeviceName name;

   if(false == isValidStartRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // use given address
      createPairingFinishedMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, request.getBDAddress(), BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);
      createStartPairingResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), request.getBDAddress(), BTS_REQ_INVALID_PARAM);
      return;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   createPairingFinishedMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, workingAddress, BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);
   createStartPairingResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, BTS_REQ_PAIRING_UNKNOWN);
}

bool Pairing::isValidStartRequest(IN const App2Bts_StartPairing& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool Pairing::isValidCancelRequest(IN const App2Bts_CancelPairing& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool Pairing::isValidSetPinCodeRequest(IN const App2Bts_SetPinCode& request) const
{
   // pin code can be empty

   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool Pairing::isValidConfirmSecureSimplePairingRequest(IN const App2Bts_ConfirmSecureSimplePairing& request) const
{
   // TODO: check _sspMode, _numericValue, _confirmationMode

   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool Pairing::startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request)
{
   ETG_TRACE_USR2((" startPairing: address=%s", request.getBDAddress().c_str()));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_IF_NULL_PTR_RETURN_FALSE(_deviceManager);

   if(false == isValidStartRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // use given address
      const BTSLinkKey linkKey;
      const BTSDLinkKey dLinkKey;
      const BTSMajorServiceClass serviceClass;
      const BTSDeviceName name;
      createPairingFinishedMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, request.getBDAddress(), BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);
      createStartPairingResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), request.getBDAddress(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   // check for existing entry
   ::std::map< BTSBDAddress, PairingData >::iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      PairingData& entry = it->second;

      if(BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.pairingOriginator)
      {
         // there is already an outgoing pairing for the given device
         // wait for end of pairing
         return false;
      }
      else if(BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE == entry.pairingOriginator)
      {
         if(App2BtsOC_StartPairing != entry.requestItem.item.opCode)
         {
            // there is already an incoming pairing for the given device
            // wait for end of pairing but update meta data
            entry.info.setBit(PairingData::LOCAL);
            request.getCompareItem(entry.requestItem.item);
            entry.requestItem.user = request.getUser();
            entry.requestItem.handle = request.getSessionHandle();
            return true;
         }
         else
         {
            // there is already an incoming pairing for the given device and there is already an outgoing pairing triggered too
            // wait for end of pairing, update meta data is already done
            return false;
         }
      }
      else /*if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.pairingOriginator)*/
      {
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();
         return false;
      }
   }

   // check if pairing to/from another device is ongoing
   if(0 < _deviceList.size())
   {
      // pairing to/from another device is ongoing => reject given request
      const BTSLinkKey linkKey;
      const BTSDLinkKey dLinkKey;
      const BTSMajorServiceClass serviceClass;
      const BTSDeviceName name;

      createPairingFinishedMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, workingAddress, BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);
      createStartPairingResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, BTS_REQ_PAIRING_UNKNOWN);
      return false;
   }

   // create new entry
   PairingData& entry = checkPairingList(workingAddress);

   // store data and process request
   entry.pairingOriginator = BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE;
   request.getCompareItem(entry.requestItem.item);
   entry.requestItem.user = request.getUser();
   entry.requestItem.handle = request.getSessionHandle();
   entry.createPending = false;
   entry.cancelRequested = false;

   if(true == _deviceManager->isDeviceAvailable(workingAddress))
   {
      _requestIf->startPairing(bts2IpcMsgList, bts2AppMsgList, workingAddress);
   }
   else
   {
      // trigger create device
      entry.createPending = true;
      _deviceManager->createDevice(bts2IpcMsgList, bts2AppMsgList, workingAddress);
   }

   return true;
}

bool Pairing::cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelPairing& request)
{
   ETG_TRACE_USR2((" cancelPairing: address=%s", request.getBDAddress().c_str()));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);
   FW_IF_NULL_PTR_RETURN_FALSE(_deviceManager);

   if(false == isValidCancelRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // there is no result message for cancel => wait for end of pairing
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, PairingData >::iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() == it)
   {
      // no matching entry => nothing to do
      return false;
   }

   // continue with existing entry
   PairingData& entry = it->second;

   if(BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.pairingOriginator)
   {
      // there is an outgoing pairing for the given device => send cancel

      // check if device creation is ongoing
      if(true == entry.createPending)
      {
         entry.cancelRequested = true;
      }

      // check for device available: it could be possible that the device is no longer available or device creation was started before (valid scenario)
      if(false == _deviceManager->isDeviceAvailable(workingAddress))
      {
         return false;
      }

      // pairing ongoing? => checked via available entry in _deviceList

      // cancel already requested?
      if(true == entry.cancelRequested)
      {
         return false;
      }

      _requestIf->cancelPairing(bts2IpcMsgList, bts2AppMsgList, workingAddress);

      entry.cancelRequested = true;

      return false;
   }
   else if(BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE == entry.pairingOriginator)
   {
      // there is an incoming pairing for the given device => cancel via specific answers
      if(true == entry.info.getBit(PairingData::WAITING))
      {
         entry.info.resetBit(PairingData::WAITING);

         if(BTS_SSP_LAST == entry.sspMode)
         {
            const BTSPinCode pinCode;
            _requestIf->setPinCode(bts2IpcMsgList, bts2AppMsgList, workingAddress, pinCode);
         }
         else
         {
            const BTSNumericValue numValue;
            _requestIf->confirmSecureSimplePairing(bts2IpcMsgList, bts2AppMsgList, workingAddress, entry.sspMode, numValue, BTS_CONFIRM_REJECT);
         }
      }
      return false;
   }
   else /*if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.pairingOriginator)*/
   {
      // there is no matching pairing ongoing => should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return false;
   }
}

void Pairing::cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   // this function is called only internally

   App2Bts_CancelPairing request;
   request.setBDAddress(address);
   (void)cancelPairing(bts2IpcMsgList, bts2AppMsgList, request);
}

bool Pairing::setPinCode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetPinCode& request)
{
   ETG_TRACE_USR2((" setPinCode: address=%s", request.getBDAddress().c_str()));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidSetPinCodeRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // there is no result message => wait for end of pairing
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   if(_deviceList.end() == _deviceList.find(workingAddress))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // no matching entry => nothing to do
      return false;
   }

   PairingData& entry = checkPairingList(workingAddress);

   if(true == entry.info.getBit(PairingData::WAITING))
   {
      entry.info.resetBit(PairingData::WAITING);

      _requestIf->setPinCode(bts2IpcMsgList, bts2AppMsgList, workingAddress, request.getPinCode());
   }

   return false;
}

bool Pairing::confirmSecureSimplePairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConfirmSecureSimplePairing& request)
{
   ETG_TRACE_USR2((" confirmSecureSimplePairing: address=%s", request.getBDAddress().c_str()));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidConfirmSecureSimplePairingRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // there is no result message => wait for end of pairing
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   if(_deviceList.end() == _deviceList.find(workingAddress))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // no matching entry => nothing to do
      return false;
   }

   PairingData& entry = checkPairingList(workingAddress);

   if(true == entry.info.getBit(PairingData::WAITING))
   {
      entry.info.resetBit(PairingData::WAITING);

      // ignore given ssp mode => use stored one
      _requestIf->confirmSecureSimplePairing(bts2IpcMsgList, bts2AppMsgList, workingAddress, entry.sspMode, request.getNumericValue(), request.getConfirmationMode());
   }

   return false;
}

bool Pairing::isPairingOngoing(void) const
{
   return (0 < _deviceList.size());
}

void Pairing::setTestTriggerIgnoreReportedLinkKey(IN const bool enable)
{
   _testTriggerIgnoreReportedLinkKey = enable;
}

void Pairing::startPairingResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // is received after pairing is completed

   (void)(bts2IpcMsgList);

   ETG_TRACE_USR2((" startPairingResult: address=%s", address.c_str()));

   const BTSBDAddress& workingAddress = address;

   if(_deviceList.end() == _deviceList.find(workingAddress))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   PairingData& entry = checkPairingList(workingAddress);
   bool finished(false);

   if(BTS_REQ_SUCCESS == result)
   {
      // link key was updated before
      // send result
      createStartPairingResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, BTS_REQ_SUCCESS);

      // pairing is completed => remove related entry
      finished = true;
   }
   else
   {
      bool handleError(false);

      // following scenarios are possible:
      // - rejected because of incoming pairing request => error code org.bluez.Error.InProgress expected; test result: org.bluez.Error.UnknownReason
      // - rejected because of other ongoing request => error code org.bluez.Error.InProgress expected; 2 outgoing pairing requests at the same time will be avoided
      // - pairing rejected/aborted on remote side => normal error
      // => check error code, device address and ONGOING flag
      if(true == entry.info.getBit(PairingData::ONGOING))
      {
         // SSP or PinCode request received for this device
         if(BTS_REQ_BUSY == result)
         {
            // rejected because of incoming pairing request (same device)
            // wait for end of pairing
            entry.info.setBit(PairingData::LOCAL);
         }
         else
         {
            // rejected because of other reason (same device); could be that pairing rejected/aborted on remote side
            // handle as error
            handleError = true;
         }
      }
      else
      {
         // no SSP or PinCode request received for this device
         if(BTS_REQ_BUSY == result)
         {
            // rejected because of incoming pairing request (different device)
            // handle as error
            handleError = true;
         }
         else
         {
            // rejected because of other reason
            // handle as error
            handleError = true;
         }
      }

      if(true == handleError)
      {
         // send pairing finished and result messages
         const BTSLinkKey linkKey;
         const BTSDLinkKey dLinkKey;
         const BTSMajorServiceClass serviceClass;
         const BTSDeviceName name;

         createPairingFinishedMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, false, workingAddress, result, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);
         createStartPairingResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, result);

         // pairing is completed => remove related entry
         finished = true;
      }
   }

   if(true == finished)
   {
      // pairing sequence is finished

      handleActionFinished(messageItem, entry.requestItem.item);

      // reset control data because action is finished
      entry.requestItem.reset();

      // remove entry
      removeDevice(workingAddress);
   }
}

void Pairing::indicateLinkKey(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey)
{
   (void)(bts2IpcMsgList);

   //===================================================================================================================
   // debug section start
   if(true == _testTriggerIgnoreReportedLinkKey)
   {
      _testTriggerIgnoreReportedLinkKey = false;

      return;
   }
   // debug section end
   //===================================================================================================================

   ETG_TRACE_USR2((" indicateLinkKey: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_deviceManager);
   FW_IF_NULL_PTR_RETURN(_restrictionIf);

   const BTSBDAddress& workingAddress = address;

   // following scenario was observed:
   // - incoming pairing request ongoing
   // - outgoing pairing request initiated from host side before reception of incoming pairing request
   // - outgoing pairing request rejected because of incoming pairing request => error code: org.bluez.Error.UnknownReason
   // - entry in device list removed in startPairingResult()
   // - therefore do following check but continue
   if(_deviceList.end() == _deviceList.find(workingAddress))
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      // continue --- return;
   }

   PairingData& entry = checkPairingList(workingAddress);
   bool rejected(false);

   // check originator information
   if((BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.pairingOriginator) || (true == entry.info.getBit(PairingData::LOCAL)))
   {
      // pairing request was somehow initiated/requested from local side => do not check restriction
   }
   else
   {
      BTSBDAddress restrictedAddress;
      if(true == _restrictionIf->isPairingRestricted(restrictedAddress))
      {
         if(workingAddress != restrictedAddress)
         {
            // we have to reject given pairing request
            rejected = true;
         }
      }
   }

   if(true == rejected)
   {
      // do not send any pairing finished message to application in case of rejected pairing request
   }
   else
   {
      BTSDeviceName name;
      BTSCod cod(0);
      _deviceManager->getDeviceData(name, cod, workingAddress);

      const unsigned char minor = (unsigned char)((cod >> 2) & 0x0000003F); // TODO: move cod conversion to FW
      const unsigned char major = (unsigned char)((cod >> 8) & 0x0000001F);
      const unsigned short int service = (unsigned short int)((cod >> 13) & 0x000007FF);
      BTSMajorServiceClass serviceClass;
      serviceClass.setData(service);

      createPairingFinishedMsg(bts2AppMsgList, 0, 0, true, workingAddress, result, linkKeyType, linkKey, dLinkKey, serviceClass, (BTSMajorDeviceClass)major, (BTSMinorDeviceClass)minor, name);
   }

   if(true == entry.info.getBit(PairingData::LOCAL))
   {
      // incoming pairing but also triggered from local side but no further startPairingResult() will follow => send result message now
      if(App2BtsOC_StartPairing == entry.requestItem.item.opCode)
      {
         createStartPairingResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, result);
      }

      // pairing sequence is finished

      if(App2BtsOC_StartPairing == entry.requestItem.item.opCode)
      {
         handleActionFinished(messageItem, entry.requestItem.item);
      }
      else
      {
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();

         entry.requestItem.item.opCode = App2BtsOC_StartPairing;
         entry.requestItem.item.deviceAddress = workingAddress;
         handleActionFinished(messageItem, entry.requestItem.item);
      }

      // reset control data because action is finished
      entry.requestItem.reset();

      // remove entry
      removeDevice(workingAddress);
   }
   else if(BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.pairingOriginator)
   {
      // outgoing pairing, startPairingResult() will follow
   }
   else
   {
      // pure incoming pairing
      // pairing sequence finished

      // we have no user request => fake necessary data
      messageItem.item.opCode = App2BtsOC_StartPairing;
      messageItem.item.deviceAddress = workingAddress;
      messageItem.deleteMessage = true;

      // reset control data because action is finished
      entry.requestItem.reset();

      // remove entry
      removeDevice(workingAddress);
   }
}

void Pairing::handleEndOfSuccessfulIncomingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);

   // pure incoming pairing
   // pairing sequence finished

   // we have no user request => fake necessary data
   messageItem.item.opCode = App2BtsOC_StartPairing;
   messageItem.item.deviceAddress = address;
   messageItem.deleteMessage = true;
}

void Pairing::indicatePinCodeRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(messageItem);

   ETG_TRACE_USR2((" indicatePinCodeRequest: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_deviceManager);
   FW_IF_NULL_PTR_RETURN(_restrictionIf);
   FW_IF_NULL_PTR_RETURN(_requestIf);

   const BTSBDAddress& workingAddress = address;
   PairingData& entry = checkPairingList(workingAddress);
   bool rejected(false);

   // set pairing information
   if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.pairingOriginator)
   {
      entry.pairingOriginator = BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE;
   }
   // entry.sspMode
   // entry.numValue
   entry.info.setBit(PairingData::ONGOING);
   // entry.requestItem

   // check originator information
   if((BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.pairingOriginator) || (true == entry.info.getBit(PairingData::LOCAL)))
   {
      // pairing request was somehow initiated/requested from local side => do not check restriction
   }
   else
   {
      BTSBDAddress restrictedAddress;
      if(true == _restrictionIf->isPairingRestricted(restrictedAddress))
      {
         if(workingAddress != restrictedAddress)
         {
            // we have to reject given pairing request
            rejected = true;
         }
      }
   }

   if(true == rejected)
   {
      const BTSPinCode pinCode;
      _requestIf->setPinCode(bts2IpcMsgList, bts2AppMsgList, workingAddress, pinCode);
   }
   else
   {
      // check for available device
      if(true == _deviceManager->isDeviceAvailable(workingAddress))
      {
         // we can directly forward to application
         BTSDeviceName name;
         BTSCod cod(0);
         _deviceManager->getDeviceData(name, cod, workingAddress);
         createPinCodeRequestedMsg(bts2AppMsgList, 0, 0, true, workingAddress, name, entry.pairingOriginator);
         entry.info.setBit(PairingData::WAITING);

         // remove pending flag
         entry.info.resetBit(PairingData::PENDING);
      }
      else
      {
         // we have to mark as pending
         entry.info.setBit(PairingData::PENDING);
      }

      // stop inquiry
      stopInquiry(bts2IpcMsgList, bts2AppMsgList);
   }
}

void Pairing::indicateSecureSimplePairingRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue)
{
   (void)(messageItem);

   ETG_TRACE_USR2((" indicateSecureSimplePairingRequest: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_deviceManager);
   FW_IF_NULL_PTR_RETURN(_restrictionIf);
   FW_IF_NULL_PTR_RETURN(_requestIf);

   const BTSBDAddress& workingAddress = address;
   PairingData& entry = checkPairingList(workingAddress);
   bool rejected(false);

   // set pairing information
   if(BTS_PAIR_CONNECT_INITIATED_LAST == entry.pairingOriginator)
   {
      entry.pairingOriginator = BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE;
   }
   entry.sspMode = sspMode;
   entry.numValue = numericValue;
   entry.info.setBit(PairingData::ONGOING);
   // entry.requestItem

   // check originator information
   if((BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE == entry.pairingOriginator) || (true == entry.info.getBit(PairingData::LOCAL)))
   {
      // pairing request was somehow initiated/requested from local side => do not check restriction
   }
   else
   {
      BTSBDAddress restrictedAddress;
      if(true == _restrictionIf->isPairingRestricted(restrictedAddress))
      {
         if(workingAddress != restrictedAddress)
         {
            // we have to reject given pairing request
            rejected = true;
         }
      }
   }

   if(true == rejected)
   {
      const BTSNumericValue numValue;
      _requestIf->confirmSecureSimplePairing(bts2IpcMsgList, bts2AppMsgList, workingAddress, entry.sspMode, numValue, BTS_CONFIRM_REJECT);
   }
   else
   {
      // check for available device
      if(true == _deviceManager->isDeviceAvailable(workingAddress))
      {
         // we can directly forward to application
         BTSDeviceName name;
         BTSCod cod(0);
         _deviceManager->getDeviceData(name, cod, workingAddress);
         createSecureSimplePairingRequestedMsg(bts2AppMsgList, 0, 0, true, workingAddress, name, entry.pairingOriginator, entry.sspMode, entry.numValue);
         entry.info.setBit(PairingData::WAITING);

         // remove pending flag
         entry.info.resetBit(PairingData::PENDING);
      }
      else
      {
         // we have to mark as pending
         entry.info.setBit(PairingData::PENDING);
      }

      // stop inquiry
      stopInquiry(bts2IpcMsgList, bts2AppMsgList);
   }
}

App2Bts_BaseMessage* Pairing::getApp2BtsWorkingMessage(IN const BTSBDAddress& address)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   if(_deviceList.end() == _deviceList.find(address))
   {
      // no matching entry
      return 0;
   }

   PairingData& entry = checkPairingList(address);

   return _controlIf->findApp2BtsWorkingMessageWrapper(entry.requestItem.item);
}

PairingData& Pairing::getEntry(IN const BTSBDAddress& address)
{
   return checkPairingList(address);
}

void Pairing::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void Pairing::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // check for ongoing incoming pairing
   checkForOngoingPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
}

void Pairing::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   FW_IF_NULL_PTR_RETURN(_deviceManager);

   const BTSBDAddress& workingAddress = address;

   // check for pending device
   if(_deviceList.end() == _deviceList.find(workingAddress))
   {
      // no matching entry
      return;
   }

   PairingData& entry = checkPairingList(workingAddress);

   // check for outgoing pairing sequence
   if(true == entry.createPending)
   {
      entry.createPending = false;

      // check if cancel was requested
      if(true == entry.cancelRequested)
      {
         // no pairing request was sent until now
         // no cancel request was sent until now
         // => send answer for pairing

         // pairing sequence is finished
         handleEndOfFailedOutgoingPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, 0, entry, workingAddress);
      }
      else
      {
         // start pairing
         _requestIf->startPairing(bts2IpcMsgList, bts2AppMsgList, workingAddress);
      }

      return;
   }

   // check for incoming pairing sequence
   if(false == entry.info.getBit(PairingData::PENDING))
   {
      // not pending
      return;
   }

   BTSDeviceName name;
   BTSCod cod(0);
   _deviceManager->getDeviceData(name, cod, workingAddress);

   // forward to application
   if(BTS_SSP_LAST == entry.sspMode)
   {
      createPinCodeRequestedMsg(bts2AppMsgList, 0, 0, true, workingAddress, name, entry.pairingOriginator);
   }
   else
   {
      createSecureSimplePairingRequestedMsg(bts2AppMsgList, 0, 0, true, workingAddress, name, entry.pairingOriginator, entry.sspMode, entry.numValue);
   }

   // set / reset flags
   entry.info.setBit(PairingData::WAITING);
   entry.info.resetBit(PairingData::PENDING);
}

void Pairing::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
}

void Pairing::deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(aclTransition);
   (void)(aclConnected);
   (void)(anyProfileTransition);
   (void)(anyProfileConnected);
}

void Pairing::deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   if(BTS_REQ_SUCCESS == result)
   {
      return;
   }

   const BTSBDAddress& workingAddress(address);

   // check for pending device
   if(_deviceList.end() == _deviceList.find(workingAddress))
   {
      // no matching entry
      return;
   }

   PairingData& entry = checkPairingList(workingAddress);

   if(true == entry.createPending)
   {
      entry.createPending = false;

      // pairing sequence is finished
      handleEndOfFailedOutgoingPairing(bts2IpcMsgList, bts2AppMsgList, messageItem, 0, entry, workingAddress);
   }
}

void Pairing::deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // information not needed
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(address);
   (void)(result);
}

PairingData& Pairing::checkPairingList(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, PairingData >::iterator it = _deviceList.find(address);
   if(_deviceList.end() == it)
   {
      // add new entry
      ETG_TRACE_USR2((" checkPairingList: _deviceList.size()=%u", (1 + _deviceList.size())));
      return _deviceList[address];
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}

void Pairing::removeDevice(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, PairingData >::iterator it = _deviceList.find(address);
   if(_deviceList.end() != it)
   {
      _deviceList.erase(it);
      ETG_TRACE_USR2((" removeDevice: _deviceList.size()=%u", _deviceList.size()));
   }
}

void Pairing::createPinCodeRequestedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSDeviceName& deviceName, IN const BTSPairConnectOriginator originator) const
{
   Bts2App_PinCodeRequested* msg = ptrNew_Bts2App_PinCodeRequested();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBDAddress(address);
      msg->setDeviceName(deviceName);
      msg->setPairConnectOriginator(originator);

      bts2AppMsgList.push_back(msg);
   }
}

void Pairing::createSecureSimplePairingRequestedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSDeviceName& deviceName, IN const BTSPairConnectOriginator originator, IN const BTSSspMode mode, IN const BTSNumericValue& value) const
{
   Bts2App_SecureSimplePairingRequested* msg = ptrNew_Bts2App_SecureSimplePairingRequested();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBDAddress(address);
      msg->setDeviceName(deviceName);
      msg->setSspMode(mode);
      msg->setNumericValue(value);
      msg->setPairConnectOriginator(originator);

      bts2AppMsgList.push_back(msg);
   }
}

void Pairing::createPairingFinishedMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSRequestResult result,
      IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSMajorServiceClass& majorServiceClass, IN const BTSMajorDeviceClass majorDeviceClass, IN const BTSMinorDeviceClass minorDeviceClass, IN const BTSDeviceName& deviceName) const
{
   Bts2App_PairingFinished* msg = ptrNew_Bts2App_PairingFinished();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBDAddress(address);
      msg->setRequestResult(result);
      msg->setLinkKeyType(linkKeyType);
      msg->setLinkKey(linkKey);
      msg->setDLinkKey(dLinkKey);
      msg->setMajorServiceClass(majorServiceClass);
      msg->setMajorDeviceClass(majorDeviceClass);
      msg->setMinorDeviceClass(minorDeviceClass);
      msg->setDeviceName(deviceName);

      bts2AppMsgList.push_back(msg);
   }
}

void Pairing::createStartPairingResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_StartPairingResult* msg = ptrNew_Bts2App_StartPairingResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setBDAddress(address);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void Pairing::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSApp2BtsMessageCompareItem& item)
{
   messageItem.item = item;
   messageItem.deleteMessage = true;
   if(0 == messageItem.message)
   {
      messageItem.message = getApp2BtsWorkingMessage(item);
   }
   FW_NORMAL_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* Pairing::getApp2BtsWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(item);
}

void Pairing::stopInquiry(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
#if 0
#else
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
#endif

   FW_IF_NULL_PTR_RETURN(_configurationIf);
   FW_IF_NULL_PTR_RETURN(_inquiryIf);

   if(true == _configurationIf->getConfiguration().stopInquiryDuringIncomingPairing)
   {
#if 0
      _inquiryIf->setStopRequested(bts2IpcMsgList, bts2AppMsgList);
#else
      if((true == _inquiryIf->isDiscoveryOngoing()) || (true == _inquiryIf->isRequestOngoing()))
      {
         _inquiryIf->triggerStopDiscovery();
      }
#endif
   }
}

void Pairing::checkForOngoingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);

   ETG_TRACE_USR2((" checkForOngoingPairing: address=%s", address.c_str()));

   // either ACL link was disconnected or device was removed
   // in case of ongoing pairing the pairing sequence failed

   /*
    * in case of removed device all results/indications are filtered in PairingGenivi SW part: no impact (only normal assert)
    * in case of disconnected ACL link following functions were checked w.r.t. necessary changes:
    * - startPairingResult: no impact (only normal assert)
    * - indicateLinkKey: impact: it can happen that a PairingFinished is sent to client (e.g. BM core) although pairing is already finished; client shall ignore this update
    * - handleEndOfSuccessfulIncomingPairing: no impact
    *
    * NOTE: this function is not called due to disconnected ACL link because it could happen that this is a normal sequence
    */

   ::std::map< BTSBDAddress, PairingData >::iterator it = _deviceList.find(address);

   if(_deviceList.end() == it)
   {
      // no matching entry
      return;
   }

   // if entry is in list then pairing is ongoing
   PairingData& entry(it->second);

   // handle only non-pending devices (pending devices are not forwarded to application)
   if(true == entry.info.getBit(PairingData::PENDING))
   {
      return;
   }

   const BTSBDAddress& workingAddress(address);
   const BTSLinkKey linkKey;
   const BTSDLinkKey dLinkKey;
   const BTSMajorServiceClass serviceClass;
   const BTSDeviceName name;

   // send pairing finished
   createPairingFinishedMsg(bts2AppMsgList, 0, 0, true, workingAddress, BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);

   // check for pending start pairing result
   if(App2BtsOC_StartPairing == entry.requestItem.item.opCode)
   {
      // send result as failed
      createStartPairingResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, BTS_REQ_PAIRING_UNKNOWN);
   }

   // pairing sequence finished
   if(App2BtsOC_StartPairing == entry.requestItem.item.opCode)
   {
      handleActionFinished(messageItem, entry.requestItem.item);
   }
   else
   {
      // we have no user request => fake necessary data
      messageItem.item.opCode = App2BtsOC_StartPairing;
      messageItem.item.deviceAddress = workingAddress;
      messageItem.deleteMessage = true;
   }

   // reset control data because action is finished
   entry.requestItem.reset();

   // remove entry
   removeDevice(workingAddress);
}

void Pairing::handleEndOfFailedOutgoingPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, OUT bool* connectFailed, INOUT PairingData& entry, IN const BTSBDAddress& address)
{
   (void)(bts2IpcMsgList);
   (void)(connectFailed);

   const BTSBDAddress& workingAddress(address);
   const BTSLinkKey linkKey;
   const BTSDLinkKey dLinkKey;
   const BTSMajorServiceClass serviceClass;
   const BTSDeviceName name;

   // send pairing finished
   createPairingFinishedMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, false, workingAddress, BTS_REQ_PAIRING_UNKNOWN, BTS_LINK_KEY_TYPE_UNKNOWN, linkKey, dLinkKey, serviceClass, BTS_COD_MAJORDC_UNCATEGORIZED, BTS_COD_MINORDC_UNCATEGORIZED, name);
   // send result as failed
   createStartPairingResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, BTS_REQ_PAIRING_UNKNOWN);

   handleActionFinished(messageItem, entry.requestItem.item);

   // reset control data because action is finished
   entry.requestItem.reset();

   // remove entry
   removeDevice(workingAddress);
}

} //btstackif
