/**
 * @file PairingData.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Implementation of pairing data.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of pairing data.
 */

#ifndef _PAIRING_DATA_H_
#define _PAIRING_DATA_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 * Pairing data class.
 */
class PairingData
{
public:
   /**
    * Type definition for device data information bits.
    * #PairingDataInfoBit#ignore#
    */
   enum PairingDataInfoBit
   {
      PENDING = 0, /**< request pending, not forwarded to application */
      WAITING, /**< waiting for answer from application */
      ONGOING, /**< pairing ongoing */
      LOCAL, /**< "initiated" from local side */
      // -------------------------------------------------------------------------
      PAIRING_DATA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for device data information.
    */
   typedef BTSBitfield<unsigned int, PairingDataInfoBit, PAIRING_DATA_LAST> PairingDataInfo;

   /**
    * Default constructor.
    */
   PairingData();

   /**
    * Destructor.
    */
   virtual ~PairingData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   PairingData(const PairingData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   PairingData& operator=(const PairingData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const PairingData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const PairingData& ref) const;

   // member with public access
   BTSPairConnectOriginator pairingOriginator; /**< pairing origin type */
   BTSSspMode sspMode; /**< pairing mode */
   BTSNumericValue numValue; /**< numeric value */
   PairingDataInfo info; /**< pairing data information */
   // name is taken from device manager
   // cod is taken from device manager
   // link key is received in last message of sequence
   // d link key is received in last message of sequence
   // link key type is received in last message of sequence
   BTSApp2BtsMessageRequestItem requestItem; /**< request item */
   bool createPending; /**< flag for sent create request (pending device available) */
   bool cancelRequested; /**< flag for requested cancel */
};

} //btstackif

#endif //_PAIRING_DATA_H_
