/**
 * @file IPairing.h
 *
 * @par SW-Component
 * State machine for pairing
 *
 * @brief Interface definition for generic pairing state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for pairing state machine.
 */

#ifndef _I_PAIRING_H_
#define _I_PAIRING_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IPairingRequest;
class IBasicControl;
class IDeviceManager;
class IRestrictedPairingConnecting;
class IConfiguration;
class IInquiry;
class IStateMachine;
class App2Bts_StartPairing;
class App2Bts_CancelPairing;
class App2Bts_SetPinCode;
class App2Bts_ConfirmSecureSimplePairing;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IPairing
{
public:
   /**
    * Destructor.
    */
   virtual ~IPairing() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN IPairingRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManager: device manager
    */
   virtual void setDeviceManager(IN IDeviceManager* deviceManager) = 0;

   /**
    * Set interface for restricted pairing and connecting.
    *
    * @param[in] restrictionIf: interface for restricted pairing and connecting
    */
   virtual void setRestrictionIf(IN IRestrictedPairingConnecting* restrictionIf) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Set inquiry interface.
    *
    * @param[in] inquiryIf: inquiry interface
    */
   virtual void setInquiryIf(IN IInquiry* inquiryIf) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidStartRequest(IN const App2Bts_StartPairing& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidCancelRequest(IN const App2Bts_CancelPairing& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidSetPinCodeRequest(IN const App2Bts_SetPinCode& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidConfirmSecureSimplePairingRequest(IN const App2Bts_ConfirmSecureSimplePairing& request) const = 0;

   /**
    * Start pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request) = 0;

   /**
    * Cancel pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelPairing& request) = 0;

   /**
    * Cancel pairing (internal).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] address: device address
    */
   virtual void cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address) = 0;

   /**
    * Set pin code.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool setPinCode(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetPinCode& request) = 0;

   /**
    * Confirm secure simple pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool confirmSecureSimplePairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConfirmSecureSimplePairing& request) = 0;

   /**
    * Check for ongoing pairing.
    *
    * @return = true: pairing ongoing,
    * @return = false: no pairing ongoing
    */
   virtual bool isPairingOngoing(void) const = 0;

   /**
    * Set test flag.
    *
    * @param[in] enable: test flag
    */
   virtual void setTestTriggerIgnoreReportedLinkKey(IN const bool enable) = 0;
};

} //btstackif

#endif //_I_PAIRING_H_
