/**
 * @file ILocalBluetoothName.h
 *
 * @par SW-Component
 * State machine for setting local Bluetooth name
 *
 * @brief Interface definition for generic setting local Bluetooth name state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for setting local Bluetooth name state machine.
 */

#ifndef _I_LOCAL_BLUETOOTH_NAME_H_
#define _I_LOCAL_BLUETOOTH_NAME_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ILocalBluetoothNameRequest;
class IBasicControl;
class ITimerPool;
class ISwitchBluetooth;
class IConfiguration;
class IConfigurationClient;
class IStateMachine;
class App2Bts_GetLocalBtName;
class App2Bts_SetLocalBtName;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ILocalBluetoothName
{
public:
   /**
    * Destructor.
    */
   virtual ~ILocalBluetoothName() {}

   /**
    * Set specific instance (implementation).
    *
    * @param[in] instance: instance (implementation)
    */
   virtual void setInstance(IN ILocalBluetoothNameRequest* instance) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Set switch Bluetooth interface.
    *
    * @param[in] switchBluetooth: witch Bluetooth interface
    */
   virtual void setSwitchBluetoothIf(IN ISwitchBluetooth* switchBluetooth) = 0;

   /**
    * Set configuration interface.
    *
    * @param[in] configurationIf: configuration interface
    */
   virtual void setConfigurationIf(IN IConfiguration* configurationIf) = 0;

   /**
    * Get configuration client handler.
    *
    * @return = configuration client handler
    */
   virtual IConfigurationClient* getConfigurationClientHandler(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Send current status for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] statusCode: status code
    */
   virtual void sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetLocalBtName& request, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Send status and result for given control data.
    * <BR><B>This method shall be used to reject application requests or handle doubled application requests.</B>
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    * @param[in] sendStatusToAll: flag for sending status to all or not
    * @param[in] resultCode: result code
    * @param[in] statusCode: status code
    */
   virtual void sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalBtName& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidGetRequest(IN const App2Bts_GetLocalBtName& request) const = 0;

   /**
    * Check if given request is valid.
    *
    * @param[in] request: request
    *
    * @return = true: valid request,
    * @return = false: invalid request
    */
   virtual bool isValidSetRequest(IN const App2Bts_SetLocalBtName& request) const = 0;

   /**
    * Set local Bluetooth name.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual bool setLocalBluetoothName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_SetLocalBtName& request) = 0;

   /**
    * Get current BT stack value for local name.
    *
    * @return = local name
    */
   virtual const BTSDeviceName& getBtStackName(void) const = 0;
};

} //btstackif

#endif //_I_LOCAL_BLUETOOTH_NAME_H_
