/**
 * @file ILocalAdapterModesCallback.h
 *
 * @par SW-Component
 * State machine for local adapter modes
 *
 * @brief Callback interface definition for local adapter modes state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the callback interface to be used by specific local adapter modes state machine.
 *          Callback interface must be provided by generic local adapter modes state machine.
 */

#ifndef _I_LOCAL_ADAPTER_MODES_CALLBACK_H_
#define _I_LOCAL_ADAPTER_MODES_CALLBACK_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class ILocalAdapterModesCallback
{
public:
   /**
    * Destructor.
    */
   virtual ~ILocalAdapterModesCallback() {}

   /**
    * Update discoverable mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: discoverable mode
    * @param[in] success: success/failed flag
    */
   virtual void updateDiscoverable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force = false) = 0;

   /**
    * Update pairable mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: pairable mode
    * @param[in] success: success/failed flag
    */
   virtual void updatePairable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force = false) = 0;

   /**
    * Update connectable mode.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] enabled: connectable mode
    * @param[in] success: success/failed flag
    */
   virtual void updateConnectable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool enabled, IN const bool success, IN const bool force = false) = 0;

   /**
    * Update discoverable timeout.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] timeout: discoverable timeout
    * @param[in] success: success/failed flag
    */
   virtual void updateDiscoverableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const bool success, IN const bool force = false) = 0;

   /**
    * Update pairable timeout.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] timeout: pairable timeout
    * @param[in] success: success/failed flag
    */
   virtual void updatePairableTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoverableTimeout timeout, IN const bool success, IN const bool force = false) = 0;

   /**
    * Set initial stack discoverable mode.
    *
    * @param[in] enabled: discoverable mode
    */
   virtual void setInitialStackDiscoverableMode(IN const bool enabled) = 0;

   /**
    * Set initial stack pairable mode.
    *
    * @param[in] enabled: pairable mode
    */
   virtual void setInitialStackPairableMode(IN const bool enabled) = 0;

   /**
    * Set initial stack connectable mode.
    *
    * @param[in] enabled: connectable mode
    */
   virtual void setInitialStackConnectableMode(IN const bool enabled) = 0;
};

} //btstackif

#endif //_I_LOCAL_ADAPTER_MODES_CALLBACK_H_
