/**
 * @file Inquiry.cpp
 *
 * @par SW-Component
 * State machine for inquiry
 *
 * @brief Implementation of generic inquiry state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic inquiry state machine.
 */

#include "Inquiry.h"
#include "IInquiryRequest.h"
#include "IBasicControl.h"
#include "IDeviceManager.h"
#include "IInquiryObserver.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/Inquiry.cpp.trc.h"
#endif
#endif

namespace btstackif {

Inquiry::Inquiry() :
_requestIf(0),
_controlIf(0),
_deviceManagerIf(0),
_timerPoolIf(0),
_requestItem(),
_discovering(false),
_ongoing(false),
_pending(false),
_stopRequested(false),
_internalStop(false),
_deviceList(),
_observerList()
{
}

Inquiry::~Inquiry()
{
   _requestIf = 0;
   _controlIf = 0;
   _deviceManagerIf = 0;
   _timerPoolIf = 0;
}

void Inquiry::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _deviceManagerIf
   // keep _timerPoolIf
   _requestItem.reset();
   _discovering = false;
   _ongoing = false;
   _pending = false;
   _stopRequested = false;
   _internalStop = false;
   _deviceList.clear();
   // keep _observerList

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void Inquiry::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current state/action
   switch(_requestItem.item.opCode)
   {
      case App2BtsOC_Last:
         // no request is pending
         if(true == _discovering)
         {
            // inquiry is active => update status as false
            createDiscoveringStatusMsg(bts2AppMsgList, 0, 0, true, BTS_DISCOVERING_OFF);
         }
         else
         {
            // inquiry is inactive => nothing to do
         }
         break;
      case App2BtsOC_StartDiscovery:
         // starting inquiry ongoing => update status as false and result as failed
         createDiscoveringStatusMsg(bts2AppMsgList, 0, 0, true, BTS_DISCOVERING_OFF);
         createStartDiscoveryResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_FAILED);
         break;
      case App2BtsOC_StopDiscovery:
         // stopping inquiry ongoing => update status as false and result as success
         createDiscoveringStatusMsg(bts2AppMsgList, 0, 0, true, BTS_DISCOVERING_OFF);
         createStopDiscoveryResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_SUCCESS);
         break;
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   // reset control data
   reset();
}

void Inquiry::setInstance(IN IInquiryRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void Inquiry::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);
}

void Inquiry::setDeviceManagerIf(IN IDeviceManager* deviceManager)
{
   _deviceManagerIf = deviceManager;

   FW_NORMAL_ASSERT(0 != _deviceManagerIf);
}

void Inquiry::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

IStateMachine* Inquiry::getSmEntryInterface(void)
{
   return this;
}

void Inquiry::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // send current status for local control data
   createDiscoveringStatusMsg(bts2AppMsgList, user, handle, false, getStatus());
}

void Inquiry::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetDiscoveringStatus& request, IN const BTSCommonEnumClass statusCode) const
{
   if(false == isValidGetRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   sendStatus(bts2AppMsgList, request.getUser(), request.getSessionHandle(), statusCode);
}

void Inquiry::sendStatusAndStartResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // collect data for sending status and result
   BTSDiscoveringStatus sendStatus;
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if(false == isValidStartRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      sendStatus = BTS_DISCOVERING_OFF;
      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else if(true == _discovering)
   {
      // inquiry is already active
      sendStatus = getStatus();
      sendResult = BTS_REQ_SUCCESS;
   }
   else
   {
      // inquiry is inactive
      sendStatus = getStatus();
      sendResult = BTS_REQ_FAILED;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // update status
   createDiscoveringStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, sendStatus);
   // update result
   createStartDiscoveryResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

void Inquiry::sendStatusAndStopResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   // collect data for sending status and result
   BTSDiscoveringStatus sendStatus;
   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;

   if(false == isValidStopRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      sendStatus = BTS_DISCOVERING_OFF;
      sendResult = BTS_REQ_INVALID_PARAM;
   }
   else if(true == _discovering)
   {
      // inquiry is active
      sendStatus = getStatus();
      sendResult = BTS_REQ_FAILED;
   }
   else
   {
      // inquiry is already inactive
      sendStatus = getStatus();
      sendResult = BTS_REQ_SUCCESS;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   // update status
   createDiscoveringStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, sendStatus);
   // update result
   createStopDiscoveryResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendResult);
}

bool Inquiry::isValidGetRequest(IN const App2Bts_GetDiscoveringStatus& request) const
{
   (void)(request);

   return true;
}

bool Inquiry::isValidStartRequest(IN const App2Bts_StartDiscovery& request) const
{
   (void)(request);

   return true;
}

bool Inquiry::isValidStopRequest(IN const App2Bts_StopDiscovery& request) const
{
   (void)(request);

   return true;
}

bool Inquiry::startDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request)
{
   ETG_TRACE_USR2((" startDiscovery"));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidStartRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      createDiscoveringStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, BTS_DISCOVERING_OFF);
      createStartDiscoveryResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   // reset flag for stop requested
   _stopRequested = false;
   _internalStop = false;

   // check if inquiry is already active
   if(true == _discovering)
   {
      // inquiry is already active
      createDiscoveringStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, BTS_DISCOVERING_ON);
      createStartDiscoveryResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);
      return false;
   }

   // clear list
   _deviceList.clear();

   // store data and process request
   _ongoing = true;
   _pending = true;
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   _requestIf->startDiscovery(bts2IpcMsgList, bts2AppMsgList);

   return true;
}

bool Inquiry::stopDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request)
{
   ETG_TRACE_USR2((" stopDiscovery"));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidStopRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      createDiscoveringStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, BTS_DISCOVERING_OFF);
      createStopDiscoveryResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   // reset flag for stop requested
   _stopRequested = false;
   _internalStop = false;

   // check if inquiry is already inactive
   if(false == _discovering)
   {
      // inquiry is already inactive
      createDiscoveringStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, BTS_DISCOVERING_OFF);
      createStopDiscoveryResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), BTS_REQ_SUCCESS);
      return false;
   }

   // store data and process request
   _ongoing = true;
   _pending = true;
   request.getCompareItem(_requestItem.item);
   _requestItem.user = request.getUser();
   _requestItem.handle = request.getSessionHandle();
   _requestIf->stopDiscovery(bts2IpcMsgList, bts2AppMsgList);

   return true;
}

void Inquiry::setStopRequested(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // set flag for stop requested
   _stopRequested = true;

   // check if stop can be sent now
   checkForSendingStop(bts2IpcMsgList, bts2AppMsgList);
}

void Inquiry::triggerStopDiscovery(IN const bool highPrio /*= true*/)
{
   FW_IF_NULL_PTR_RETURN(_controlIf);

   App2Bts_StopDiscovery* msg = ptrNew_App2Bts_StopDiscovery();
   if(0 != msg)
   {
      // keep user as NULL
      // keep session handle as 0
      msg->setInternalMessageFlag(true);
   }

   _controlIf->sendInternalApp2BtsMessage(msg, highPrio);
}

bool Inquiry::isDiscoveryOngoing(void) const
{
   // consider current status
   return _discovering;
}

bool Inquiry::isRequestOngoing(void) const
{
   // consider ongoing request (start or stop)
   return _ongoing;
}

void Inquiry::startDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   ETG_TRACE_USR2((" startDiscoveryResult"));

   _pending = false;

   // check result (if inquiry is already ongoing then handle this as success)
   if((BTS_REQ_SUCCESS == result) || (true == _discovering))
   {
      // start inquiry is successful
      // wait for status update --- DISCOVERING ON
      checkForStartInquiryCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
   else
   {
      if(false == _ongoing)
      {
         // start inquiry sequence is already finished
         FW_NORMAL_ASSERT_ALWAYS();
         return;
      }

      // start inquiry sequence is finished
      handleRequestCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, false, true);
   }
}

void Inquiry::stopDiscoveryResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   ETG_TRACE_USR2((" stopDiscoveryResult"));

   _pending = false;

   // check result (if inquiry is already stopped then handle this as success)
   if((BTS_REQ_SUCCESS == result) || (false == _discovering))
   {
      // stop inquiry is successful
      // wait for status update --- DISCOVERING OFF
      checkForStopInquiryCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem);
   }
   else
   {
      if(false == _ongoing)
      {
         // stop inquiry sequence is already finished
         FW_NORMAL_ASSERT_ALWAYS();
         return;
      }

      // stop inquiry sequence is finished
      handleRequestCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, false, false);
   }
}

void Inquiry::updateStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveringStatus status)
{
   ETG_TRACE_USR2((" updateStatus"));

   // store new status
   _discovering = (BTS_DISCOVERING_ON == status);

   // check for ongoing start or stop request
   if(true == _ongoing)
   {
      // check if completed
      if(App2BtsOC_StartDiscovery == _requestItem.item.opCode)
      {
         checkForStartInquiryCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem);
      }
      else
      {
         checkForStopInquiryCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem);
      }
   }
   else
   {
      // spontaneous update => forward
      createDiscoveringStatusMsg(bts2AppMsgList, 0, 0, true, getStatus());
      informObservers(bts2IpcMsgList, bts2AppMsgList, messageItem, getStatus());
   }
}

void Inquiry::deviceFound(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveredDeviceInfo& device)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   ETG_TRACE_USR2((" deviceFound: address=%s", device.address.c_str()));

   // add to list
   _deviceList.insert(device.address);

   // forward only if inquiry is ongoing
   if(false == _discovering)
   {
      return;
   }

   // forward found device to user requested the inquiry
   createDiscoveredDeviceFoundMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, false, device);
}

App2Bts_BaseMessage* Inquiry::getApp2BtsWorkingMessage(void)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(_requestItem.item);
}

void Inquiry::registerObserver(IN IInquiryObserver* observer)
{
   FW_IF_NULL_PTR_RETURN(observer);

   _observerList.insert(observer);
}

void Inquiry::createDiscoveringStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDiscoveringStatus status) const
{
   Bts2App_DiscoveringStatus* msg = ptrNew_Bts2App_DiscoveringStatus();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setDiscoveringStatus(status);

      bts2AppMsgList.push_back(msg);
   }
}

void Inquiry::createStartDiscoveryResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_StartDiscoveryResult* msg = ptrNew_Bts2App_StartDiscoveryResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void Inquiry::createStopDiscoveryResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_StopDiscoveryResult* msg = ptrNew_Bts2App_StopDiscoveryResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void Inquiry::createDiscoveredDeviceFoundMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSDiscoveredDeviceInfo& device) const
{
   Bts2App_DiscoveredDeviceFound* msg = ptrNew_Bts2App_DiscoveredDeviceFound();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setDiscoveredDeviceInfo(device);

      bts2AppMsgList.push_back(msg);
   }
}

void Inquiry::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   if(true == _internalStop)
   {
      // in case of internal stop we have no user request => fake necessary data
      messageItem.item.opCode = App2BtsOC_StopDiscovery;
      messageItem.deleteMessage = true;
   }
   else
   {
      messageItem.item = _requestItem.item;
      messageItem.deleteMessage = true;
      if(0 == messageItem.message)
      {
         messageItem.message = getApp2BtsWorkingMessage();
      }
      FW_NORMAL_ASSERT(0 != messageItem.message);
   }
}

void Inquiry::checkForStartInquiryCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   /*
    * If we trigger a start inquiry we have to check following:
    * - start inquiry result
    * - discovering status
    *
    * This function is not called in case of start inquiry result returns with failed result.
    */

   if(false == _ongoing)
   {
      // start inquiry sequence is already finished
      return;
   }

   bool pending = false;

   if(true == _pending)
   {
      // start inquiry result is still pending
      pending = true;
   }

   else if(false == _discovering)
   {
      // inquiry not started
      pending = true;
   }

   if(true == pending)
   {
      // start inquiry sequence is ongoing
   }
   else
   {
      // start inquiry sequence is finished
      handleRequestCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, true, true);
      informObservers(bts2IpcMsgList, bts2AppMsgList, messageItem, getStatus());
   }
}

void Inquiry::checkForStopInquiryCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem)
{
   /*
    * If we trigger a stop inquiry we have to check following:
    * - stop inquiry result
    * - discovering status
    *
    * This function is not called in case of stop inquiry result returns with failed result.
    */

   if(false == _ongoing)
   {
      // stop inquiry sequence is already finished
      return;
   }

   bool pending = false;

   if(true == _pending)
   {
      // stop inquiry result is still pending
      pending = true;
   }

   else if(true == _discovering)
   {
      // inquiry not stopped
      pending = true;
   }

   if(true == pending)
   {
      // stop inquiry sequence is ongoing
   }
   else
   {
      // stop inquiry sequence is finished
      handleRequestCompleted(bts2IpcMsgList, bts2AppMsgList, messageItem, true, false);
      informObservers(bts2IpcMsgList, bts2AppMsgList, messageItem, getStatus());
   }
}

void Inquiry::handleRequestCompleted(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool sendStatusToAll, IN const bool start)
{
   // start/stop inquiry sequence is finished
   _ongoing = false;

   // update status and result
   createDiscoveringStatusMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, sendStatusToAll, getStatus());

   if(true == start)
   {
      createStartDiscoveryResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_SUCCESS);
   }
   else
   {
      createStopDiscoveryResultMsg(bts2AppMsgList, _requestItem.user, _requestItem.handle, BTS_REQ_SUCCESS);
   }

   // action is finished
   handleActionFinished(messageItem);

   // reset control data because action is finished
   _requestItem.reset();

   // request is completed now => reset flag
   _internalStop = false;

   // check if stop can be sent now
   checkForSendingStop(bts2IpcMsgList, bts2AppMsgList);
}

void Inquiry::informObservers(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSDiscoveringStatus status) const
{
   for(::std::set< IInquiryObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
   {
      if(0 != *it)
      {
         (*it)->discoveringStatusChanged(bts2IpcMsgList, bts2AppMsgList, messageItem, status);
      }
   }
}

void Inquiry::checkForSendingStop(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   // stop requested ?
   if(false == _stopRequested)
   {
      // no stop requested
      return;
   }
   // start/stop request ongoing?
   else if(true == _ongoing)
   {
      // request ongoing
      return;
   }
   // discovering on?
   else if(false == _discovering)
   {
      // discovering inactive
      return;
   }

   ETG_TRACE_USR2((" checkForSendingStop"));

   // all preconditions are now full filled to send a stop
   _ongoing = true;
   _pending = true;
   _internalStop = true;
   _requestIf->stopDiscovery(bts2IpcMsgList, bts2AppMsgList);
}

} //btstackif
