/**
 * @file HfpDeviceCapabilitiesData.h
 *
 * @par SW-Component
 * State machine for HFP device capabilities
 *
 * @brief Implementation of HFP device capabilities data.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of HFP device capabilities data.
 */

#ifndef _HFP_DEVICE_CAPABILITIES_DATA_H_
#define _HFP_DEVICE_CAPABILITIES_DATA_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

/**
 * HFP device capabilities data class.
 */
class HfpDeviceCapabilitiesData
{
public:
   /**
    * Type definition for inband ringtone data available.
    * #InbandRingtoneDataAvailableBit#ignore#
    */
   enum InbandRingtoneDataAvailableBit
   {
      SUPPORTED_RECEIVED = 0, /**< flag for supported information received */
      ENABLED_RECEIVED, /**< flag for enabled information received */
      UPDATE_DONE, /**< flag for update status to application */
      // -------------------------------------------------------------------------
      IRDA_LAST /**< end marker, must be the last one */
   };

   /**
    * Type definition for inband ringtone data available.
    */
   typedef BTSBitfield<unsigned char, InbandRingtoneDataAvailableBit, IRDA_LAST> InbandRingtoneDataAvailable;

   /**
    * Default constructor.
    */
   HfpDeviceCapabilitiesData();

   /**
    * Destructor.
    */
   virtual ~HfpDeviceCapabilitiesData();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   HfpDeviceCapabilitiesData(const HfpDeviceCapabilitiesData& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   HfpDeviceCapabilitiesData& operator=(const HfpDeviceCapabilitiesData& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator==(const HfpDeviceCapabilitiesData& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    */
   bool operator!=(const HfpDeviceCapabilitiesData& ref) const;

   // member with public access
   bool modemAdded; /**< flag for added modem */
   bool modemAvailable; /**< flag for available modem */
   bool hfInterfaceAdded; /**< flag for added HF interface */
   bool hfInterfaceAvailable; /**< flag for available HF interface */
   bool ringtoneSupported; /**< flag for inband ringtone support */
   bool ringtoneEnabled; /**< flag for inband ringtone enabled */
   InbandRingtoneDataAvailable ringtoneDataAvailable; /**< inband ringtone data availability information */
};

} //btstackif

#endif //_HFP_DEVICE_CAPABILITIES_DATA_H_
