/**
 * @file HfpDeviceCapabilities.h
 *
 * @par SW-Component
 * State machine for HFP device capabilities
 *
 * @brief Implementation of generic HFP device capabilities state machine.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic HFP device capabilities state machine.
 */

#ifndef _HFP_DEVICE_CAPABILITIES_H_
#define _HFP_DEVICE_CAPABILITIES_H_

#include "StateMachine.h"
#include "IHfpDeviceCapabilities.h"
#include "IHfpDeviceCapabilitiesCallback.h"
#include "HfpDeviceCapabilitiesData.h"

namespace btstackif {

/**
 * HfpDeviceCapabilities class.
 */
class HfpDeviceCapabilities : public StateMachine, public IHfpDeviceCapabilities, public IHfpDeviceCapabilitiesCallback
{
public:
   /**
    * Default constructor.
    */
   HfpDeviceCapabilities();

   /**
    * Destructor.
    */
   virtual ~HfpDeviceCapabilities();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setInstance(IN IHfpDeviceCapabilitiesRequest* instance);

   virtual void setControlIf(IN IBasicControl* control);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual void getModemsResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result);

   virtual void getModemPropertiesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void getHandsfreePropertiesResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result);

   virtual void updateHandsfreeFeatures(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSHandsfreeAgFeatures& features);

   virtual void updateHandsfreeInbandRinging(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool inbandRingingEnabled);

   virtual void indicateManagerServiceAvailability(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool available);

   virtual void modemAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void modemRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void modemAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void modemUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void handsfreeInterfaceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void handsfreeInterfaceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void handsfreeInterfaceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

   virtual void handsfreeInterfaceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address);

private:
   IHfpDeviceCapabilitiesRequest* _requestIf; /**< request interface */
   IBasicControl* _controlIf; /**< basic control interface */
   bool _managerServiceAvailable; /**< availability flag for HFP manager service */
   ::std::map< BTSBDAddress, HfpDeviceCapabilitiesData > _deviceList; /**< map containing device capabilities information */

   HfpDeviceCapabilitiesData& checkDeviceList(IN const BTSBDAddress& address);

   void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const HfpDeviceCapabilitiesData& entry) const;
};

} //btstackif

#endif //_HFP_DEVICE_CAPABILITIES_H_
