/**
 * @file DeviceManager.cpp
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Implementation of generic device manager state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic device manager state machine.
 */

#include "DeviceManager.h"
#include "IDeviceManagerRequest.h"
#include "IBasicControl.h"
#include "IDeviceObserver.h"
#include "ITimerPool.h"
#include "Timer.h"
#include "FwAssert.h"
#include "FwStringUtils.h"
#include "FwBluetoothStringUtils.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DeviceManager.cpp.trc.h"
#endif
#endif

namespace btstackif {

DeviceManager::DeviceManager() :
_requestIf(0),
_controlIf(0),
_timerPoolIf(0),
_deviceList(),
_observerList(),
_timeoutUpdateDisconnected(12000),
_retryTimeout(200),
_defaultRetryMax(1)
{
}

DeviceManager::~DeviceManager()
{
   _requestIf = 0;
   _controlIf = 0;
   _timerPoolIf = 0;
}

void DeviceManager::reset(void)
{
   StateMachine::reset();
   // keep _requestIf
   // keep _controlIf
   // keep _timerPoolIf
   // stop and release all timer
   for(::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      releaseTimer(it->second);
   }
   _deviceList.clear();
   // keep _observerList

   FW_IF_NULL_PTR_RETURN(_requestIf);
   _requestIf->reset();
}

void DeviceManager::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   // check current state/action
   for(::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      const BTSBDAddress& workingAddress = it->first;
      const DeviceManagerData& entry = it->second;

      // check for ongoing disconnect
      if(true == entry.info.getBit(DeviceManagerData::ONGOING))
      {
         // send result as success
         createDeviceConnectionStatusMsg(bts2AppMsgList, 0, 0, true, workingAddress, BTS_CONN_DISCONNECTED, BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL);
         createDisconnectDeviceResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, BTS_REQ_SUCCESS);
      }
      else
      {
         // check for connected
         if(true == isConnected(entry))
         {
            // update status as disconnected
            createDeviceConnectionStatusMsg(bts2AppMsgList, 0, 0, true, workingAddress, BTS_CONN_DISCONNECTED, BTS_DISCONNECT_REASON_ABNORMAL_LOSS);
         }
      }
   }

   // reset control data
   reset();
}

void DeviceManager::setInstance(IN IDeviceManagerRequest* instance)
{
   _requestIf = instance;

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setCallback(this);
}

void DeviceManager::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);

   FW_IF_NULL_PTR_RETURN(_requestIf);

   _requestIf->setControlIf(_controlIf);
}

void DeviceManager::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

IStateMachine* DeviceManager::getSmEntryInterface(void)
{
   return this;
}

void DeviceManager::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   if(false == ::fw::isValidBdAddress(address))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   BTSConnectionStatus status(BTS_CONN_DISCONNECTED);
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_NOT_VALID);
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      if(true == isConnected(it->second))
      {
         status = BTS_CONN_CONNECTED;
      }
      else
      {
         reason = it->second.reason;
      }
   }

   createDeviceConnectionStatusMsg(bts2AppMsgList, user, handle, false, workingAddress, status, reason);
}

void DeviceManager::sendStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_GetDeviceConnectionStatus& request, IN const BTSCommonEnumClass statusCode) const
{
   if(false == isValidGetRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   sendStatus(bts2AppMsgList, request.getBDAddress(), request.getUser(), request.getSessionHandle(), statusCode);
}

void DeviceManager::sendStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) const
{
   (void)(statusCode);

   if(false == isValidDisconnectRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // use given address
      createDeviceConnectionStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, request.getBDAddress(), BTS_CONN_DISCONNECTED, BTS_DISCONNECT_REASON_NOT_VALID);
      createDisconnectDeviceResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), request.getBDAddress(), BTS_REQ_INVALID_PARAM);
      return;
   }

   BTSRequestResult sendResult;
   const BTSRequestResult inputResult = (BTSRequestResult)resultCode;
   BTSConnectionStatus status(BTS_CONN_DISCONNECTED);
   BTSDisconnectReason reason(BTS_DISCONNECT_REASON_NOT_VALID);
   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      if(true == isConnected(it->second))
      {
         status = BTS_CONN_CONNECTED;
      }
      else
      {
         reason = it->second.reason;
      }
   }

   // check if result code has to be corrected
   if(BTS_CONN_DISCONNECTED == status)
   {
      // device is disconnected => device disconnect was successful
      sendResult = BTS_REQ_SUCCESS;
   }
   else
   {
      // device is connected => device disconnect failed
      sendResult = BTS_REQ_FAILED;
   }

   // check given result
   if((BTS_REQ_LAST > inputResult) && (BTS_REQ_SUCCESS != inputResult) && (BTS_REQ_SUCCESS != sendResult))
   {
      // use given result
      sendResult = inputResult;
   }

   createDeviceConnectionStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), sendStatusToAll, workingAddress, status, reason);
   createDisconnectDeviceResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, sendResult);
}

bool DeviceManager::isValidGetRequest(IN const App2Bts_GetDeviceConnectionStatus& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool DeviceManager::isValidDisconnectRequest(IN const App2Bts_DisconnectDevice& request) const
{
   return ::fw::isValidBdAddress(request.getBDAddress());
}

bool DeviceManager::disconnectDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request)
{
   ETG_TRACE_USR2((" disconnectDevice: address=%s", request.getBDAddress().c_str()));

   FW_IF_NULL_PTR_RETURN_FALSE(_requestIf);

   if(false == isValidDisconnectRequest(request))
   {
      FW_NORMAL_ASSERT_ALWAYS();

      // use given address
      createDeviceConnectionStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, request.getBDAddress(), BTS_CONN_DISCONNECTED, BTS_DISCONNECT_REASON_NOT_VALID);
      createDisconnectDeviceResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), request.getBDAddress(), BTS_REQ_INVALID_PARAM);
      return false;
   }

   BTSBDAddress workingAddress(request.getBDAddress());
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() == it)
   {
      // device not in list => send direct answer
      createDeviceConnectionStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, workingAddress, BTS_CONN_DISCONNECTED, BTS_DISCONNECT_REASON_NOT_VALID);
      createDisconnectDeviceResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, BTS_REQ_SUCCESS);
      return false;
   }

   DeviceManagerData& entry = it->second;

   // in case of getting device disconnect request we will also check ACL status
   const bool connected(((true == entry.info.getBit(DeviceManagerData::CONNECTED)) || (true == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)) || (true == entry.info.getBit(DeviceManagerData::ACL_CONNECTED))));

   if(false == connected)
   {
      // device not connected => send direct answer
      createDeviceConnectionStatusMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), false, workingAddress, BTS_CONN_DISCONNECTED, entry.reason);
      createDisconnectDeviceResultMsg(bts2AppMsgList, request.getUser(), request.getSessionHandle(), workingAddress, BTS_REQ_SUCCESS);
      return false;
   }

   // store data and process request
   entry.info.setBit(DeviceManagerData::ONGOING);
   entry.info.setBit(DeviceManagerData::PENDING);
   entry.retryMax = _defaultRetryMax;
   entry.retryNmb = 0;
   request.getCompareItem(entry.requestItem.item);
   entry.requestItem.user = request.getUser();
   entry.requestItem.handle = request.getSessionHandle();
   entry.pauseBtStreaming = request.getPauseBtStreaming();
   _requestIf->disconnect(bts2IpcMsgList, bts2AppMsgList, workingAddress, request.getPauseBtStreaming());

   return true;
}

void DeviceManager::createDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   ETG_TRACE_USR2((" createDevice: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_requestIf);

   // all necessary checks are done before (valid address and device not available)

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   FW_NORMAL_ASSERT(true == ::fw::isValidBdAddress(workingAddress));

   DeviceManagerData& entry = checkDeviceList(workingAddress);

   if((false == entry.info.getBit(DeviceManagerData::DEVICE_AVAILABLE)) && (false == entry.info.getBit(DeviceManagerData::CREATING)))
   {
      entry.info.setBit(DeviceManagerData::CREATING);

      // TODO: [low]: timer is needed for device available

      _requestIf->create(bts2IpcMsgList, bts2AppMsgList, address);
   }
}

void DeviceManager::removeDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address)
{
   ETG_TRACE_USR2((" removeDevice: address=%s", address.c_str()));

   FW_IF_NULL_PTR_RETURN(_requestIf);

   // all necessary checks are done before (valid address and device available)

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);
   FW_NORMAL_ASSERT(true == ::fw::isValidBdAddress(workingAddress));

   DeviceManagerData& entry = checkDeviceList(workingAddress);

   if(false == entry.info.getBit(DeviceManagerData::REMOVING))
   {
      entry.info.setBit(DeviceManagerData::REMOVING);
      _requestIf->remove(bts2IpcMsgList, bts2AppMsgList, address);
   }
}

void DeviceManager::removeAllDevices(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   ETG_TRACE_USR2((" removeAllDevices"));

   FW_IF_NULL_PTR_RETURN(_requestIf);

   // all necessary checks are done before

   for(::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      if(false == it->second.info.getBit(DeviceManagerData::REMOVING))
      {
         it->second.info.setBit(DeviceManagerData::REMOVING);
         _requestIf->remove(bts2IpcMsgList, bts2AppMsgList, it->first);
      }
   }
}

bool DeviceManager::isAnyDeviceConnected(IN const bool includingAclLink /*= false*/) const
{
   for(::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      if(true == includingAclLink)
      {
         if((true == it->second.info.getBit(DeviceManagerData::CONNECTED)) || (true == it->second.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)) || (true == it->second.info.getBit(DeviceManagerData::ACL_CONNECTED)))
         {
            return true;
         }
      }
      else
      {
         if(true == isConnected(it->second))
         {
            return true;
         }
      }
   }

   return false;
}

bool DeviceManager::isDeviceConnected(IN const BTSBDAddress& address, IN const bool includingAclLink /*= false*/) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      if(true == includingAclLink)
      {
         return ((true == it->second.info.getBit(DeviceManagerData::CONNECTED)) || (true == it->second.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)) || (true == it->second.info.getBit(DeviceManagerData::ACL_CONNECTED)));
      }
      else
      {
         return isConnected(it->second);
      }
   }

   return false;
}

bool DeviceManager::isAnyDisconnectOngoing(void) const
{
   for(::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      if(true == it->second.info.getBit(DeviceManagerData::ONGOING))
      {
         return true;
      }
   }

   return false;
}

bool DeviceManager::isDisconnectOngoing(IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      return it->second.info.getBit(DeviceManagerData::ONGOING);
   }

   return false;
}

bool DeviceManager::isDeviceAdded(IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      return it->second.info.getBit(DeviceManagerData::DEVICE_ADDED);
   }

   return false;
}

bool DeviceManager::isDeviceAvailable(IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      return it->second.info.getBit(DeviceManagerData::DEVICE_AVAILABLE);
   }

   return false;
}

void DeviceManager::getDeviceData(OUT BTSDeviceName& name, OUT BTSCod& cod, IN const BTSBDAddress& address)
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      name = it->second.name;
      cod = it->second.cod;
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

bool DeviceManager::getRole(IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      return it->second.master;
   }

   FW_NORMAL_ASSERT_ALWAYS();
   return true;
}

bool DeviceManager::getEnableAvpPause(IN const BTSBDAddress& address) const
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   ::std::map< BTSBDAddress, DeviceManagerData >::const_iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() != it)
   {
      return it->second.enableAvpPause;
   }

   FW_NORMAL_ASSERT_ALWAYS();
   return true;
}

void DeviceManager::updateAnyProfileConnected(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const bool connected, IN const BTSDisconnectReason reason)
{
   (void)(bts2IpcMsgList);

   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   DeviceManagerData& entry = checkDeviceList(workingAddress);

   // remember old connected data
   const bool oldConnected((true == isConnected(entry)));
   BTSStatusTransition profileTransition(BTS_STATUS_TRANSITION_NO_CHANGE);

   // set device connected state to true only in case of first protocol is connected
   if((false == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)) && (true == connected))
   {
      entry.info.setBit(DeviceManagerData::CONNECTED);

      entry.disconnectedByException = false;

      profileTransition = BTS_STATUS_TRANSITION_ENABLED;
   }

   // check for last protocol disconnect
   if((true == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)) && (false == connected))
   {
      // take over disconnect reason
      if(BTS_DISCONNECT_REASON_LAST > reason)
      {
         entry.reason = reason;
      }

      profileTransition = BTS_STATUS_TRANSITION_DISABLED;
   }

   if(true == connected)
   {
      entry.info.setBit(DeviceManagerData::ANY_PROFILE_CONNECTED);
   }
   else
   {
      entry.info.resetBit(DeviceManagerData::ANY_PROFILE_CONNECTED);
   }

   // check new connected data
   const bool newConnected((true == isConnected(entry)));

   checkConnectionStatus(bts2AppMsgList, workingAddress, entry, oldConnected, newConnected);

   checkForDisconnectCompleted(bts2AppMsgList, messageItem, workingAddress, entry);

   sendConnectionStatusToObserver(bts2IpcMsgList, bts2AppMsgList, messageItem, workingAddress, BTS_STATUS_TRANSITION_NO_CHANGE, entry.info.getBit(DeviceManagerData::ACL_CONNECTED), profileTransition, entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED));
}

void DeviceManager::indicateDisconnectByException(IN const BTSBDAddress& address)
{
   BTSBDAddress workingAddress(address);
   ::fw::convertString2LowerCase(workingAddress);

   // if this method is called then an entry for given device shall exist
   DeviceManagerData& entry = checkDeviceList(workingAddress);

   if(false == entry.disconnectedByException)
   {
      // write only one error memory entry during same device connection cycle
      entry.disconnectedByException = true;

      // write error memory entry
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: device %s disconnected by EXCEPTION", workingAddress.c_str()));
   }
}

void DeviceManager::registerObserver(IN IDeviceObserver* observer)
{
   FW_IF_NULL_PTR_RETURN(observer);

   _observerList.insert(observer);
}

void DeviceManager::disconnectResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   (void)(bts2IpcMsgList);

   ETG_TRACE_USR2((" disconnectResult: address=%s", address.c_str()));

   const BTSBDAddress& workingAddress = address;

   ::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.find(workingAddress);
   if(_deviceList.end() == it)
   {
      // should never happen
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   DeviceManagerData& entry = it->second;

   entry.info.resetBit(DeviceManagerData::PENDING);

   // check result (if ACL is already disconnected then handle this as success)
   if((BTS_REQ_SUCCESS == result) || (false == entry.info.getBit(DeviceManagerData::ACL_CONNECTED)))
   {
      // disconnect request is successful
      // wait for status update --- CONNECTED
      // wait for status update --- ACL_CONNECTED
      // entry.reason was set before --- we have to ensure that disconnect result is sent after last protocol was indicated as disconnected
      checkForDisconnectCompleted(bts2AppMsgList, messageItem, workingAddress, entry);

      // start timer if device is still connected (ACL or any profile)
      if((true == entry.info.getBit(DeviceManagerData::ACL_CONNECTED)) || (true == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)))
      {
         startTimer(entry, _timeoutUpdateDisconnected);
         entry.waitForDisconnectedUpdate = true;
      }
   }
   else
   {
      if(false == entry.info.getBit(DeviceManagerData::ONGOING))
      {
         // disconnect sequence is already finished
         FW_NORMAL_ASSERT_ALWAYS();
         return;
      }

      // check for retry in case of failed disconnect (result is failed and ACL is still connected)
      if(entry.retryMax > entry.retryNmb)
      {
         // execute retry after timeout
         entry.retryNmb++;
         startTimer(entry, _retryTimeout);
         entry.waitForDisconnectedUpdate = false;
      }
      else
      {
         // disconnect sequence is finished
         entry.info.resetBit(DeviceManagerData::ONGOING);

         // keep current connection status
         BTSConnectionStatus status(BTS_CONN_DISCONNECTED);
         const BTSRequestResult appResult(BTS_REQ_FAILED);

         // in case of getting device disconnect request we will also check ACL status
         const bool connected(((true == entry.info.getBit(DeviceManagerData::CONNECTED)) || (true == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED)) || (true == entry.info.getBit(DeviceManagerData::ACL_CONNECTED))));

         if(true == connected)
         {
            status = BTS_CONN_CONNECTED;
         }

         // update status and result
         createDeviceConnectionStatusMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, false, workingAddress, status, entry.reason);
         createDisconnectDeviceResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, workingAddress, appResult);

         // disconnect sequence is finished; keep device entry in every case

         handleActionFinished(messageItem, entry.requestItem.item);

         // reset control data because action is finished
         entry.requestItem.reset();
         entry.pauseBtStreaming = false;
      }
   }
}

void DeviceManager::createResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result)
{
   // wait for AddDeviceObjectPathMapping and deviceAvailable() call, related entry is updated

   if(BTS_REQ_SUCCESS != result)
   {
      for(::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.begin(); it != _deviceList.end(); ++it)
      {
         if((address == it->first) && (true == it->second.info.getBit(DeviceManagerData::CREATING)))
         {
            it->second.info.resetBit(DeviceManagerData::CREATING);

            // check observer list
            for(::std::set< IDeviceObserver* >::const_iterator itObs = _observerList.begin(); itObs != _observerList.end(); ++itObs)
            {
               if(0 != *itObs)
               {
                  (*itObs)->deviceCreationFinished(bts2IpcMsgList, bts2AppMsgList, messageItem, it->first, BTS_REQ_FAILED);
               }
            }

            break;
         }
      }
   }
}

void DeviceManager::removeResult(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result)
{
   (void)(bts2IpcMsgList);
   (void)(bts2AppMsgList);
   (void)(messageItem);
   (void)(result);

   // wait for DelDeviceObjectPathMapping, related entry is removed from device list

   // note: this feature is currently only used for testing purpose, therefore no error handling necessary
}

void DeviceManager::updateAclStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason)
{
   (void)(bts2IpcMsgList);

   DeviceManagerData& entry = checkDeviceList(address);

   // remember old connected data
   const bool oldConnected((true == isConnected(entry)));
   BTSStatusTransition aclTransition(BTS_STATUS_TRANSITION_NO_CHANGE);

   // set device connected state to true only in case of first protocol is connected
   if(BTS_CONN_CONNECTED != status)
   {
      entry.info.resetBit(DeviceManagerData::CONNECTED);
   }

   // check for device connect
   if((false == entry.info.getBit(DeviceManagerData::ACL_CONNECTED)) && (BTS_CONN_CONNECTED == status))
   {
      aclTransition = BTS_STATUS_TRANSITION_ENABLED;
   }

   // check for device disconnect
   if((true == entry.info.getBit(DeviceManagerData::ACL_CONNECTED)) && (BTS_CONN_CONNECTED != status))
   {
      // take over disconnect reason
      if(BTS_DISCONNECT_REASON_LAST > reason)
      {
         entry.reason = reason;
      }

      aclTransition = BTS_STATUS_TRANSITION_DISABLED;
   }

   if(BTS_CONN_CONNECTED == status)
   {
      entry.info.setBit(DeviceManagerData::ACL_CONNECTED);
   }
   else
   {
      entry.info.resetBit(DeviceManagerData::ACL_CONNECTED);
   }

   entry.info.setBit(DeviceManagerData::ACL_STATUS_UPDATED);

   // check for device available shall be done in specific implementation

   // check new connected data
   const bool newConnected((true == isConnected(entry)));

   // handle following situation:
   // - ACL is connected but no protocol is connected
   // - disconnect device is triggered from upper layer
   // - ACL gets disconnected but no update happens for device connection status
   // => force device connection status update
   if((false == oldConnected) && (oldConnected == newConnected) && (BTS_STATUS_TRANSITION_DISABLED == aclTransition))
   {
      ETG_TRACE_USR2((" updateAclStatus: address=%s: forced update", address.c_str()));
      checkConnectionStatus(bts2AppMsgList, address, entry, true, newConnected);
   }
   else
   {
      checkConnectionStatus(bts2AppMsgList, address, entry, oldConnected, newConnected);
   }

   checkForDisconnectCompleted(bts2AppMsgList, messageItem, address, entry);

   sendConnectionStatusToObserver(bts2IpcMsgList, bts2AppMsgList, messageItem, address, aclTransition, entry.info.getBit(DeviceManagerData::ACL_CONNECTED), BTS_STATUS_TRANSITION_NO_CHANGE, entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED));
}

void DeviceManager::updateName(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSDeviceName& name)
{
   (void)(bts2IpcMsgList);
   (void)(messageItem);

   DeviceManagerData& entry = checkDeviceList(address);
   entry.name = name;
   entry.info.setBit(DeviceManagerData::NAME_UPDATED);

   // check for device available shall be done in specific implementation

   // update name to clients if device is marked as available
   updateDeviceName(bts2AppMsgList, address, entry.name, entry.info.getBit(DeviceManagerData::DEVICE_AVAILABLE));
}

void DeviceManager::updateCod(IN const BTSBDAddress& address, IN const BTSCod cod)
{
   DeviceManagerData& entry = checkDeviceList(address);
   entry.cod = cod;
}

void DeviceManager::updateRole(IN const BTSBDAddress& address, IN const bool master)
{
   DeviceManagerData& entry = checkDeviceList(address);
   entry.master = master;
   entry.info.setBit(DeviceManagerData::ROLE_UPDATED);

   // check for device available shall be done in specific implementation
}

void DeviceManager::updateEnableAvpPause(IN const BTSBDAddress& address, IN const bool enableAvpPause)
{
   DeviceManagerData& entry = checkDeviceList(address);
   entry.enableAvpPause = enableAvpPause;
   entry.info.setBit(DeviceManagerData::ENABLEAVPPAUSE_UPDATED);
}

void DeviceManager::updatePowerMode(IN const BTSBDAddress& address, IN const BTSBtPowerMode mode)
{
   DeviceManagerData& entry = checkDeviceList(address);
   entry.powerMode = mode;
   entry.info.setBit(DeviceManagerData::MODE_UPDATED);

   // check for device available shall be done in specific implementation
}

void DeviceManager::deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   DeviceManagerData& entry = checkDeviceList(address);

   if(false == entry.info.getBit(DeviceManagerData::DEVICE_ADDED))
   {
      ETG_TRACE_USR2((" deviceAdded: address=%s", address.c_str()));

      entry.info.setBit(DeviceManagerData::DEVICE_ADDED);

      // check observer list
      for(::std::set< IDeviceObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->deviceAdded(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
         }
      }
   }
}

void DeviceManager::deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.find(address);
   if(_deviceList.end() != it)
   {
      // stop and release timer before removing from list
      releaseTimer(it->second);

      _deviceList.erase(it);

      ETG_TRACE_USR2((" deviceRemoved: _deviceList.size()=%u", _deviceList.size()));

      // check observer list
      for(::std::set< IDeviceObserver* >::const_iterator itObs = _observerList.begin(); itObs != _observerList.end(); ++itObs)
      {
         if(0 != *itObs)
         {
            (*itObs)->deviceRemoved(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
         }
      }
   }
}

void DeviceManager::deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   DeviceManagerData& entry = checkDeviceList(address);

   if(false == entry.info.getBit(DeviceManagerData::DEVICE_AVAILABLE))
   {
      ETG_TRACE_USR2((" deviceAvailable: address=%s", address.c_str()));

      entry.info.setBit(DeviceManagerData::DEVICE_AVAILABLE);
      entry.info.resetBit(DeviceManagerData::CREATING);

      // check observer list
      for(::std::set< IDeviceObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->deviceAvailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
         }
      }

      // device is now available => update name to clients
      updateDeviceName(bts2AppMsgList, address, entry.name, entry.info.getBit(DeviceManagerData::DEVICE_AVAILABLE));
   }
}

void DeviceManager::deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address)
{
   DeviceManagerData& entry = checkDeviceList(address);

   if(true == entry.info.getBit(DeviceManagerData::DEVICE_AVAILABLE))
   {
      ETG_TRACE_USR2((" deviceUnavailable: address=%s", address.c_str()));

      entry.info.resetBit(DeviceManagerData::DEVICE_AVAILABLE);

      // check observer list
      for(::std::set< IDeviceObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->deviceUnavailable(bts2IpcMsgList, bts2AppMsgList, messageItem, address);
         }
      }
   }
}

App2Bts_BaseMessage* DeviceManager::getApp2BtsWorkingMessage(IN const BTSBDAddress& address)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   DeviceManagerData& entry = checkDeviceList(address);

   return _controlIf->findApp2BtsWorkingMessageWrapper(entry.requestItem.item);
}

DeviceManagerData& DeviceManager::getEntry(IN const BTSBDAddress& address)
{
   return checkDeviceList(address);
}

void DeviceManager::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   FW_IF_NULL_PTR_RETURN(_requestIf);

   // find matching entry via timer id
   ::std::map< BTSBDAddress, DeviceManagerData >::iterator it;
   for(it = _deviceList.begin(); it != _deviceList.end(); ++it)
   {
      const DeviceManagerData& entry = it->second;

      if(true == entry.timer.compare(timerId))
      {
         // matching timer found
         break;
      }
   }

   if(_deviceList.end() == it)
   {
      // no match
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   const BTSBDAddress& address = it->first;
   DeviceManagerData& entry = it->second;

   ETG_TRACE_USR2((" handleExtendedTimeout: address=%s", address.c_str()));

   if(true == entry.waitForDisconnectedUpdate)
   {
      // send virtual ACL status update message (to follow normal sequence)
      _requestIf->sendVirtualDeviceConnectedUpdate(bts2IpcMsgList, bts2AppMsgList, address, false, BTS_IPC_SUCCESS);

      // restart timer
      startTimer(entry, _timeoutUpdateDisconnected);

      // write error memory entry
      ETG_TRACE_ERRMEM((" #CONN: BtStackIf: disconnect for device %s failed", address.c_str()));

      /*
       * background information:
       * during last months we observed several times that device disconnect was not successful
       * all issues are not reproducible and therefore no full logs are available
       * there is no evidence that the issue is caused by connectivity middleware or Bluetooth stack (full logs are needed to get the evidence)
       * current assumption is following:
       * - issue is caused by ALPS Evolution stack because same issue was also observed long time ago

       * following workaround will be implemented in BtStackIf layer:
       * - background notice: BM core triggers disconnect for each profile before sending device disconnect; this means that only the ACL is available while sending device disconnect
       * - BtStackIf layer starts a timer while sending device disconnect
       * - timer is stopped while device connection status changes to disconnected
       * - in case of timeout set device connection status to disconnected and continue
       */
   }
   else
   {
      // mark result as pending (all other meta data is still set)
      entry.info.setBit(DeviceManagerData::PENDING);

      // find related application message (to have compare item for disconnect request)
      if(0 == messageItem.message)
      {
         messageItem.message = getWorkingMessage(entry.requestItem.item);
      }
      FW_NORMAL_ASSERT(0 != messageItem.message);

      // send disconnect request
      _requestIf->disconnect(bts2IpcMsgList, bts2AppMsgList, address, entry.pauseBtStreaming);
   }
}

DeviceManagerData& DeviceManager::checkDeviceList(IN const BTSBDAddress& address)
{
   ::std::map< BTSBDAddress, DeviceManagerData >::iterator it = _deviceList.find(address);
   if(_deviceList.end() == it)
   {
      // add new entry
      ETG_TRACE_USR2((" checkDeviceList: _deviceList.size()=%u", (1 + _deviceList.size())));
      return _deviceList[address];
   }
   else
   {
      // continue with existing entry
      return it->second;
   }
}

void DeviceManager::checkConnectionStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN DeviceManagerData& entry, IN const bool oldConnected, IN const bool newConnected) const
{
   // handle only changed status
   if(oldConnected != newConnected)
   {
      BTSConnectionStatus status(BTS_CONN_DISCONNECTED);

      // in case of connected we have to reset the disconnect reason
      if(true == newConnected)
      {
         entry.reason = BTS_DISCONNECT_REASON_NOT_VALID;
         status = BTS_CONN_CONNECTED;
      }

      createDeviceConnectionStatusMsg(bts2AppMsgList, 0, 0, true, address, status, entry.reason);
   }
}

void DeviceManager::checkForDisconnectCompleted(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN DeviceManagerData& entry)
{
   /*
    * If we trigger a device disconnect we have to check following:
    * - disconnect result
    * - device connection status
    * - protocol connection status
    *
    * This function is not called in case of disconnect result returns with failed result.
    *
    * HINT: If disconnect result is received as last message from stack there will be 2 device connection status messages to application. TODO: [low]: check with google test
    */

   if(false == entry.info.getBit(DeviceManagerData::ONGOING))
   {
      // disconnect sequence is already finished
      return;
   }

   bool pending = false;

   if(true == entry.info.getBit(DeviceManagerData::PENDING))
   {
      // disconnect result is still pending
      pending = true;
   }

   else if(true == entry.info.getBit(DeviceManagerData::ACL_CONNECTED))
   {
      // device is still connected
      pending = true;
   }

   else if(true == entry.info.getBit(DeviceManagerData::ANY_PROFILE_CONNECTED))
   {
      // any protocol is still connected
      pending = true;
   }

   if(true == pending)
   {
      // disconnect sequence is ongoing
   }
   else
   {
      // disconnect sequence is finished
      entry.info.resetBit(DeviceManagerData::ONGOING);

      // stop timer for waiting for disconnected update
      stopTimer(entry);
      entry.waitForDisconnectedUpdate = false;

      // connection status was updated before
      // createDeviceConnectionStatusMsg(bts2AppMsgList, 0, 0, true, address, BTS_CONN_DISCONNECTED, entry.reason);

      // update result
      createDisconnectDeviceResultMsg(bts2AppMsgList, entry.requestItem.user, entry.requestItem.handle, address, BTS_REQ_SUCCESS);

      // disconnect sequence is finished; keep device entry in every case

      handleActionFinished(messageItem, entry.requestItem.item);

      // reset control data because action is finished
      entry.requestItem.reset();
      entry.pauseBtStreaming = false;
   }
}

void DeviceManager::updateDeviceName(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address, IN const BTSDeviceName& name, IN const bool available) const
{
   if(true == name.empty())
   {
      // ignore, do not update
      return;
   }

   if(false == available)
   {
      // device is marked as not available, do not update
      return;
   }

   Bts2App_RemoteNameStatus* msg = ptrNew_Bts2App_RemoteNameStatus();
   if(0 != msg)
   {
      msg->setUser(0); // send status message to all
      msg->setSessionHandle(0); // send status message to all
      msg->setBDAddress(address);
      msg->setDeviceName(name);

      bts2AppMsgList.push_back(msg);
   }
}

void DeviceManager::createDeviceConnectionStatusMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const bool sendStatusToAll, IN const BTSBDAddress& address, IN const BTSConnectionStatus status, IN const BTSDisconnectReason reason) const
{
   Bts2App_DeviceConnectionStatus* msg = ptrNew_Bts2App_DeviceConnectionStatus();
   if(0 != msg)
   {
      if((true == sendStatusToAll) || (0 == user))
      {
         msg->setUser(0); // send status message to all
         msg->setSessionHandle(0); // send status message to all
      }
      else
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
      }
      msg->setBDAddress(address);
      msg->setConnectionStatus(status);
      if(BTS_CONN_CONNECTED == status)
      {
         // as long as device is connected the disconnect reason is not valid
         msg->setDisconnectReason(BTS_DISCONNECT_REASON_NOT_VALID);
      }
      else
      {
         msg->setDisconnectReason(reason);
      }

      bts2AppMsgList.push_back(msg);
   }
}

void DeviceManager::createDisconnectDeviceResultMsg(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* user, IN const BTSSessionHandle handle, IN const BTSBDAddress& address, IN const BTSRequestResult result) const
{
   if(0 != user)
   {
      Bts2App_DisconnectDeviceResult* msg = ptrNew_Bts2App_DisconnectDeviceResult();
      if(0 != msg)
      {
         msg->setUser(user);
         msg->setSessionHandle(handle);
         msg->setBDAddress(address);
         msg->setRequestResult(result);

         bts2AppMsgList.push_back(msg);
      }
   }
}

void DeviceManager::handleActionFinished(OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSApp2BtsMessageCompareItem& item)
{
   messageItem.item = item;
   messageItem.deleteMessage = true;
   if(0 == messageItem.message)
   {
      messageItem.message = getWorkingMessage(item);
   }
   FW_NORMAL_ASSERT(0 != messageItem.message);
}

App2Bts_BaseMessage* DeviceManager::getWorkingMessage(IN const BTSApp2BtsMessageCompareItem& item)
{
   FW_IF_NULL_PTR_RETURN_NULL(_controlIf);

   return _controlIf->findApp2BtsWorkingMessageWrapper(item);
}

void DeviceManager::startTimer(IN DeviceManagerData& entry, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   entry.timer.setTimerPool(_timerPoolIf);
   entry.timer.start(_controlIf, this, timeout);
}

void DeviceManager::stopTimer(IN DeviceManagerData& entry) const
{
   ETG_TRACE_USR3((" stopTimer"));

   entry.timer.stop();

   // do not release timer
}

void DeviceManager::releaseTimer(IN DeviceManagerData& entry) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   entry.timer.release();
}

void DeviceManager::sendConnectionStatusToObserver(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected)
{
   // check for any change
   if((BTS_STATUS_TRANSITION_ENABLED == aclTransition) || (BTS_STATUS_TRANSITION_DISABLED == aclTransition) || (BTS_STATUS_TRANSITION_ENABLED == anyProfileTransition) || (BTS_STATUS_TRANSITION_DISABLED == anyProfileTransition))
   {
      // check observer list
      for(::std::set< IDeviceObserver* >::const_iterator it = _observerList.begin(); it != _observerList.end(); ++it)
      {
         if(0 != *it)
         {
            (*it)->deviceConnectionStatus(bts2IpcMsgList, bts2AppMsgList, messageItem, address, aclTransition, aclConnected, anyProfileTransition, anyProfileConnected);
         }
      }
   }
}

} //btstackif
