/**
 * @file IDeviceObserver.h
 *
 * @par SW-Component
 * State machine for device manager
 *
 * @brief Interface definition for device observer.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for device observer.
 */

#ifndef _I_DEVICE_OBSERVER_H_
#define _I_DEVICE_OBSERVER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IDeviceObserver
{
public:
   /**
    * Destructor.
    */
   virtual ~IDeviceObserver() {}

   /**
    * Signal for device added.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceAdded(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device removed.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceRemoved(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device available.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceAvailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device unavailable.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    */
   virtual void deviceUnavailable(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address) = 0;

   /**
    * Signal for device connection status.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] aclTransition: transition of ACL connection status
    * @param[in] aclConnected: ACL connection status
    * @param[in] anyProfileTransition: transition of any profile connection status (first profile connected or last profile disconnected)
    * @param[in] anyProfileConnected: any profile connection status
    */
   virtual void deviceConnectionStatus(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSStatusTransition aclTransition, IN const bool aclConnected, IN const BTSStatusTransition anyProfileTransition, IN const bool anyProfileConnected) = 0;

   /**
    * Signal for device creation finished.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] result: result
    */
   virtual void deviceCreationFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result) = 0;

   /**
    * Signal for device removal finished.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] address: device address
    * @param[in] result: result
    */
   virtual void deviceRemovalFinished(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSBDAddress& address, IN const BTSRequestResult result) = 0;
};

} //btstackif

#endif //_I_DEVICE_OBSERVER_H_
