/**
 * @file IConnectionPriority.h
 *
 * @par SW-Component
 * State machine for connection priority
 *
 * @brief Interface definition for generic connection priority state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for connection priority state machine.
 */

#ifndef _I_CONNECTION_PRIORITY_H_
#define _I_CONNECTION_PRIORITY_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IBasicControl;
class IDeviceManager;
class IProtocolManager;
class IInquiry;
class IPairing;
class IServiceSearch;
class IStateMachine;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class App2Bts_DisconnectDevice;
class App2Bts_ConnectProtocol;
class App2Bts_DisconnectProtocol;
class App2Bts_StartDiscovery;
class App2Bts_StopDiscovery;
class App2Bts_StartPairing;
class App2Bts_CancelPairing;
class App2Bts_StartRemoteServiceSearch;
class App2Bts_CancelRemoteServiceSearch;

/**
 * Interface definition.
 */
class IConnectionPriority
{
public:
   /**
    * Destructor.
    */
   virtual ~IConnectionPriority() {}

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set device manager interface.
    *
    * @param[in] deviceManagerIf: device manager interface
    */
   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf) = 0;

   /**
    * Set protocol manager interface.
    *
    * @param[in] deviceManagerIf: protocol manager interface
    */
   virtual void setProtocolManagerIf(IN IProtocolManager* protocolManagerIf) = 0;

   /**
    * Set inquiry interface.
    *
    * @param[in] inquiryIf: inquiry interface
    */
   virtual void setInquiryIf(IN IInquiry* inquiryIf) = 0;

   /**
    * Set pairing interface.
    *
    * @param[in] pairingIf: pairing interface
    */
   virtual void setPairingIf(IN IPairing* pairingIf) = 0;

   /**
    * Set service search interface.
    *
    * @param[in] serviceSearchIf: service search interface
    */
   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Disconnect device.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType disconnectDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request) = 0;

   /**
    * Connect protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = message handling type
    */
   virtual BTSApp2BtsMessageHandlingType connectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request) = 0;

   /**
    * Disconnect protocol.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType disconnectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request) = 0;

   /**
    * Start inquiry.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType startDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request) = 0;

   /**
    * Stop inquiry.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType stopDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request) = 0;

   /**
    * Start pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request) = 0;

   /**
    * Cancel pairing.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelPairing& request) = 0;

   /**
    * Start service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request) = 0;

   /**
    * Cancel service search.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] request: request
    *
    * @return = true: request processing was started (request shall not be deleted),
    * @return = false: no further processing necessary (request can be deleted)
    */
   virtual BTSApp2BtsMessageHandlingType cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelRemoteServiceSearch& request) = 0;

   /**
    * Check end of sequence.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] finished: flag indicating that sequence is finished
    * @param[in] firstConnectFailed: flag indicating that first connect to the device failed
    * @param[in] address: address of the device (only valid if firstConnectFailed flag is true)
    */
   virtual void checkEndOfSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool finished, IN const bool firstConnectFailed = false, IN const BTSBDAddress& address = BTSBDAddress()) = 0;

   /**
    * Get list of similar items (opcodes).
    *
    * @param[out] itemList: similar item list
    * @param[in] message: message we want to have a similar item list for
    */
   virtual void getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage& message) const = 0;

   /**
    * Get list of matching items (opcodes).
    *
    * @param[out] itemList: matching item list
    * @param[out] highPrioItemList: high priority item list
    * @param[in] ptrMessage: message we want to have a matching item list for
    */
   virtual void getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage& message) const = 0;
};

} //btstackif

#endif //_I_CONNECTION_PRIORITY_H_
