/**
 * @file ConnectionPriority.h
 *
 * @par SW-Component
 * State machine for connection priority
 *
 * @brief Implementation of generic connection priority state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Header file for implementation of generic connection priority state machine.
 */

#ifndef _CONNECTION_PRIORITY_H_
#define _CONNECTION_PRIORITY_H_

#include "StateMachine.h"
#include "IConnectionPriority.h"

namespace btstackif {

/**
 * Connection priority class.
 */
class ConnectionPriority : public StateMachine, public IConnectionPriority
{
public:
   /**
    * Default constructor.
    */
   ConnectionPriority();

   /**
    * Destructor.
    */
   virtual ~ConnectionPriority();

   virtual void reset(void);

   virtual void forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList);

   virtual void setControlIf(IN IBasicControl* control);

   virtual void setDeviceManagerIf(IN IDeviceManager* deviceManagerIf);

   virtual void setProtocolManagerIf(IN IProtocolManager* protocolManagerIf);

   virtual void setInquiryIf(IN IInquiry* inquiryIf);

   virtual void setPairingIf(IN IPairing* pairingIf);

   virtual void setServiceSearchIf(IN IServiceSearch* serviceSearchIf);

   virtual IStateMachine* getSmEntryInterface(void);

   virtual BTSApp2BtsMessageHandlingType disconnectDevice(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectDevice& request);

   virtual BTSApp2BtsMessageHandlingType connectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_ConnectProtocol& request);

   virtual BTSApp2BtsMessageHandlingType disconnectProtocol(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_DisconnectProtocol& request);

   virtual BTSApp2BtsMessageHandlingType startDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartDiscovery& request);

   virtual BTSApp2BtsMessageHandlingType stopDiscovery(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StopDiscovery& request);

   virtual BTSApp2BtsMessageHandlingType startPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartPairing& request);

   virtual BTSApp2BtsMessageHandlingType cancelPairing(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelPairing& request);

   virtual BTSApp2BtsMessageHandlingType startSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_StartRemoteServiceSearch& request);

   virtual BTSApp2BtsMessageHandlingType cancelSearch(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const App2Bts_CancelRemoteServiceSearch& request);

   virtual void checkEndOfSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool finished, IN const bool firstConnectFailed = false, IN const BTSBDAddress& address = BTSBDAddress());

   virtual void getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage& message) const;

   virtual void getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage& message) const;

private:
   /**
    * Type definition for priority mode.
    * #ConnectionPriorityMode#ignore#
    */
   enum ConnectionPriorityMode
   {
      PRIO_MODE_DEFAULT = 0, /**< default mode (matches initial implementation => only sequential request execution) */
      PRIO_MODE_PARALLEL_PIM_MSG, /**< all requests parallel to PIM and MSG connect */
      // -------------------------------------------------------------------------
      PRIO_MODE_LAST /**< end marker, must be the last one */
   };

   IBasicControl* _controlIf; /**< basic control interface */
   IDeviceManager* _deviceManagerIf; /**< device manager interface */
   IProtocolManager* _protocolManagerIf; /**< protocol manager interface */
   IInquiry* _inquiryIf; /**< inquiry interface */
   IPairing* _pairingIf; /**< pairing interface */
   IServiceSearch* _serviceSearchIf; /**< service search interface */
   ConnectionPriorityMode _priorityMode; /**< priority mode */

   bool checkDisconnectDevicePriority(IN const App2Bts_DisconnectDevice& request) const;

   bool checkConnectProtocolPriority(IN const App2Bts_ConnectProtocol& request) const;

   bool checkDisconnectProtocolPriority(IN const App2Bts_DisconnectProtocol& request) const;

   bool checkStartDiscoveryPriority(IN const App2Bts_StartDiscovery& request) const;

   bool checkStopDiscoveryPriority(IN const App2Bts_StopDiscovery& request) const;

   bool checkStartPairingPriority(IN const App2Bts_StartPairing& request) const;

   bool checkStartRemoteServiceSearchPriority(IN const App2Bts_StartRemoteServiceSearch& request) const;

   bool checkPrioModeDefault(IN const BTSApp2BtsOpcode opCode) const;

   void handleAclConnectFailed(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSBDAddress& address);

   bool continueWithNextAclBasedAction(void) const;

   bool continueWithNextAclBasedActionPrioModeDefault(void) const;

   bool isDisconnectRequestInWaitingQueue(IN const BTSBDAddress& address) const;

   bool isConnectDisconnectRequestInWaitingQueue(IN const BTSBDAddress& address, IN const BTSProtocolId protocol, IN const BTSUuid& uuid, IN const BTSMasInstanceName& masInstance) const;

   bool isDiscoveryRequestInWaitingQueue(void) const;

   bool isPairingRequestInWaitingQueue(IN const BTSBDAddress& address) const;

   bool isServiceSearchRequestInWaitingQueue(IN const BTSBDAddress& address, IN const BTSSearchType type) const;

   bool isAnyAclBasedRequestInWorkingQueue(void) const;

   BTSApp2BtsMessageHandlingType checkRequestResult(IN const bool processFlag) const;

   bool isConnectionRelatedOpCode(IN const BTSApp2BtsOpcode opCode) const;

   void getSimilarOpCodesPrioModeDefault(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage& message) const;

   void getMatchingOpCodesPrioModeDefault(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage& message) const;
};

} //btstackif

#endif //_CONNECTION_PRIORITY_H_
