/**
 * @file BasicConfigurationItems.cpp
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Implementation of basic configuration items.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of basic configuration items.
 */

#include "BasicConfigurationItems.h"
#include "IConfiguration.h"
#include "IBasicConfigurationRequest.h"
#include "FwErrmemPrint.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BasicConfigurationItems.cpp.trc.h"
#endif
#endif

//#define ENABLE_TRACES
#undef ENABLE_TRACES

namespace btstackif {

BasicConfigurationItemLocalServices::BasicConfigurationItemLocalServices() :
_services(),
_dummy()
{
}

BasicConfigurationItemLocalServices::~BasicConfigurationItemLocalServices()
{
}

void BasicConfigurationItemLocalServices::reset(void)
{
   _services.setData(0);
}

uint64_t BasicConfigurationItemLocalServices::getReceivedIntegerData(IN const BTSConfigurationParameter configurationParameter) const
{
   (void)(configurationParameter);

   return _services.getData();
}

const ::std::string& BasicConfigurationItemLocalServices::getReceivedStringData(IN const BTSConfigurationParameter configurationParameter) const
{
   (void)(configurationParameter);

   FW_ERRMEM_ASSERT_ALWAYS();

   return _dummy;
}

void BasicConfigurationItemLocalServices::setReceivedData(IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data)
{
   (void)(configurationParameter);

#ifdef ENABLE_TRACES
   ETG_TRACE_USR3((" setReceivedData: stack value =0x%04x", (unsigned short int)data));
#endif

   _services.setData((unsigned short int)data);
}

void BasicConfigurationItemLocalServices::setReceivedData(IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data)
{
   (void)(configurationParameter);
   (void)(data);

   FW_ERRMEM_ASSERT_ALWAYS();
}

void BasicConfigurationItemLocalServices::setRequestedData(IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data)
{
   (void)(configurationParameter);
   (void)(data);

   FW_ERRMEM_ASSERT_ALWAYS();
}

void BasicConfigurationItemLocalServices::setRequestedData(IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data)
{
   (void)(configurationParameter);
   (void)(data);

   FW_ERRMEM_ASSERT_ALWAYS();
}

bool BasicConfigurationItemLocalServices::isEqual(IN const BTSConfigurationParameter configurationParameter, IN const IConfiguration& configurationIf) const
{
   (void)(configurationParameter);

#ifdef ENABLE_TRACES
   ETG_TRACE_USR3((" isEqual: stack value =0x%04x", _services.getData()));
   ETG_TRACE_USR3((" isEqual: config value=0x%04x", configurationIf.getConfiguration().supportedServices.getData()));
#endif

   return (_services.getData() == configurationIf.getConfiguration().supportedServices.getData());
}

void BasicConfigurationItemLocalServices::sendRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const IConfiguration& configurationIf, IN IBasicConfigurationRequest& requestIf)
{
   requestIf.setConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, configurationParameter, configurationIf.getConfiguration().supportedServices.getData());
}

void BasicConfigurationItemLocalServices::sendVirtualUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const BTSIpcCommonErrorCode errorCode, IN IBasicConfigurationRequest& requestIf)
{
   // currently only the error case is needed to be supported (=> update timeout)
   // if success case shall be supported then use stored data (member variable)
   const BTSSupportedServices services;
   requestIf.sendVirtualConfigurationParameterUpdate(bts2IpcMsgList, bts2AppMsgList, configurationParameter, services.getData(), errorCode);
}

//----------------------------------------------------------------------------------------------------------------------

BasicConfigurationItemPageTimeout::BasicConfigurationItemPageTimeout() :
_pageTimeout(0),
_requestedPageTimeout(0),
_dummy()
{
}

BasicConfigurationItemPageTimeout::~BasicConfigurationItemPageTimeout()
{
}

void BasicConfigurationItemPageTimeout::reset(void)
{
   _pageTimeout = 0;
   _requestedPageTimeout = 0;
}

uint64_t BasicConfigurationItemPageTimeout::getReceivedIntegerData(IN const BTSConfigurationParameter configurationParameter) const
{
   (void)(configurationParameter);

   return _pageTimeout;
}

const ::std::string& BasicConfigurationItemPageTimeout::getReceivedStringData(IN const BTSConfigurationParameter configurationParameter) const
{
   (void)(configurationParameter);

   FW_ERRMEM_ASSERT_ALWAYS();

   return _dummy;
}

void BasicConfigurationItemPageTimeout::setReceivedData(IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data)
{
   (void)(configurationParameter);

   _pageTimeout = (BTSTimeValue)data;
}

void BasicConfigurationItemPageTimeout::setReceivedData(IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data)
{
   (void)(configurationParameter);
   (void)(data);

   FW_ERRMEM_ASSERT_ALWAYS();
}

void BasicConfigurationItemPageTimeout::setRequestedData(IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data)
{
   (void)(configurationParameter);

   _requestedPageTimeout = (BTSTimeValue)data;
}

void BasicConfigurationItemPageTimeout::setRequestedData(IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data)
{
   (void)(configurationParameter);
   (void)(data);

   FW_ERRMEM_ASSERT_ALWAYS();
}

bool BasicConfigurationItemPageTimeout::isEqual(IN const BTSConfigurationParameter configurationParameter, IN const IConfiguration& configurationIf) const
{
   (void)(configurationParameter);
   (void)(configurationIf);

   return (_pageTimeout == _requestedPageTimeout);
}

void BasicConfigurationItemPageTimeout::sendRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const IConfiguration& configurationIf, IN IBasicConfigurationRequest& requestIf)
{
   (void)(configurationIf);

   requestIf.setConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, configurationParameter, _requestedPageTimeout);
}

void BasicConfigurationItemPageTimeout::sendVirtualUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const BTSIpcCommonErrorCode errorCode, IN IBasicConfigurationRequest& requestIf)
{
   // currently only the error case is needed to be supported (=> update timeout)
   // if success case shall be supported then use stored data (member variable)
   requestIf.sendVirtualConfigurationParameterUpdate(bts2IpcMsgList, bts2AppMsgList, configurationParameter, 0, errorCode);
}

} //btstackif
