/**
 * @file BasicConfiguration.cpp
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Implementation of generic basic configuration state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Source file for implementation of generic basic configuration state machine.
 */

#include "BasicConfiguration.h"
#include "IBasicConfigurationRequest.h"
#include "IBasicConfigurationItem.h"
#include "IBasicControl.h"
#include "ITimerPool.h"
// #include "Timer.h"
#include "IConfiguration.h"
#include "IBasicResultCallback.h"
#include "BasicConfigurationItems.h"
#include "FwAssert.h"
#include "FwFormattedDataPrint.h"
#include "App2Bts_MessageWrapper.h"
#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/BasicConfiguration.cpp.trc.h"
#endif
#endif

namespace btstackif {

BasicConfiguration::BasicConfiguration() :
_requestIfList(),
_controlIf(0),
_timerPoolIf(0),
_configurationIf(0),
_configurationMaster(0),
_globalConfigActive(false),
_timerConfiguration(),
_timeoutConfiguration(3000),
_configItemList(),
_testTriggerIgnoreUpdate(false)
{
}

BasicConfiguration::~BasicConfiguration()
{
   _controlIf = 0;
   _timerPoolIf = 0;
   _configurationIf = 0;
   _configurationMaster = 0;
   clearConfigItemList();
}

void BasicConfiguration::reset(void)
{
   StateMachine::reset();
   // keep _requestIfList
   // keep _controlIf
   // keep _timerPoolIf
   // keep _configurationIf
   // keep _configurationMaster
   _globalConfigActive = false;
   resetConfigRequestControlData();
   // stop all timer
   stopTimer(_timerConfiguration);
   // keep _configItemList
   _testTriggerIgnoreUpdate = false;

   for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
   {
      if(0 != it->second.requestIf)
      {
         it->second.requestIf->reset();
      }
   }
}

void BasicConfiguration::forceInitialState(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList)
{
   (void)(bts2AppMsgList);

   // reset control data
   reset();
}

void BasicConfiguration::setInstance(IN IBasicConfigurationRequest* instance, IN const BTSConfigurationParameter configurationParameter, IN const bool beforeBtOn, IN const bool configureDuringStartup /*= true*/)
{
   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _requestIfList[configurationParameter].requestIf = instance;
   _requestIfList[configurationParameter].configItem = createConfigItem(configurationParameter);
   _requestIfList[configurationParameter].beforeBtOn = beforeBtOn;
   _requestIfList[configurationParameter].configureDuringStartup = configureDuringStartup;

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].requestIf);

   _requestIfList[configurationParameter].requestIf->setCallback(this);
}

void BasicConfiguration::setControlIf(IN IBasicControl* control)
{
   _controlIf = control;

   FW_NORMAL_ASSERT(0 != _controlIf);

   for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
   {
      if(0 != it->second.requestIf)
      {
         it->second.requestIf->setControlIf(_controlIf);
      }
   }
}

void BasicConfiguration::setTimerPoolIf(IN ITimerPool* timerPool)
{
   _timerPoolIf = timerPool;

   FW_NORMAL_ASSERT(0 != _timerPoolIf);
}

void BasicConfiguration::setConfigurationIf(IN IConfiguration* configuration)
{
   _configurationIf = configuration;

   FW_NORMAL_ASSERT(0 != _configurationIf);
}

IConfigurationClient* BasicConfiguration::getConfigurationClientHandler(void)
{
   return this;
}

IStateMachine* BasicConfiguration::getSmEntryInterface(void)
{
   return this;
}

bool BasicConfiguration::isStartRuntimeConfigurationPossible(void) const
{
   if(true == _globalConfigActive)
   {
      return false;
   }

   if(true == isAnyConfigRequestOngoing())
   {
      // configuration request is ongoing => only 1 single configuration sequence at the same time is allowed
      // NOTE: if different behavior is needed then a queue mechanism shall be implemented
      return false;
   }

   return true;
}

void BasicConfiguration::setRuntimeConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback, IN const uint64_t data)
{
   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].configItem);

   // store requested data
   _requestIfList[configurationParameter].configItem->setRequestedData(configurationParameter, data);

   checkAndStartSingleConfigurationSequence(bts2IpcMsgList, bts2AppMsgList, configurationParameter, callback);
}

void BasicConfiguration::setRuntimeConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback, IN const ::std::string& data)
{
   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].configItem);

   // store requested data
   _requestIfList[configurationParameter].configItem->setRequestedData(configurationParameter, data);

   checkAndStartSingleConfigurationSequence(bts2IpcMsgList, bts2AppMsgList, configurationParameter, callback);
}

void BasicConfiguration::setTestTriggerIgnoreUpdate(IN const bool enable)
{
   _testTriggerIgnoreUpdate = enable;
}

void BasicConfiguration::updateConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data, IN const bool success)
{
   ::fw::FormattedOutputU64 printValue(data, true);
   ETG_TRACE_USR2((" updateConfigurationParameter: success=%d configurationParameter=%d data=%s", success, configurationParameter, printValue.c_str()));

   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   //===================================================================================================================
   // debug section start
   if(true == _testTriggerIgnoreUpdate)
   {
      // check if get request is ongoing
      if(true == _requestIfList[configurationParameter].configRequestOngoing)
      {
         _testTriggerIgnoreUpdate = false;

         return;
      }
   }
   // debug section end
   //===================================================================================================================

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].configItem);

   // store data if successful
   if(true == success)
   {
      _requestIfList[configurationParameter].configItem->setReceivedData(configurationParameter, data);
   }

   // handle update
   handleConfigurationParameterUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, configurationParameter, success);
}

void BasicConfiguration::updateConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data, IN const bool success)
{
   ETG_TRACE_USR2((" updateConfigurationParameter: success=%d configurationParameter=%d data=%s", success, configurationParameter, data.c_str()));

   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].configItem);

   // store data if successful
   if(true == success)
   {
      _requestIfList[configurationParameter].configItem->setReceivedData(configurationParameter, data);
   }

   // handle update
   handleConfigurationParameterUpdate(bts2IpcMsgList, bts2AppMsgList, messageItem, configurationParameter, success);
}

void BasicConfiguration::handleExtendedTimeout(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSTimerId timerId)
{
   (void)(messageItem);

   if(true == _timerConfiguration.compare(timerId))
   {
      // find ongoing configuration request
      for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
      {
         if(true == it->second.configRequestOngoing)
         {
            // configuration is done step by step => only 1 configuration request can be active at the same time
            const BTSConfigurationParameter configurationParameter(it->first);
            FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].requestIf);
            FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].configItem);

            _requestIfList[configurationParameter].configItem->sendVirtualUpdate(bts2IpcMsgList, bts2AppMsgList, configurationParameter, BTS_IPC_UPDATE_TIMEOUT, *_requestIfList[configurationParameter].requestIf);

            return;
         }
      }
   }
}

void BasicConfiguration::setConfigurationMasterIf(IN IConfigurationMaster* master)
{
   _configurationMaster = master;

   FW_NORMAL_ASSERT(0 != _configurationMaster);
}

void BasicConfiguration::startGlobalConfiguration(void)
{
   _globalConfigActive = true;
   resetConfigRequestOngoingData();
   // do not reset configRequestDone flags because this configuration shall be done only once
}

void BasicConfiguration::stopGlobalConfiguration(void)
{
   _globalConfigActive = false;
   resetConfigRequestOngoingData();
}

bool BasicConfiguration::isSingleConfigurationOngoing(void) const
{
   if(false == _globalConfigActive)
   {
      return false;
   }

   // check if call to configure a parameter is ongoing
   return isAnyConfigRequestOngoing();
}

bool BasicConfiguration::setSingleConfiguration(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const bool beforeBtOn, IN const bool errorOccurred)
{
   (void)(messageItem);

   if(false == _globalConfigActive)
   {
      return false;
   }

   // ETG_TRACE_USR3((" BasicConfiguration::setSingleConfiguration(): beforeBtOn=%d isAnyConfigRequestOngoing=%d errorOccurred=%d", beforeBtOn, isAnyConfigRequestOngoing(), errorOccurred));

   if(true == isAnyConfigRequestOngoing())
   {
      // configuration request is ongoing => wait for result
      return true;
   }

   if(true == errorOccurred)
   {
      // error occurred => do not continue
      return false;
   }

   // find pending configuration item
   for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
   {
      if((true == it->second.configureDuringStartup) && (beforeBtOn == it->second.beforeBtOn) && (false == it->second.configRequestDone))
      {
         const BTSConfigurationParameter configurationParameter(it->first);

         // first step: get current data
         if(0 == _requestIfList[configurationParameter].requestIf)
         {
            FW_NORMAL_ASSERT_ALWAYS();
            // mark as finished
            it->second.configRequestDone = true;
         }
         else
         {
            startSingleConfigurationSequence(bts2IpcMsgList, bts2AppMsgList, configurationParameter);

            return true;
         }
      }
   }

   return false;
}

void BasicConfiguration::startSingleConfigurationSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter)
{
   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].requestIf);

   _requestIfList[configurationParameter].requestIf->getConfigurationParameter(bts2IpcMsgList, bts2AppMsgList, configurationParameter);
   // do not start timer for get request
   // mark as ongoing
   _requestIfList[configurationParameter].getRequestOngoing = true;
}

void BasicConfiguration::checkAndStartSingleConfigurationSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN IBasicResultCallback* callback)
{
   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(callback);

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].requestIf);

   if(true == _globalConfigActive)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   if(true == isAnyConfigRequestOngoing())
   {
      // configuration request is ongoing => only 1 single configuration sequence at the same time is allowed
      // NOTE: if different behavior is needed then a queue mechanism shall be implemented
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // note: requested data was stored before calling this function

   // remember callback
   _requestIfList[configurationParameter].callback = callback;

   // start configuration sequence
   startSingleConfigurationSequence(bts2IpcMsgList, bts2AppMsgList, configurationParameter);
}

void BasicConfiguration::handleConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSConfigurationParameter configurationParameter, IN const bool success)
{
   FW_IF_NULL_PTR_RETURN(_configurationIf);
   FW_IF_NULL_PTR_RETURN(_configurationMaster);

   if(BTS_CONFIG_PARAM_LAST <= configurationParameter)
   {
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].requestIf);
   FW_IF_NULL_PTR_RETURN(_requestIfList[configurationParameter].configItem);

   // received data was stored before calling this function

   // check if get request is ongoing
   if(true == _requestIfList[configurationParameter].getRequestOngoing)
   {
      // mark as not ongoing
      _requestIfList[configurationParameter].getRequestOngoing = false;

      // check if update was successful
      if(true == success)
      {
         // compare requested and received configuration data
         if(true == _requestIfList[configurationParameter].configItem->isEqual(configurationParameter, *_configurationIf))
         {
            // requested configuration is already set => handle as successful configuration
            ETG_TRACE_USR2((" handleConfigurationParameterUpdate: configurationParameter=%d: requested and stack configuration matches", configurationParameter));
            _requestIfList[configurationParameter].configRequestOngoing = true;
         }
         else
         {
            // send configuration request
            _requestIfList[configurationParameter].configItem->sendRequest(bts2IpcMsgList, bts2AppMsgList, configurationParameter, *_configurationIf, *_requestIfList[configurationParameter].requestIf);
            // start timer
            startTimer(_timerConfiguration, _timeoutConfiguration);
            // mark as ongoing
            _requestIfList[configurationParameter].configRequestOngoing = true;
            // return now
            return;
         }
      }
      else
      {
         // failed => handle as successful configuration
         ETG_TRACE_USR2((" handleConfigurationParameterUpdate: configurationParameter=%d: get stack configuration failed", configurationParameter));
         ETG_TRACE_ERRMEM((" #CONN: BtStackIf: configurationParameter=%d: get stack configuration failed", configurationParameter));
         _requestIfList[configurationParameter].configRequestOngoing = true;
      }
   }

   // check if configuration request is ongoing
   if(true == _requestIfList[configurationParameter].configRequestOngoing)
   {
      // either global configuration phase or single configuration during runtime

      // stop timer
      stopTimer(_timerConfiguration);
      // mark as not ongoing
      _requestIfList[configurationParameter].configRequestOngoing = false;
      // mark as finished (global configuration phase always happens before single configuration during runtime => therefore next assignment can be done always)
      _requestIfList[configurationParameter].configRequestDone = true;
      // ignore result because there is no retry

      if(true == _globalConfigActive)
      {
         FW_NORMAL_ASSERT(0 == _requestIfList[configurationParameter].callback);

#ifdef ENABLE_NEW_CONFIGURATION_HANDLING
         // inform configuration master to execute next step
         (void)_configurationMaster->doNextConfigurationStep(bts2IpcMsgList, bts2AppMsgList, messageItem);
#endif
      }
      else
      {
         FW_NORMAL_ASSERT(0 != _requestIfList[configurationParameter].callback);

         const BTSRequestResult result((true == success) ? BTS_REQ_SUCCESS : BTS_REQ_FAILED);

         // inform initiator
         informInitiator(bts2IpcMsgList, bts2AppMsgList, messageItem, result, _requestIfList[configurationParameter].callback);

         // reset callback
         _requestIfList[configurationParameter].callback = 0;
      }
   }
}

void BasicConfiguration::resetConfigRequestControlData(void)
{
   for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
   {
      it->second.configRequestOngoing = false;
      it->second.getRequestOngoing = false;
      it->second.configRequestDone = false;
      if(0 != it->second.configItem)
      {
         it->second.configItem->reset();
      }
      it->second.callback = 0;
   }
}

void BasicConfiguration::resetConfigRequestOngoingData(void)
{
   for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
   {
      it->second.configRequestOngoing = false;
      it->second.getRequestOngoing = false;
   }
}

bool BasicConfiguration::isAnyConfigRequestOngoing(void) const
{
   for(::std::map< BTSConfigurationParameter, BasicConfigurationData >::const_iterator it = _requestIfList.begin(); it != _requestIfList.end(); ++it)
   {
      if((true == it->second.configRequestOngoing) || (true == it->second.getRequestOngoing))
      {
         return true;
      }
   }

   return false;
}

IBasicConfigurationItem* BasicConfiguration::createConfigItem(IN const BTSConfigurationParameter configurationParameter)
{
   IBasicConfigurationItem* item(0);

   ::std::map< BTSConfigurationParameter, IBasicConfigurationItem* >::iterator it = _configItemList.find(configurationParameter);

   if(_configItemList.end() == it)
   {
      // create new entry
      switch(configurationParameter)
      {
         case BTS_CONFIG_PARAM_LOCAL_SERVICES:
            item = new BasicConfigurationItemLocalServices();
            _configItemList[configurationParameter] = item;
            break;
         case BTS_CONFIG_PARAM_PAGE_TIMEOUT:
            item = new BasicConfigurationItemPageTimeout();
            _configItemList[configurationParameter] = item;
            break;
         case BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND:
            item = new BasicConfigurationItemPageTimeout();
            _configItemList[configurationParameter] = item;
            break;
         case BTS_CONFIG_PARAM_LAST:
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
   else
   {
      // continue with existing entry
      item = it->second;
   }

   FW_NORMAL_ASSERT(0 != item);

   return item;
}

void BasicConfiguration::clearConfigItemList(void)
{
   for(::std::map< BTSConfigurationParameter, IBasicConfigurationItem* >::iterator it = _configItemList.begin(); it != _configItemList.end(); ++it)
   {
      if(0 != it->second)
      {
         delete it->second;
      }
   }

   _configItemList.clear();
}

void BasicConfiguration::startTimer(IN ExtendedTimerEntry& timer, IN const BTSTimeValue timeout)
{
   ETG_TRACE_USR3((" startTimer: timeout=%u", timeout));

   FW_IF_NULL_PTR_RETURN(_timerPoolIf);
   FW_IF_NULL_PTR_RETURN(_controlIf);

   timer.setTimerPool(_timerPoolIf);
   timer.start(_controlIf, this, timeout);
}

void BasicConfiguration::stopTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" stopTimer"));

   timer.stop();

   // do not release timer
}

void BasicConfiguration::releaseTimer(IN ExtendedTimerEntry& timer) const
{
   ETG_TRACE_USR3((" releaseTimer"));

   timer.release();
}

bool BasicConfiguration::isTimerActive(IN const ExtendedTimerEntry& timer) const
{
   return timer.isActive();
}

void BasicConfiguration::informInitiator(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN const BTSRequestResult result, IN IBasicResultCallback* callback) const
{
   FW_IF_NULL_PTR_RETURN(callback);

   callback->handleBasicResult(bts2IpcMsgList, bts2AppMsgList, messageItem, result);
}

} //btstackif
