/**
 * @file IConfiguration.h
 *
 * @par SW-Component
 * State machine for configuration
 *
 * @brief Interface definition for generic configuration state machine.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the generic interface for configuration state machine.
 */

#ifndef _I_CONFIGURATION_H_
#define _I_CONFIGURATION_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IConfigurationMaster;
class IStateMachine;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IConfiguration
{
public:
   /**
    * Destructor.
    */
   virtual ~IConfiguration() {}

   /**
    * Set stack configuration.
    *
    * @param[in] configuration: configuration
    */
   virtual void setConfiguration(IN const BTSLocalStackConfiguration& configuration) = 0;

   /**
    * Get stack configuration.
    *
    * @return = stack configuration
    */
   virtual const BTSLocalStackConfiguration& getConfiguration(void) const = 0;

   /**
    * Get stack configuration.
    *
    * @return = stack configuration
    */
   virtual BTSLocalStackConfiguration& getConfiguration(void) = 0;

   /**
    * Get configuration master handler.
    *
    * @return = configuration master handler
    */
   virtual IConfigurationMaster* getConfigurationMasterHandler(void) = 0;

   /**
    * Get SM entry interface.
    *
    * @return = SM entry interface
    */
   virtual IStateMachine* getSmEntryInterface(void) = 0;

   /**
    * Set ongoing flag.
    *
    * @param[in] enable: flag
    */
   virtual void setOngoing(IN const bool enable) = 0;

   /**
    * Get ongoing flag.
    *
    * @return = ongoing flag
    */
   virtual bool getOngoing(void) const = 0;

   /**
    * Start configuration sequence.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void startSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Continue configuration sequence.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    */
   virtual void continueSequence(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem) = 0;

   /**
    * Set error flag.
    *
    * @param[in] configurationError: flag
    */
   virtual void setError(IN const bool configurationError) = 0;
};

} //btstackif

#endif //_I_CONFIGURATION_H_
