/**
 * @file IBasicConfigurationRequest.h
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Request interface definition for basic configuration state machine.
 *
 * @copyright (C) 2017 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the request interface to be implemented by specific basic configuration state machine.
 */

#ifndef _I_BASIC_CONFIGURATION_REQUEST_H_
#define _I_BASIC_CONFIGURATION_REQUEST_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IBasicConfigurationCallback;
class IBasicControl;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IBasicConfigurationRequest
{
public:
   /**
    * Destructor.
    */
   virtual ~IBasicConfigurationRequest() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set callback instance (implementation).
    *
    * @param[in] callback: callback instance (implementation)
    */
   virtual void setCallback(IN IBasicConfigurationCallback* callback) = 0;

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Get configuration parameter.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    */
   virtual void getConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter) = 0;

   /**
    * Set configuration parameter (integer data).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    */
   virtual void setConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data) = 0;

   /**
    * Set configuration parameter (string data).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    */
   virtual void setConfigurationParameter(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data) = 0;

   /**
    * Send virtual update message for configuration parameter (integer data, loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data, IN const BTSIpcCommonErrorCode errorCode) = 0;

   /**
    * Send virtual update message for configuration parameter (string data, loopback).
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    * @param[in] errorCode: error code
    */
   virtual void sendVirtualConfigurationParameterUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data, IN const BTSIpcCommonErrorCode errorCode) = 0;
};

} //btstackif

#endif //_I_BASIC_CONFIGURATION_REQUEST_H_
