/**
 * @file IBasicConfigurationItem.h
 *
 * @par SW-Component
 * State machine for basic configuration
 *
 * @brief Interface definition for basic configuration item.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for basic configuration item.
 */

#ifndef _I_BASIC_CONFIGURATION_ITEM_H_
#define _I_BASIC_CONFIGURATION_ITEM_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IConfiguration;
class IBasicConfigurationRequest;
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;

/**
 * Interface definition.
 */
class IBasicConfigurationItem
{
public:
   /**
    * Destructor.
    */
   virtual ~IBasicConfigurationItem() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Get received data of configuration parameter.
    *
    * @param[in] configurationParameter: configuration parameter id
    */
   virtual uint64_t getReceivedIntegerData(IN const BTSConfigurationParameter configurationParameter) const = 0;

   /**
    * Get received data of configuration parameter.
    *
    * @param[in] configurationParameter: configuration parameter id
    */
   virtual const ::std::string& getReceivedStringData(IN const BTSConfigurationParameter configurationParameter) const = 0;

   /**
    * Set received data of configuration parameter.
    *
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    */
   virtual void setReceivedData(IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data) = 0;

   /**
    * Set received data of configuration parameter.
    *
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    */
   virtual void setReceivedData(IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data) = 0;

   /**
    * Set requested data of configuration parameter.
    *
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    */
   virtual void setRequestedData(IN const BTSConfigurationParameter configurationParameter, IN const uint64_t data) = 0;

   /**
    * Set requested data of configuration parameter.
    *
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] data: configuration data
    */
   virtual void setRequestedData(IN const BTSConfigurationParameter configurationParameter, IN const ::std::string& data) = 0;

   /**
    * Checks if requested and received configuration data is equal.
    *
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] configurationIf: configuration interface
    *
    * @return = true: configuration data is equal,
    * @return = false: configuration data is not equal
    */
   virtual bool isEqual(IN const BTSConfigurationParameter configurationParameter, IN const IConfiguration& configurationIf) const = 0;

   /**
    * Send configuration request.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] configurationIf: configuration interface
    * @param[in] requestIf: request interface
    */
   virtual void sendRequest(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const IConfiguration& configurationIf, IN IBasicConfigurationRequest& requestIf) = 0;

   /**
    * Send virtual update message.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] configurationParameter: configuration parameter id
    * @param[in] errorCode: error code
    * @param[in] requestIf: request interface
    */
   virtual void sendVirtualUpdate(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN const BTSConfigurationParameter configurationParameter, IN const BTSIpcCommonErrorCode errorCode, IN IBasicConfigurationRequest& requestIf) = 0;
};

} //btstackif

#endif //_I_BASIC_CONFIGURATION_ITEM_H_
