/**
 * @file MainControl.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief Main Control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of main control functionality.
 */

#include <cstring>

#include "MainControl.h"
#include "App2Bts_BaseMessage.h"
#include "Bts2App_BaseMessage.h"
#include "Bts2Ipc_BaseMessage.h"
#include "Ipc2Bts_BaseMessage.h"
#include "App2Bts_MessageWrapper.h"
#include "InstanceFactory.h"
#include "IBasicControl.h"
#include "IDbusBase.h"
#include "IGenericStackIf.h"
#include "BtsUtils.h"
#include "TraceBase.h"
#include "TraceClasses.h"
#include "FwSingleThread.h"
#include "FwFormattedDataPrint.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/MainControl.cpp.trc.h"
#endif
#endif

namespace btstackif {

MainControl* MainControl::_exemplar = 0;

MainControl::MainControl() :
_app2BtsInputQueue(),
_app2BtsHighPrioInputQueue(),
_bts2AppOutputQueue(),
_bts2IpcOutputQueue(),
_ipc2BtsInputQueue(),
_ipc2BtsHighPrioInputQueue(),
_eventQueue(),
_ttyMonitor(),
_timerSource(),
_timerMaster(),
_timer1(),
_timer2(),
_timer3(),
_terminateWorkerThread(false),
_workerThreadTerminated(true),
_singleThread(0),
_genericStackIf(0),
_controlHandlerList(),
_dbusIfList(),
_resetDbusIfHandlerList()
{
   _exemplar = this;

   _app2BtsInputQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _app2BtsHighPrioInputQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _bts2AppOutputQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _bts2IpcOutputQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _ipc2BtsInputQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _ipc2BtsHighPrioInputQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);

   _timerMaster.setDefaultContext();
   _timerMaster.requestCurrentTime();
}

MainControl::MainControl(const MainControl& ref) :
_app2BtsInputQueue(),
_app2BtsHighPrioInputQueue(),
_bts2AppOutputQueue(),
_bts2IpcOutputQueue(),
_ipc2BtsInputQueue(),
_ipc2BtsHighPrioInputQueue(),
_eventQueue(),
_ttyMonitor(),
_timerSource(),
_timerMaster(),
_timer1(),
_timer2(),
_timer3(),
_terminateWorkerThread(false),
_workerThreadTerminated(true),
_singleThread(0),
_genericStackIf(0),
_controlHandlerList(),
_dbusIfList(),
_resetDbusIfHandlerList()
{
   // ignore given parameter
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

MainControl& MainControl::operator=(const MainControl& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   // ignore given parameter

   return *this;
}

MainControl::~MainControl()
{
   _app2BtsInputQueue.empty();
   _app2BtsHighPrioInputQueue.empty();
   _bts2AppOutputQueue.empty();
   _bts2IpcOutputQueue.empty();
   _ipc2BtsInputQueue.empty();
   _ipc2BtsHighPrioInputQueue.empty();
   if(_singleThread)
   {
      delete _singleThread;
   }
   _singleThread = 0;
   _genericStackIf = 0;

   _exemplar = 0;
}

void MainControl::setControlAndDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN IBasicControl* controlHandler, IN IDbusBase* dbusIf)
{
   FW_IF_NULL_PTR_RETURN(controlHandler);
   FW_IF_NULL_PTR_RETURN(dbusIf);

   ::fw::FormattedOutputPtr controlPtr((uintptr_t)controlHandler);
   ::fw::FormattedOutputPtr dbusPtr((uintptr_t)dbusIf);
   ETG_TRACE_USR1((" setControlAndDbusIf(): component=%d stackInterface=%d subComponent=%d: controlHandler=%20s dbusIf=%s", component, stackInterface, subComponent, controlPtr.c_str(), dbusPtr.c_str()));

   BTSComponentEntry controlEntry;
   controlEntry = getControlComponentKey(controlEntry, component, stackInterface, subComponent);
   if(_controlHandlerList.end() == _controlHandlerList.find(controlEntry))
   {
      _controlHandlerList[controlEntry] = controlHandler;
      controlHandler->setQueueWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   BTSComponentEntry dbusEntry;
   dbusEntry = getDbusComponentKey(dbusEntry, component, stackInterface, subComponent);
   if(_dbusIfList.end() == _dbusIfList.find(dbusEntry))
   {
      _dbusIfList[dbusEntry] = dbusIf;
   }
   else
   {
      switch(subComponent)
      {
         case BTS_FB_CONFIG:
         case BTS_FB_CONNECTION:
         case BTS_FB_TELEPHONY:
         case BTS_FB_PHONEBOOK:
         case BTS_FB_MESSAGING:
         case BTS_FB_MEDIAPLAYER:
            // can happen, it is the same DBUS interface
            break;
         case BTS_FB_WBL:
         case BTS_FB_ECNR:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
         case BTS_FB_NONE:
         case BTS_FB_LAST:
         default:
            FW_NORMAL_ASSERT_ALWAYS();
            break;
      }
   }
}

void MainControl::resetControlAndDbusIf(void)
{
   ETG_TRACE_USR1((" resetControlAndDbusIf()"));

   // destruction is done outside
   _controlHandlerList.clear();
   _dbusIfList.clear();
}

void MainControl::setGenericStackIf(IN IGenericStackIf* genericStackIf)
{
   FW_IF_NULL_PTR_RETURN(genericStackIf);

   _genericStackIf = genericStackIf;
}

void MainControl::resetGenericStackIf(void)
{
   _genericStackIf = 0;
}

void MainControl::pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   if(0 != ptrMessage)
   {
      const BTSFunctionBlock functionBlock(ptrMessage->getFunctionBlock()); // sub component

      if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
      {
         if(true == highPrio)
         {
            _app2BtsHighPrioInputQueue.push(ptrMessage);
         }
         else
         {
            _app2BtsInputQueue.push(ptrMessage);
         }

         _eventQueue.post(EV_APP2BTS_INPUT);
      }
      else
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();

         delete ptrMessage;
      }
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void MainControl::pushBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage)
{
   if(0 != ptrMessage)
   {
      const BTSFunctionBlock functionBlock(ptrMessage->getFunctionBlock()); // sub component

      if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
      {
         _bts2AppOutputQueue.push(ptrMessage, false); // single worker thread
      }
      else
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();

         delete ptrMessage;
      }
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void MainControl::pushBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(0 != ptrMessage)
   {
      // no check of function block because this depends on used supplier Bluetooth stack
      _bts2IpcOutputQueue.push(ptrMessage, false); // single worker thread
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void MainControl::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   if(0 != ptrMessage)
   {
      // no check of function block because this depends on used supplier Bluetooth stack
      if(true == highPrio)
      {
         _ipc2BtsHighPrioInputQueue.push(ptrMessage);
      }
      else
      {
         _ipc2BtsInputQueue.push(ptrMessage);
      }

      _eventQueue.post(EV_IPC2BTS_INPUT);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void MainControl::stop(void)
{
   _ttyMonitor.stop();
   _timerSource.stop();

   if(true != _workerThreadTerminated)
   {
      _terminateWorkerThread = true;

      if(_singleThread)
      {
         _singleThread->stop();
      }
   }
}

void MainControl::run(void)
{
   if(0 == _singleThread)
   {
      _singleThread = new ::fw::SingleThread();
   }

   if(_singleThread)
   {
      _terminateWorkerThread = false;
      _workerThreadTerminated = false;
      const ::std::string name("BTS_MAIN_CTRL");
      _singleThread->start(this, name, 0);
   }

   _ttyMonitor.start();
   _timerSource.start(this);
}

void MainControl::triggerRequestCurrentTime(void)
{
   // set current time
   _timerMaster.requestCurrentTime();
}

void MainControl::addTTYMonitorHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter)
{
   _ttyMonitor.addHandler(handler, filter);
}

void MainControl::printQueueStatistics(void)
{
   // _app2BtsInputQueue
   _app2BtsInputQueue.lockAccess();
   ETG_TRACE_USR1((" ***MainControl::_app2BtsInputQueue.getSize()=%u", _app2BtsInputQueue.getSize()));
   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsInputQueue.getBegin(); it != _app2BtsInputQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***MainControl::_app2BtsInputQueue[]: App2Bts 0x%04X", (*it)->getTraceOpCode()));
   }
   _app2BtsInputQueue.unlockAccess();

   // _app2BtsHighPrioInputQueue
   _app2BtsHighPrioInputQueue.lockAccess();
   ETG_TRACE_USR1((" ***MainControl::_app2BtsHighPrioInputQueue.getSize()=%u", _app2BtsHighPrioInputQueue.getSize()));
   for(MessageQueue<App2Bts_BaseMessage>::It it = _app2BtsHighPrioInputQueue.getBegin(); it != _app2BtsHighPrioInputQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***MainControl::_app2BtsHighPrioInputQueue[]: App2Bts 0x%04X", (*it)->getTraceOpCode()));
   }
   _app2BtsHighPrioInputQueue.unlockAccess();

   // _bts2AppOutputQueue
   _bts2AppOutputQueue.lockAccess();
   ETG_TRACE_USR1((" ***MainControl::_bts2AppOutputQueue.getSize()=%u", _bts2AppOutputQueue.getSize()));
   for(MessageQueue<Bts2App_BaseMessage>::It it = _bts2AppOutputQueue.getBegin(); it != _bts2AppOutputQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***MainControl::_bts2AppOutputQueue[]: Bts2App 0x%04X", (*it)->getTraceOpCode()));
   }
   _bts2AppOutputQueue.unlockAccess();

   // _bts2IpcOutputQueue
   _bts2IpcOutputQueue.lockAccess();
   ETG_TRACE_USR1((" ***MainControl::_bts2IpcOutputQueue.getSize()=%u", _bts2IpcOutputQueue.getSize()));
   for(MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcOutputQueue.getBegin(); it != _bts2IpcOutputQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***MainControl::_bts2IpcOutputQueue[]: Bts2Ipc 0x%04X", (*it)->getTraceOpCode()));
   }
   _bts2IpcOutputQueue.unlockAccess();

   // _ipc2BtsInputQueue
   _ipc2BtsInputQueue.lockAccess();
   ETG_TRACE_USR1((" ***MainControl::_ipc2BtsInputQueue.getSize()=%u", _ipc2BtsInputQueue.getSize()));
   for(MessageQueue<Ipc2Bts_BaseMessage>::It it = _ipc2BtsInputQueue.getBegin(); it != _ipc2BtsInputQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***MainControl::_ipc2BtsInputQueue[]: Ipc2Bts 0x%04X", (*it)->getTraceOpCode()));
   }
   _ipc2BtsInputQueue.unlockAccess();

   // _ipc2BtsHighPrioInputQueue
   _ipc2BtsHighPrioInputQueue.lockAccess();
   ETG_TRACE_USR1((" ***MainControl::_ipc2BtsHighPrioInputQueue.getSize()=%u", _ipc2BtsHighPrioInputQueue.getSize()));
   for(MessageQueue<Ipc2Bts_BaseMessage>::It it = _ipc2BtsHighPrioInputQueue.getBegin(); it != _ipc2BtsHighPrioInputQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***MainControl::_ipc2BtsHighPrioInputQueue[]: Ipc2Bts 0x%04X", (*it)->getTraceOpCode()));
   }
   _ipc2BtsHighPrioInputQueue.unlockAccess();
}

void MainControl::setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   if(0 == testCommand)
   {
      return;
   }

   if(0 == strcmp(BTS_MAIN_CONTROL_START_TIMER_TEST, testCommand))
   {
      if(1 == testData)
      {
         _eventQueue.post(EV_TIMER_TEST_START);
      }
      else if(0 == testData)
      {
         _eventQueue.post(EV_TIMER_TEST_STOP);
      }
   }
   else if(0 == strcmp(BTS_MAIN_CONTROL_ENABLE_TRACE_4_CURRENT_TIME, testCommand))
   {
      if(1 == testData)
      {
         TimerBase::setTrace4CurrentTime(true);
      }
      else if(0 == testData)
      {
         TimerBase::setTrace4CurrentTime(false);
      }
   }
   else if(0 == strcmp(BTS_MAIN_CONTROL_ENABLE_TRACE_4_TIMER_STATISTIC, testCommand))
   {
      if(1 == testData)
      {
         TimerBase::setTrace4TimerStatistic(true);
      }
      else if(0 == testData)
      {
         TimerBase::setTrace4TimerStatistic(false);
      }
   }
   else if(0 == strcmp(BTS_MAIN_CONTROL_ENABLE_TRACE_4_TIMOUT_CHECK, testCommand))
   {
      if(1 == testData)
      {
         TimerBase::setTrace4TimoutCheck(true);
      }
      else if(0 == testData)
      {
         TimerBase::setTrace4TimoutCheck(false);
      }
   }
}

void MainControl::triggerResetOfDbusIfHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   (void)(component);

   // create and post event
   IDbusBase* dbusIf(getDbusIf(BTS_FB_NONE, stackInterface, subComponent));
   if(0 != dbusIf)
   {
      _resetDbusIfHandlerList.push_back(dbusIf); // works because single thread
      _eventQueue.post(EV_RESET_DBUS_IF);
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void MainControl::threadFunction(void* arguments)
{
   mainThread(arguments);
}

void MainControl::setTerminate(void* arguments)
{
   (void)(arguments);

   _terminateWorkerThread = true;
   _eventQueue.post(EV_TERMINATE);
}

void MainControl::setTimerTick(void)
{
   _eventQueue.post(EV_TIMER_TICK);
}

void MainControl::mainThread(void* arguments)
{
   (void)arguments;

   ThreadInfo threadInfo(this);
   ETG_TRACE_USR1((" Do(): %s --- thread start", threadInfo.getInfo()));

   BTSEventMask eventMask;

   while(false == _terminateWorkerThread)
   {
      eventMask = 0;

      if(BTS_OK == _eventQueue.wait(eventMask)) // event mask is only set in success case
      {
         // ETG_TRACE_USR4((" Do(): eventMask=0x%08X (start)", eventMask));

         _timerMaster.requestCurrentTime();

         if(EV_TERMINATE & eventMask)
         {
            eventMask &= ~EV_TERMINATE;

            _terminateWorkerThread = true;
            continue;
         }

         if(EV_APP2BTS_INPUT & eventMask)
         {
            eventMask &= ~EV_APP2BTS_INPUT;

            if(true == handleApp2BtsMessage())
            {
               _eventQueue.post(EV_APP2BTS_INPUT);
            }
         }

         if(EV_IPC2BTS_INPUT & eventMask)
         {
            eventMask &= ~EV_IPC2BTS_INPUT;

            if(true == handleIpc2BtsMessage())
            {
               _eventQueue.post(EV_IPC2BTS_INPUT);
            }
         }

         if(EV_RESET_DBUS_IF & eventMask)
         {
            eventMask &= ~EV_RESET_DBUS_IF;

            handleResetOfDbusIfHandler();
         }

         // NOTE: add further event handling here

         // do timer handling at the end of event loop
         if(EV_TIMER_TICK & eventMask)
         {
            eventMask &= ~EV_TIMER_TICK;

            _timerMaster.checkElapsedTimer();

            // check Bts2AppOutputQueue
            handleBts2AppMessage();

            // check Bts2IpcOutputQueue
            handleBts2IpcMessage();

            handleTimerTickForDbusIf();
         }

         // check test functionality
         if(EV_TIMER_TEST_START & eventMask)
         {
            eventMask &= ~EV_TIMER_TEST_START;

            _timer1.start(250, &timer1FuncWrapper);
            _timer2.start(150, &timer2FuncWrapper);
            _timer3.start(350, &timer3FuncWrapper);
            _timer1.start(250, &timer1FuncWrapper, true);
            _timer2.stop();
            _timer2.start();
         }

         if(EV_TIMER_TEST_STOP & eventMask)
         {
            eventMask &= ~EV_TIMER_TEST_STOP;

            _timer1.stop();
            _timer2.stop();
            _timer3.stop();
         }

         // ETG_TRACE_USR4((" Do(): eventMask=0x%08X (end)", eventMask));
      }
   }

   ETG_TRACE_USR1((" Do(): %s --- thread end", threadInfo.getInfo()));

   _workerThreadTerminated = true;
}

bool MainControl::handleApp2BtsMessage(void)
{
   bool moreMessages = false;
   int counter = 0;
   App2Bts_BaseMessage* ptrApp2BtsMessage;
   BTSFunctionBlock functionBlock;

   while(counter < MAX_NUMBER_APP2BTS_MGS_PROCESSING)
   {
      ptrApp2BtsMessage = _app2BtsHighPrioInputQueue.pop();
      if(0 == ptrApp2BtsMessage)
      {
         ptrApp2BtsMessage = _app2BtsInputQueue.pop();
      }

      if(0 != ptrApp2BtsMessage)
      {
         functionBlock = ptrApp2BtsMessage->getFunctionBlock();

         if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
         {
            IBasicControl* control(getControlHandler(BTS_FB_NONE, ptrApp2BtsMessage->getStackInterface(), functionBlock));
            if(0 != control)
            {
               control->pushApp2BtsMessage(ptrApp2BtsMessage);

               // do not delete message
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
               delete ptrApp2BtsMessage;
            }
         }
         else
         {
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
            delete ptrApp2BtsMessage;
         }

         // check Bts2AppOutputQueue
         handleBts2AppMessage();

         // check Bts2IpcOutputQueue
         handleBts2IpcMessage();

         counter++;

         if(counter >= MAX_NUMBER_APP2BTS_MGS_PROCESSING)
         {
            moreMessages = true;
         }
      }
      else
      {
         counter = MAX_NUMBER_APP2BTS_MGS_PROCESSING;
      }
   }

   return moreMessages;
}

void MainControl::handleBts2AppMessage(void)
{
   Bts2App_BaseMessage* ptrBts2AppMessage;

   while(0 != (ptrBts2AppMessage = _bts2AppOutputQueue.pop(false))) // single worker thread
   {
      // do trace
      ptrBts2AppMessage->doOutputTrace();

      // concept: any registered callback instance is only destroyed after communication has stopped (NOTE: not the best solution!!!)
      BtStackIfCallback* callback = ptrBts2AppMessage->getCallback();
      if(callback)
      {
         // single receiver: result message
         ptrBts2AppMessage->triggerCallback(*callback);
      }
      else
      {
         // multiple receiver: status message
         if(_genericStackIf)
         {
            _genericStackIf->LockAccess();

            // const BTSFunctionBlock subComponent(extractSubComponentFromBts2AppOpCode(ptrBts2AppMessage->getOpCode()));
            const BTSFunctionBlock subComponent((ptrBts2AppMessage->getFunctionBlock()));
            if((true == isValidInterfaceType(ptrBts2AppMessage->getStackInterface())) && (true == isValidComponentType(subComponent)))
            {
               bool sent(false);
               ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >& userList = _genericStackIf->getUserList();

               for(::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >::iterator it = userList.begin(); it != userList.end(); ++it)
               {
                  if(it->first)
                  {
                     // check stack interface and sub component to report to correct callback instance
                     if((ptrBts2AppMessage->getStackInterface() == it->second.stackInterface) && (subComponent == it->second.subComponent))
                     {
                        if(it->second.callback)
                        {
                           sent = true;
                           ptrBts2AppMessage->triggerCallback(*(it->second.callback));
                        }
                        else
                        {
                           // can happen if callback is not registered until now
                        }

                        break; // works because max 1 user for same sub component per stack interface
                     }
                  }
                  else
                  {
                     FW_NORMAL_ASSERT_ALWAYS();
                  }
               }

               FW_NORMAL_ASSERT(true == sent);
            }
            else
            {
               FW_NORMAL_ASSERT(true == isValidInterfaceType(ptrBts2AppMessage->getStackInterface()));
               FW_NORMAL_ASSERT(true == isValidComponentType(subComponent));
            }

            _genericStackIf->UnlockAccess();
         }
      }

      delete ptrBts2AppMessage;
   }
}

void MainControl::handleBts2IpcMessage(void)
{
   Bts2Ipc_BaseMessage* ptrBts2IpcMessage;

   while(0 != (ptrBts2IpcMessage = _bts2IpcOutputQueue.pop(false))) // single worker thread
   {
      // forward to IPC handler
      IDbusBase* dbusIf(getDbusIf(BTS_FB_NONE, ptrBts2IpcMessage->getStackInterface(), (BTSFunctionBlock)ptrBts2IpcMessage->getFunctionBlock())); // getFunctionBlock() because this indicates the destination DBUS interface
      if(0 != dbusIf)
      {
         dbusIf->sendBts2IpcMessage(ptrBts2IpcMessage);

         // do not delete message
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
         delete ptrBts2IpcMessage;
      }
   }
}

bool MainControl::handleIpc2BtsMessage(void)
{
   bool moreMessages = false;
   int counter = 0;
   Ipc2Bts_BaseMessage* ptrIpc2BtsMessage;
   BTSFunctionBlock functionBlock;

   while(counter < MAX_NUMBER_IPC2BTS_MGS_PROCESSING)
   {
      ptrIpc2BtsMessage = _ipc2BtsHighPrioInputQueue.pop();
      if(0 == ptrIpc2BtsMessage)
      {
         ptrIpc2BtsMessage = _ipc2BtsInputQueue.pop();
      }

      if(0 != ptrIpc2BtsMessage)
      {
         // do trace
         ptrIpc2BtsMessage->doInputTrace();

         functionBlock = ptrIpc2BtsMessage->getBtsDestinationFunctionBlock();

         if((BTS_FB_NONE < functionBlock) && (functionBlock < BTS_FB_LAST))
         {
            IBasicControl* control(getControlHandler(BTS_FB_NONE, ptrIpc2BtsMessage->getStackInterface(), functionBlock)); // functionBlock because this indicates the destination control handler
            if(0 != control)
            {
               control->pushIpc2BtsMessage(ptrIpc2BtsMessage);

               // do not delete message
            }
            else
            {
               FW_NORMAL_ASSERT_ALWAYS();
               delete ptrIpc2BtsMessage;
            }
         }
         else
         {
            // should never happen else you have programmed something wrong
            // #error_indication
            FW_NORMAL_ASSERT_ALWAYS();
            delete ptrIpc2BtsMessage;
         }

         // check Bts2AppOutputQueue
         handleBts2AppMessage();

         // check Bts2IpcOutputQueue
         handleBts2IpcMessage();

         counter++;

         if(counter >= MAX_NUMBER_IPC2BTS_MGS_PROCESSING)
         {
            moreMessages = true;
         }
      }
      else
      {
         counter = MAX_NUMBER_IPC2BTS_MGS_PROCESSING;
      }
   }

   return moreMessages;
}

void MainControl::handleTimerTickForDbusIf(void)
{
   for(::std::map< BTSComponentEntry, IDbusBase* >::iterator it = _dbusIfList.begin(); it != _dbusIfList.end(); ++it)
   {
      if(0 != it->second)
      {
         it->second->handleTimerTick(); // TODO: to be tested
      }
   }
}

void MainControl::handleResetOfDbusIfHandler(void)
{
   for(::std::vector< IDbusBase* >::iterator it = _resetDbusIfHandlerList.begin(); it != _resetDbusIfHandlerList.end(); ++it)
   {
      if(0 != (*it))
      {
         for(::std::map< BTSComponentEntry, IDbusBase* >::iterator itHdlr = _dbusIfList.begin(); itHdlr != _dbusIfList.end(); ++itHdlr)
         {
            if((*it) == itHdlr->second)
            {
               itHdlr->second->resetHandler();
            }
         }
      }
   }

   _resetDbusIfHandlerList.clear(); // works because single thread
}

IBasicControl* MainControl::getControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const
{
   IBasicControl* returnIf(0);

   BTSComponentEntry controlEntry;
   controlEntry = getControlComponentKey(controlEntry, component, stackInterface, subComponent);
   ::std::map< BTSComponentEntry, IBasicControl* >::const_iterator it = _controlHandlerList.find(controlEntry);
   if(_controlHandlerList.end() != it)
   {
      returnIf = it->second;
   }

   return returnIf;
}

IDbusBase* MainControl::getDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const
{
   IDbusBase* returnIf(0);

   BTSComponentEntry dbusEntry;
   dbusEntry = getDbusComponentKey(dbusEntry, component, stackInterface, subComponent);
   ::std::map< BTSComponentEntry, IDbusBase* >::const_iterator it = _dbusIfList.find(dbusEntry);
   if(_dbusIfList.end() != it)
   {
      returnIf = it->second;
   }

   return returnIf;
}

void MainControl::timer1Func(void)
{
   ETG_TRACE_USR1((" timer1Func(): active=%d", _timer1.isActive()));
}

void MainControl::timer2Func(void)
{
   ETG_TRACE_USR1((" timer2Func(): active=%d", _timer2.isActive()));
}

void MainControl::timer3Func(void)
{
   ETG_TRACE_USR1((" timer3Func(): active=%d", _timer3.isActive()));
}

} //btstackif
