/**
 * @file MainControl.h
 *
 * @par SW-Component
 * Main
 *
 * @brief Main control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for main control functionality.
 */

#ifndef _MAIN_CONTROL_H_
#define _MAIN_CONTROL_H_

#include "IMainControl.h"
#include "FwISingleThreadFunction.h"
#include "BtsMessageQueue.h"
#include "EventQueue.h"
#include "TTYDeviceMonitor.h"
#include "TimerTickSource.h"
#include "ITimerTickHandler.h"
#include "BtsTimerMaster.h"
#include "BtsTimer.h"

// class forward declarations
namespace fw {
class SingleThread;
} //fw

namespace btstackif {

// class forward declarations
class App2Bts_BaseMessage;
class Bts2App_BaseMessage;
class Bts2Ipc_BaseMessage;
class Ipc2Bts_BaseMessage;
class IBasicControl;
class IDbusBase;
class IGenericStackIf;

/**
 *
 */
class MainControl : public IMainControl, public ::fw::ISingleThreadFunction, public ITimerTickHandler
{
public:
   /**
    * Default constructor.
    */
   MainControl();

private:
  /**
    * Copy constructor.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be copied
    */
   MainControl(const MainControl& ref);

   /**
    * Assignment operator.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be assigned
    */
   MainControl& operator=(const MainControl& ref);

public:
   /**
    * Destructor.
    */
   virtual ~MainControl();

   virtual void setControlAndDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN IBasicControl* controlHandler, IN IDbusBase* dbusIf);

   virtual void resetControlAndDbusIf(void);

   virtual void setGenericStackIf(IN IGenericStackIf* genericStackIf);

   virtual void resetGenericStackIf(void);

   virtual void pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false);

   virtual void pushBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage);

   virtual void pushBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage);

   virtual void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false);

   virtual void stop(void);

   virtual void run(void);

   virtual void triggerRequestCurrentTime(void);

   virtual void addTTYMonitorHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter);

   virtual void printQueueStatistics(void);

   virtual void setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual void triggerResetOfDbusIfHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

   virtual void threadFunction(void* arguments);

   virtual void setTerminate(void* arguments);

   virtual void setTimerTick(void);

private:
   enum
   {
      NMB_MESSAGES_WARNING_THRESHOLD = 128
   };

   enum
   {
      MAX_NUMBER_APP2BTS_MGS_PROCESSING = 5,
      MAX_NUMBER_IPC2BTS_MGS_PROCESSING = 5
   };

   enum
   {
      EV_TERMINATE =          0x00000001,
      EV_TIMER_TICK =         0x00000002,
      EV_TIMER_TEST_START =   0x00000004,
      EV_TIMER_TEST_STOP =    0x00000008,
      EV_APP2BTS_INPUT =      0x00000010,
      EV_IPC2BTS_INPUT =      0x00000020,
      EV_RESET_DBUS_IF =      0x00000040
   };

   MessageQueue<App2Bts_BaseMessage> _app2BtsInputQueue;
   MessageQueue<App2Bts_BaseMessage> _app2BtsHighPrioInputQueue;
   MessageQueue<Bts2App_BaseMessage> _bts2AppOutputQueue;
   MessageQueue<Bts2Ipc_BaseMessage> _bts2IpcOutputQueue;
   MessageQueue<Ipc2Bts_BaseMessage> _ipc2BtsInputQueue;
   MessageQueue<Ipc2Bts_BaseMessage> _ipc2BtsHighPrioInputQueue;
   EventQueue _eventQueue;
   TTYDeviceMonitor _ttyMonitor;
   TimerTickSource _timerSource;
   TimerMaster _timerMaster;
   Timer _timer1; /**< test timer */
   Timer _timer2; /**< test timer */
   Timer _timer3; /**< test timer */
   bool _terminateWorkerThread;
   bool _workerThreadTerminated;
   ::fw::SingleThread* _singleThread;
   IGenericStackIf* _genericStackIf;
   ::std::map< BTSComponentEntry, IBasicControl* > _controlHandlerList;
   ::std::map< BTSComponentEntry, IDbusBase* > _dbusIfList;
   ::std::vector< IDbusBase* > _resetDbusIfHandlerList;

   static MainControl* _exemplar;

   void mainThread(void* arguments);

   bool handleApp2BtsMessage(void);
   void handleBts2AppMessage(void);
   void handleBts2IpcMessage(void);
   bool handleIpc2BtsMessage(void);

   void handleTimerTickForDbusIf(void);

   void handleResetOfDbusIfHandler(void);

   IBasicControl* getControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const;

   IDbusBase* getDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const;

   /**
    * Get instance. For internal use only.
    *
    * @return = pointer to own instance
    */
   static MainControl* getInstance(void);

   /**
    * Timer callback functions. For internal use only.
    */
   static void timer1FuncWrapper(void);
   static void timer2FuncWrapper(void);
   static void timer3FuncWrapper(void);

   /**
    * Timer callback functions for testing.
    */
   void timer1Func(void);
   void timer2Func(void);
   void timer3Func(void);
};

/*
 * inline methods
 */
inline MainControl* MainControl::getInstance(void)
{
   return _exemplar;
}

inline void MainControl::timer1FuncWrapper(void)
{
   MainControl* control = MainControl::getInstance();
   if(0 != control)
   {
      control->timer1Func();
   }
}

inline void MainControl::timer2FuncWrapper(void)
{
   MainControl* control = MainControl::getInstance();
   if(0 != control)
   {
      control->timer2Func();
   }
}

inline void MainControl::timer3FuncWrapper(void)
{
   MainControl* control = MainControl::getInstance();
   if(0 != control)
   {
      control->timer3Func();
   }
}

} //btstackif

#endif //_MAIN_CONTROL_H_
