/**
 * @file IMainControl.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Main control interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for main control functionality.
 */

#ifndef _I_MAIN_CONTROL_H_
#define _I_MAIN_CONTROL_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

class App2Bts_BaseMessage;
class Bts2App_BaseMessage;
class Bts2Ipc_BaseMessage;
class Ipc2Bts_BaseMessage;
class IBasicControl;
class IDbusBase;
class IGenericStackIf;
class ITTYDeviceMonitorHandler;

/**
 *
 */
class IMainControl
{
public:
   virtual ~IMainControl() {}

   /**
    * Sets control handler and DBUS interface (pointer).
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] controlHandler: control handler (pointer)
    * @param[in] dbusIf: DBUS interface (pointer)
    */
   virtual void setControlAndDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN IBasicControl* controlHandler, IN IDbusBase* dbusIf) = 0;

   /**
    * Resets all control handlers and DBUS interfaces (pointer).
    */
   virtual void resetControlAndDbusIf(void) = 0;

   /**
    * Sets generic stack interface (pointer).
    *
    * @param[in] genericStackIf: generic stack interface (pointer)
    */
   virtual void setGenericStackIf(IN IGenericStackIf* genericStackIf) = 0;

   /**
    * Resets generic stack interface (pointer).
    */
   virtual void resetGenericStackIf(void) = 0;

   virtual void pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false) = 0;

   virtual void pushBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage) = 0;

   virtual void pushBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage) = 0;

   virtual void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false) = 0;

   virtual void stop(void) = 0;

   virtual void run(void) = 0;

   /**
    * Triggers request for current time.
    */
   virtual void triggerRequestCurrentTime(void) = 0;

   /**
    * Add TTY monitor handler.
    *
    * @param[in] handler: handler
    * @param[in] filter: filter
    */
   virtual void addTTYMonitorHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter) = 0;

   /**
    * Print queue statistics.
    */
   virtual void printQueueStatistics(void) = 0;

   /**
    * Sets a main control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Trigger reset of Dbus interface handler.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    */
   virtual void triggerResetOfDbusIfHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) = 0;
};

} //btstackif

#endif //_I_MAIN_CONTROL_H_
