/**
 * @file IControlHandler.h
 *
 * @par SW-Component
 * Interface for control handler
 *
 * @brief Interface definition for control handler.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details This file defines the interface for control handler.
 */

#ifndef _I_CONTROL_HANDLER_H_
#define _I_CONTROL_HANDLER_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class IBasicControl;
class ITimerPool;
class App2Bts_BaseMessage;
class Bts2App_BaseMessage;
class Bts2Ipc_BaseMessage;
class Ipc2Bts_BaseMessage;

/**
 * Interface definition.
 */
class IControlHandler
{
public:
   /**
    * Destructor.
    */
   virtual ~IControlHandler() {}

   /**
    * Set basic control interface.
    *
    * @param[in] control: basic control interface
    */
   virtual void setControlIf(IN IBasicControl* control) = 0;

   /**
    * Set timer pool interface.
    *
    * @param[in] timerPool: timer pool interface
    */
   virtual void setTimerPoolIf(IN ITimerPool* timerPool) = 0;

   /**
    * Enable/disable extension.
    *
    * @param[in] enable: enable flag
    */
   virtual void enable(IN const bool enable) = 0;

   /**
    * Initialize.
    */
   virtual void initialize(void) = 0;

   /**
    * Destroy.
    */
   virtual void destroy(void) = 0;

   /**
    * Set configuration data for local stack.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[out] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   virtual void setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration) = 0;

   /**
    * Do a pre-check for App2Bts messages.
    *
    * @param[out] continueProcessing: flag indicating that processing of given message shall be continued or stopped
    * @param[out] rejectRequest: flag for rejecting request
    * @param[out] resultCode: result code to be used for rejecting request
    * @param[out] statusCode: status code to be used for rejecting request
    * @param[out] skipOpCodeCheck: flag for skipping opcode check
    * @param[in] message: App2Bts message to be checked
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool doApp2BtsMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message) = 0;

   /**
    * Get list of similar items (opcodes).
    *
    * @param[out] itemList: similar item list
    * @param[in] message: message we want to have a similar item list for
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool getSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const = 0;

   /**
    * Get list of matching items (opcodes).
    *
    * @param[out] itemList: matching item list
    * @param[out] highPrioItemList: high priority item list
    * @param[in] message: message we want to have a matching item list for
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool getMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const = 0;

   /**
    * Triggers initialized callback.
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool triggerInitializedCallback(void) = 0;

   /**
    * Handle doubled entries (messages that will be not processed via the normal mechanism). A suitable answer shall be sent for such kind of messages.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] message: "doubled" message
    * @param[in] resultCode: result code to be used
    * @param[in] statusCode: status code to be used
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool handleDoubledApp2BtsMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) = 0;

   /**
    * Handle received App2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message
    * @param[in] message: received App2Bts message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool handleApp2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message) = 0;

   /**
    * Handle received Ipc2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] message: received Ipc2Bts message
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool handleIpc2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message) = 0;

   /**
    * Set a sub control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Set a sub control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) = 0;

   /**
    * Handle TTFis command.
    *
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool handleTTFisCommand(IN const unsigned char* testData) = 0;

   /**
    * Handle App2Bts test command.
    *
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   virtual bool handleApp2BtsTestCommand(IN const unsigned char* testData) = 0;
};

} //btstackif

#endif //_I_CONTROL_HANDLER_H_
