/**
 * @file BasicControl.h
 *
 * @par SW-Component
 * Main
 *
 * @brief Basic control.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for basic control functionality.
 */

#ifndef _BASIC_CONTROL_H_
#define _BASIC_CONTROL_H_

#include "IBasicControl.h"
#include "BtsMessageQueue.h"

namespace btstackif {

// class forward declarations
class Bts2Ipc_BaseMessage;
class Bts2App_BaseMessage;
class IControlHandler;

/**
 *
 */
class BasicControl : public IBasicControl
{
protected:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Constructor. Do not use!!!
    */
   BasicControl();

public:
   /**
    * Constructor.
    *
    * @param[in] functionBlock: function block
    */
   BasicControl(IN const BTSFunctionBlock functionBlock);

protected:
   /**
    * Copy constructor.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be copied
    */
   BasicControl(const BasicControl& ref);

   /**
    * Assignment operator.
    *
    * DO NOT USE!!!
    *
    * @param[in] ref: reference of object to be assigned
    */
   BasicControl& operator=(const BasicControl& ref);

public:
   /**
    * Destructor.
    */
   virtual ~BasicControl();

   virtual void setMainControl(IN IMainControl* mainControl);

   virtual void setComponent(IN const BTSFunctionBlock component);

   virtual void setStackInterface(IN const BTSInterfaceType stackInterface);

   virtual void setQueueWarningSize(IN const unsigned int maxSize);

   virtual void pushApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage);

   virtual void pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage);

   virtual void setStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector<BTSDbusInterfaceItem>& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   virtual void triggerInitializedCallback(void);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent);

   virtual void createDbusServiceAvailabilityMessage(IN const BTSCommonEnumClass interface, IN const BTSDbusServiceAvailability availabilityEvent, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSCommonEnumClass busType);

   virtual void sendInternalApp2BtsMessage(IN App2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false);

   virtual void sendInternalBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage);

   virtual void sendInternalIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio = false);

   virtual void sendInternalBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage);

   virtual void sendBts2IpcMessageList(IN ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const BTSApp2BtsMessageCompareItem& item);

   virtual void sendBts2AppMessageList(IN ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList);

   virtual void handleTimeout(IN IExtendedTimeoutHandler* handler, IN const BTSTimerId timerId);

   virtual void addTTYMonitorHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter);

   virtual void printQueueStatistics(void);

   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData);

   virtual void setSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   virtual void checkWaitingQueue(IN const App2Bts_BaseMessage* ptrMessage);

   virtual void removeApp2BtsWaitingMessages(IN const ::std::vector< App2Bts_BaseMessage* >& app2BtsMsgList);

   virtual bool getMatchingWaitingQueueEntries(OUT ::std::vector< App2Bts_BaseMessage* >& msgList, IN const ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList);

   virtual App2Bts_BaseMessage* findApp2BtsWorkingMessageWrapper(IN const BTSApp2BtsMessageCompareItem& item);

   virtual void pushApp2BtsMsgToWorkingQueue(IN App2Bts_BaseMessage* message, bool withLock = true);

   virtual void pushOnTopApp2BtsMsgToWorkingQueue(IN App2Bts_BaseMessage* message, bool withLock = true);

   virtual void pushApp2BtsMsgToWaitingQueue(IN App2Bts_BaseMessage* message, bool withLock = true);

   virtual void pushOnTopApp2BtsMsgToWaitingQueue(IN App2Bts_BaseMessage* message, bool withLock = true);

   virtual unsigned int getApp2BtsWorkingQueueSize(void) const;

   virtual unsigned int getApp2BtsWaitingQueueSize(void) const;

   virtual bool isSimilarOpCodeInWaitingQueue(IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList) const;

   virtual bool isSimilarOpCodeInWorkingQueue(IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList) const;

   virtual void sendDirectAnswerForApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   virtual void sendDirectAnswerForApp2BtsMessagesWrapper(OUT ::std::vector<Bts2App_BaseMessage*>& bts2AppMsgList, IN const ::std::vector<App2Bts_BaseMessage*>& msgList, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

   virtual void getWorkingQueueEntries(OUT ::std::vector<App2Bts_BaseMessage*>& msgList, IN bool withLock = true);

   virtual void getWaitingQueueEntries(OUT ::std::vector<App2Bts_BaseMessage*>& msgList, IN bool withLock = true);

   virtual void emptyWorkingQueue(IN bool withLock = true);

   virtual void emptyWaitingQueue(IN bool withLock = true);

   virtual void triggerResetOfDbusIfHandler(void);

protected:
   const BTSFunctionBlock _functionBlock;
   MessageQueue<App2Bts_BaseMessage> _app2BtsWaitingQueue;
   MessageQueue<App2Bts_BaseMessage> _app2BtsWorkingQueue;
   bool _bts2IpcMessageWasSent;
   bool _checkWaitingQueue;
   const App2Bts_BaseMessage* _ptrApp2BtsMessage;
   static const bool _handleDoubledRequests;
   static const bool _reorderWaitingRequests;
   ::std::vector< IControlHandler* > _extensions; /**< list of extensions */
   BTSFunctionBlock _component; /**< component (function) block => tracing only */
   // matches _functionBlock --- BTSFunctionBlock _subComponent; /**< sub-component (function) block */
   BTSInterfaceType _stackInterface; /**< stack interface to be used (Bluetooth stack) */

   /**
    * Reorder App2Bts messages in waiting queue. Queue is already locked if necessary.
    *
    * @param[in,out] waitingQueue: waiting queue
    */
   virtual void reorderApp2BtsWaitingQueue(INOUT MessageQueue< App2Bts_BaseMessage >& waitingQueue);

   /**
    * Do a pre-check for App2Bts messages.
    *
    * @param[out] rejectRequest: flag for rejecting request
    * @param[out] resultCode: result code to be used for rejecting request
    * @param[out] statusCode: status code to be used for rejecting request
    * @param[out] skipOpCodeCheck: flag for skipping opcode check
    * @param[in] ptrMessage: App2Bts message to be checked
    *
    * @return = true: continue with processing,
    * @return = false: stop processing
    */
   virtual bool doApp2BtsMsgPrecheck(OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* ptrMessage);

   /**
    * Get list of similar items (opcodes).
    *
    * @param[out] itemList: similar item list
    * @param[in] ptrMessage: message we want to have a similar item list for
    */
   virtual void getSimilarOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, IN const App2Bts_BaseMessage* ptrMessage) = 0;

   /**
    * Get list of matching items (opcodes).
    *
    * @param[out] itemList: matching item list
    * @param[out] highPrioItemList: high priority item list
    * @param[in] ptrMessage: message we want to have a matching item list for
    */
   virtual void getMatchingOpCodes(OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList, OUT ::std::vector<BTSApp2BtsMessageMasterCompareItem>& highPrioItemList, IN const App2Bts_BaseMessage* ptrMessage) = 0;

   /**
    * Check for skipping check of similar items (opcodes).
    *
    * @param[in] ptrMessage: message we want to have a similar item list for
    *
    * @return = true: skip the check,
    * @return = false: do the check
    */
   virtual bool skipSimilarOpCodeCheck(IN const App2Bts_BaseMessage* ptrMessage) = 0;

   /**
    * Extended waiting queue check.
    *
    * @param[in] compareItem: compare item with necessary input data
    */
   virtual void checkWaitingQueueExtended(IN const BTSApp2BtsMessageCompareItem& compareItem) = 0;

   /**
    * Handle doubled entries.
    *
    * @param[in] msgList: message list with double entries
    */
   virtual void handleDoubledApp2BtsMessages(IN const ::std::vector<App2Bts_BaseMessage*>& msgList) = 0;

   App2Bts_BaseMessage* findApp2BtsWaitingMessage(IN const BTSApp2BtsOpcode opcode);
   void removeAndDeleteApp2BtsWaitingMessage(IN App2Bts_BaseMessage* ptrMessage);
   App2Bts_BaseMessage* findApp2BtsWorkingMessage(IN const BTSApp2BtsMessageMasterCompareItem& masterItem);
   void removeAndDeleteApp2BtsWorkingMessage(IN App2Bts_BaseMessage* ptrMessage);
   void removeApp2BtsWaitingMessage(IN const App2Bts_BaseMessage* ptrMessage);
   void removeApp2BtsWorkingMessage(IN const App2Bts_BaseMessage* ptrMessage);
   void pushBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage);
   void pushBts2AppMessage(IN Bts2App_BaseMessage* ptrMessage);
   bool isOpCodeInWaitingQueue(OUT unsigned int& msgIndex, IN const ::std::vector<BTSApp2BtsMessageMasterCompareItem>& itemList);
   bool isMatchingFrontApp2BtsWaitingQueueMessage(IN const App2Bts_BaseMessage* ptrMessage) const;

   /**
    * Add control handler.
    *
    * @param[in] handler: control handler
    */
   void addControlHandler(IN IControlHandler* handler);

   /**
    * Process received App2Bts message. Forward to control handler.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] deleteApp2BtsMessage: marker to delete related App2Bts message
    * @param[in] message: received App2Bts message
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool processApp2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT bool& deleteApp2BtsMessage, IN App2Bts_BaseMessage* message);

   /**
    * Execute a pre-check for App2Bts messages.
    *
    * @param[out] continueProcessing: flag indicating that processing of given message shall be continued or stopped
    * @param[out] rejectRequest: flag for rejecting request
    * @param[out] resultCode: result code to be used for rejecting request
    * @param[out] statusCode: status code to be used for rejecting request
    * @param[out] skipOpCodeCheck: flag for skipping opcode check
    * @param[in] message: App2Bts message to be checked
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool executeApp2BtsMsgPrecheck(OUT bool& continueProcessing, OUT bool& rejectRequest, OUT BTSCommonEnumClass& resultCode, OUT BTSCommonEnumClass& statusCode, OUT bool& skipOpCodeCheck, IN App2Bts_BaseMessage* message);

   /**
    * Process received Ipc2Bts message.
    * Handle any error within this function.
    *
    * @param[out] bts2IpcMsgList: list of Bts2Ipc messages to be sent
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[out] messageItem: message item
    * @param[in] message: received Ipc2Bts message
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool processIpc2BtsMessage(OUT ::std::vector< Bts2Ipc_BaseMessage* >& bts2IpcMsgList, OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, OUT BTSHandleIpc2BtsMessageItem& messageItem, IN Ipc2Bts_BaseMessage* message);

   /**
    * Execute configuration data for local stack.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] userMode: user mode
    * @param[out] dbusInterfaces: list of needed DBUS interfaces
    * @param[in] configuration: local stack configuration data
    */
   void executeStackConfiguration(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
            OUT ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration);

   /**
    * Triggers initialized callback.
    *
    * @return = true: given message was handled ("extension is responsible for this message request"),
    * @return = false: given message was not handled ("extension is not responsible for this message request")
    */
   bool processTriggerInitializedCallback(void);

   /**
    * Execute a sub control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool executeSubControlTestCommand(IN const char* testCommand, IN const unsigned int testData);

   /**
    * Execute a sub control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool executeSubControlTestCommand(IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Request list of similar items (opcodes).
    *
    * @param[out] itemList: similar item list
    * @param[in] message: message we want to have a similar item list for
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool requestSimilarOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, IN const App2Bts_BaseMessage* message) const;

   /**
    * Request list of matching items (opcodes).
    *
    * @param[out] itemList: matching item list
    * @param[out] highPrioItemList: high priority item list
    * @param[in] message: message we want to have a matching item list for
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool requestMatchingOpCodes(OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& itemList, OUT ::std::vector< BTSApp2BtsMessageMasterCompareItem >& highPrioItemList, IN const App2Bts_BaseMessage* message) const;

   /**
    * Process doubled entries (messages that will be not processed via the normal mechanism). A suitable answer shall be sent for such kind of messages.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] message: "doubled" message
    * @param[in] resultCode: result code to be used
    * @param[in] statusCode: status code to be used
    *
    * @return = true: given message was handled ("control handler is responsible for this message request"),
    * @return = false: given message was not handled ("control handler is not responsible for this message request")
    */
   bool processDoubledApp2BtsMessages(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN App2Bts_BaseMessage* message, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode);

private:
   IMainControl* _mainControl;
};

} //btstackif

#endif //_BASIC_CONTROL_H_
