/**
 * @file IBtStackIfWblRequest.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Wbl request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for wbl request functionality.
 */

#ifndef _I_BT_STACK_IF_WBL_REQUEST_H_
#define _I_BT_STACK_IF_WBL_REQUEST_H_

#include "BtStackIfTypes.h"

namespace btstackif {

// class forward declaration
class IBtStackIfWblCallback;
class Ipc2Bts_BaseMessage;

/**
 *
 */
class IBtStackIfWblRequest
{
public:
   virtual ~IBtStackIfWblRequest() {}

   /**************************************************************************
    General interfaces
    **************************************************************************/
   /**
    * Register callback instance.
    *
    * @param[in] callback: callback instance
    */
   virtual void registerCallback(IBtStackIfWblCallback* callback) = 0;

   /**
    * Requests a session handle to be used for communication between BtStackIf user and BtStackIf library itself.
    * The session handle provides the possibility to distinguish different BtStackIf users while accessing same BtStackIf request interface.
    * The session handle 0 indicates a common interface call without session relation.
    *
    * @return = session handle, always > 0
    */
   virtual BTSSessionHandle getSessionHandle(void) const = 0;

   /**
    * Initialize the BtStackIf library.
    * - opens/prepares DBUS communication
    * - starts internal threads
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode init(IN const BTSLocalWblConfiguration& configuration) const = 0;

   /**
    * Deinitialize the BtStackIf library.
    * - stops internal threads
    * - closes DBUS communication
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode deInit(void) const = 0;

   /**************************************************************************
    Wbl management related interfaces
    **************************************************************************/
   virtual void setWblDummy(void) = 0;

   virtual void allowWblCommunication(void) = 0;

   virtual void blockWblCommunication(void) = 0;

   virtual void getWblCommunicationStatus(void) = 0;

   virtual void getWblServiceAvailability(void) = 0;

   virtual void getWblLastIntendedMode(void) = 0;

   virtual void getWblSupportedRestrictions(void) = 0;

   virtual void getWblWiFiSetupStatus(void) = 0;

   virtual void getWblHealthinessIndicator(void) = 0;

   /**************************************************************************
    Test related interfaces
    **************************************************************************/
   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const = 0;

   /**
    * Sets a main control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const = 0;

   /**
    * Sets a connection control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const = 0;

   /**
    * Sets a connection control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const = 0;
};

} //btstackif

#endif //_I_BT_STACK_IF_WBL_REQUEST_H_
