/**
 * @file IBtStackIfConnectionRequest.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Connection request interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for connection request functionality.
 */

#ifndef _I_BT_STACK_IF_CONNECTION_REQUEST_H_
#define _I_BT_STACK_IF_CONNECTION_REQUEST_H_

#include "BtStackIfTypes.h"

namespace btstackif {

// class forward declaration
class IBtStackIfConnectionCallback;
class Ipc2Bts_BaseMessage;

/**
 *
 */
class IBtStackIfConnectionRequest
{
public:
   virtual ~IBtStackIfConnectionRequest() {}

   /**************************************************************************
    General interfaces
    **************************************************************************/
   /**
    * Register callback instance.
    *
    * @param[in] callback: callback instance
    */
   virtual void registerCallback(IBtStackIfConnectionCallback* callback) = 0;

   /**
    * Requests a session handle to be used for communication between BtStackIf user and BtStackIf library itself.
    * The session handle provides the possibility to distinguish different BtStackIf users while accessing same BtStackIf request interface.
    * The session handle 0 indicates a common interface call without session relation.
    *
    * @return = session handle, always > 0
    */
   virtual BTSSessionHandle getSessionHandle(void) const = 0;

   /**
    * Initialize the BtStackIf library.
    * - opens/prepares DBUS communication
    * - starts internal threads
    *
    * @param[in] configuration: local stack configuration data
    * @param[in] pairedDevices: list of paired Bluetooth devices
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode init(IN const BTSLocalStackConfiguration& configuration, IN const BTSBDAddressList& pairedDevices) const = 0;

   /**
    * Deinitialize the BtStackIf library.
    * - stops internal threads
    * - closes DBUS communication
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode deInit(void) const = 0;

   /**************************************************************************
    Connection management related interfaces
    **************************************************************************/
   virtual void switchBluetoothOnOff(IN const BTSBluetoothMode bluetoothMode, IN const BTSAdapterMode adapterMode) = 0;

   virtual void getBluetoothOnOffStatus(void) = 0;

   virtual void switchBluetoothHardwareOnOff(IN const BTSBluetoothMode bluetoothMode) = 0;

   virtual void setLocalBtName(IN const BTSDeviceName& adapterName) = 0;

   virtual void getLocalBtName(void) = 0;

   virtual void setDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& serviceRecord) = 0;

   virtual void setDeviceForRestrictedPairingConnecting(IN const BTSBDAddress& expectedDeviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void setLocalAdapterModes(IN const BTSLocalMode discoverableMode, IN const BTSDiscoverableTimeout discoverableTimeout, IN const BTSLocalMode connectableMode, IN const BTSConnectableTimeout connectableTimeout, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void getLocalAdapterModes(IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void startDiscovery(IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void stopDiscovery(IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void getDiscoveringStatus(IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void startPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void cancelPairing(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void setPinCode(IN const BTSBDAddress& deviceAddress, IN const BTSPinCode& pinCode, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void confirmSecureSimplePairing(IN const BTSBDAddress& deviceAddress, IN const BTSSspMode sspMode, IN const BTSNumericValue& numericValue, IN const BTSConfirmationMode confirmationMode, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void setLinkKey(IN const BTSBDAddress& deviceAddress, IN const BTSLinkKeyType linkKeyType, IN const BTSLinkKey& linkKey, IN const BTSDLinkKey& dLinkKey, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void connectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSTimeValue pageTimeout = 10240, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void acceptRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void rejectRemoteProtocolConnect(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void disconnectProtocol(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const bool pauseBtStreaming = true, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void disconnectDevice(IN const BTSBDAddress& deviceAddress, IN const bool pauseBtStreaming = true, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void getProtocolConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSProtocolId protocolId, IN const BTSUuid& sppUuid, IN const BTSMasInstanceName& masInstance, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void getDeviceConnectionStatus(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void startRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const bool rejectIfNotConnected = true, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void cancelRemoteServiceSearch(IN const BTSBDAddress& deviceAddress, IN const BTSSearchType searchType, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void queryRemoteName(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void requestLinkQuality(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void startTestMode(IN const BTSBDAddress& deviceAddress, IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void stopTestMode(IN const BTSSessionHandle sessionHandle = 0) = 0;

   virtual void requestTestModeLinkQuality(IN const BTSSessionHandle sessionHandle = 0) = 0;

   /**************************************************************************
    Test related interfaces
    **************************************************************************/
   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData) const = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData) const = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) const = 0;

   /**
    * Sets a main control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setMainControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const = 0;

   /**
    * Sets a connection control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned int testData) const = 0;

   /**
    * Sets a connection control test command.
    *
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const char* testCommand, IN const unsigned char* testData) const = 0;
};

} //btstackif

#endif //_I_BT_STACK_IF_CONNECTION_REQUEST_H_
