/**
 * @file BtStackIfTypes.h
 *
 * @par SW-Component
 * Types
 *
 * @brief Basic types.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Definitions of all types needed for communication with BtStackIf component.
 */

#ifndef _BT_STACK_IF_TYPES_H_
#define _BT_STACK_IF_TYPES_H_

#include <vector>
#include <map>
#include <string>
#include <stdbool.h> // <cstdbool> is marked with C++11

// #define VARIANT_S_FTR_ENABLE_SPP_SERVER
#undef VARIANT_S_FTR_ENABLE_SPP_SERVER

// #define VARIANT_S_FTR_ENABLE_SPP_INSTANCE_TEST
#undef VARIANT_S_FTR_ENABLE_SPP_INSTANCE_TEST

namespace btstackif {

/*
 * BIG HINT: If you want to add a new define/type/... please check if this type is already defined in BtStackInternalTypes.h, EvolutionGeniviStackTypes.h or SimulationStackTypes.h.
 *           If yes then move it to here.
 *
 * RULE 1: use a class instead of declaring a struct (avoiding problems with copy actions), implement copy constructor, assignment operator and equality/inequality operator
 *
 * keep following order:
 * - class forward declarations
 * - defines (if really necessary)
 * - typedefs
 * - enums
 * - structs => classes
 */

/**
 * Forward declaration of class BTSServiceInfo.
 */
class BTSServiceInfo;

/**
 * Forward declaration of class BTSDiscoveredDeviceInfo.
 */
class BTSDiscoveredDeviceInfo;

/**
 * Forward declaration of class BTSDeviceIdServiceRecord.
 */
class BTSDeviceIdServiceRecord;

/**
 * Forward declaration of class BTSSppCapability.
 */
class BTSSppCapability;

/**
 * Forward declaration of class BTSBitfield.
 */
template < typename TUInt, typename TEnum, int enumMax >
class BTSBitfield;

/**
 * Forward declaration of class BTSLocalStackConfiguration.
 */
class BTSLocalStackConfiguration;

/**
 * Forward declaration of class BTSLocalConfigConfiguration.
 */
class BTSLocalConfigConfiguration;

/**
 * Forward declaration of class BTSLocalConnectionConfiguration.
 */
class BTSLocalConnectionConfiguration;

/**
 * Forward declaration of class BTSLocalTelephonyConfiguration.
 */
class BTSLocalTelephonyConfiguration;

/**
 * Forward declaration of class BTSLocalPhonebookConfiguration.
 */
class BTSLocalPhonebookConfiguration;

/**
 * Forward declaration of class BTSLocalMessagingConfiguration.
 */
class BTSLocalMessagingConfiguration;

/**
 * Forward declaration of class BTSLocalMediaPlayerConfiguration.
 */
class BTSLocalMediaPlayerConfiguration;

/**
 * Forward declaration of class BTSLocalWblConfiguration.
 */
class BTSLocalWblConfiguration;

/**
 * Forward declaration of class BTSLocalEcnrConfiguration.
 */
class BTSLocalEcnrConfiguration;

/**
 * Forward declaration of class BTSLocalVersionInfo.
 */
class BTSLocalVersionInfo;

/**
 * Forward declaration of class BTSDeviceCapabilities.
 */
class BTSDeviceCapabilities;

/**
 * Forward declaration of class BTSPbapSdpRecord.
 */
class BTSPbapSdpRecord;

/**
 * Forward declaration of class BTSMapSdpRecord.
 */
class BTSMapSdpRecord;

/**
 * Forward declaration of class BTSSppServiceInfo.
 */
class BTSSppServiceInfo;

/**
 * Forward declaration of class BTSWblWiFiSetupStatus.
 */
class BTSWblWiFiSetupStatus;

/**
 * Forward declaration of class BTSWblRestriction.
 */
class BTSWblRestriction;

/**
 * Forward declaration of class BTSWblSupportedTechnology.
 */
class BTSWblSupportedTechnology;

/**
 * Forward declaration of class BTSWblSupportedRestriction.
 */
class BTSWblSupportedRestriction;

/**
 * Forward declaration of class BTSWblSupportedRestrictionEntry.
 */
class BTSWblSupportedRestrictionEntry;

/**
 * Forward declaration of class BTSWblWiFiSetupStatusEntry.
 */
class BTSWblWiFiSetupStatusEntry;

/**
 * Forward declaration of class BTSWblLastIntendedMode.
 */
class BTSWblLastIntendedMode;

#ifndef IN
#define IN /**< direction marker for input parameter */
#endif
#ifndef OUT
#define OUT /**< direction marker for output parameter */
#endif
#ifndef INOUT
#define INOUT /**< direction marker for output parameter */
#endif

#define BTS_MAIN_CONTROL_START_TIMER_TEST "BTS_MAIN_CONTROL_START_TIMER_TEST" /**< main control test command */

#define BTS_MAIN_CONTROL_ENABLE_TRACE_4_CURRENT_TIME "BTS_MAIN_CONTROL_ENABLE_TRACE_4_CURRENT_TIME" /**< main control test command */

#define BTS_MAIN_CONTROL_ENABLE_TRACE_4_TIMER_STATISTIC "BTS_MAIN_CONTROL_ENABLE_TRACE_4_TIMER_STATISTIC" /**< main control test command */

#define BTS_MAIN_CONTROL_ENABLE_TRACE_4_TIMOUT_CHECK "BTS_MAIN_CONTROL_ENABLE_TRACE_4_TIMOUT_CHECK" /**< main control test command */

#define BTS_ENABLE_HCI_DUMP "BTS_ENABLE_HCI_DUMP" /**< connection control test command */

#define BTS_SET_HCI_DEVICE "BTS_SET_HCI_DEVICE" /**< connection control test command */

#define BTS_SET_HCI_LOG_FILE "BTS_SET_HCI_LOG_FILE" /**< connection control test command */

/**
 * Type definition for session handle.
 */
typedef unsigned int BTSSessionHandle;

/**
 * Type definition for BT device address.
 * Address value will be given as hexadecimal characters without colon.
 * Example: "aabbccddeeff".
 */
typedef ::std::string BTSBDAddress;

/**
 * Type definition for list of BT device address.
 */
typedef ::std::vector<BTSBDAddress> BTSBDAddressList;

/**
 * Type definition for UUID as string.
 */
typedef ::std::string BTSUuid;

/**
 * Type definition for list of UUID.
 */
typedef ::std::vector<BTSUuid> BTSUuidList;

/**
 * Type definition for BT device name.
 */
typedef ::std::string BTSDeviceName;

/**
 * Type definition for RSSI (Received Signal Strength Indication). If not supported by real Bluetooth stack the value is set to 127.
 */
typedef signed char BTSRssi;

/**
 * Type definition for link quality. If not supported by real Bluetooth stack the value is set to 0xFF.
 */
typedef unsigned char BTSLinkQuality;

/**
 * Type definition for COD (class of device).
 */
typedef unsigned int BTSCod;

/**
 * Type definition for link key.
 * Link key value will be given as hexadecimal characters.
 * Example: "aabbccddeeff00112233445566778899".
 */
typedef ::std::string BTSLinkKey;

/**
 * Type definition for link key.
 * D link key value will be given as hexadecimal characters.
 * Example: "aabbccddeeff00112233445566778899aabbccddeeff00112233445566778899".
 * answer from ALPS:
      Dear Mr.Lippold,

      DLinkkey is used for BT3.0+HS feature (use for AMP controller).
      For Linkkey type, upper layer could just follow the result in BT_APPL_Pairing_Complete_IND.
      -- "ucLinkKeyType" in "const PairingInfoPtr_t pstPairingInfo".

      Best Regards,
      Shigehiro Ito
 */
typedef ::std::string BTSDLinkKey;

/**
 * Type definition for pin code (legacy pairing).
 */
typedef ::std::string BTSPinCode;

/**
 * Type definition for numeric value (secure simple pairing).
 */
typedef ::std::string BTSNumericValue;

/**
 * Type definition for scalar time value. Resolution is ms.
 */
typedef unsigned int BTSTimeValue;

/**
 * Type definition for discoverable timeout.
 */
typedef unsigned int BTSDiscoverableTimeout;

/**
 * Type definition for pairable timeout.
 */
typedef unsigned int BTSPairableTimeout;

/**
 * Type definition for connectable timeout.
 */
typedef unsigned int BTSConnectableTimeout;

/**
 * Type definition for user id.
 */
typedef ::std::string BTSUserId;

/**
 * Type definition for list of Device ID Service Records.
 */
typedef ::std::vector<BTSDeviceIdServiceRecord> BTSDeviceIdServiceRecordList;

/**
 * Type definition for list of service info.
 */
typedef ::std::vector<BTSServiceInfo> BTSServiceInfoList;

/**
 * Type definition for list of SPP service capability.
 */
typedef ::std::vector<BTSSppCapability> BTSSppCapabilityList;

/**
 * Type definition for profile version. HINT: 0xFFFF means unknown.
 */
typedef unsigned short int BTSProfileVersion;

/**
 * Type definition for vendor id.
 */
typedef unsigned short int BTSVendorId;

/**
 * Type definition for product id.
 */
typedef unsigned short int BTSProductId;

/**
 * Type definition for vendor id source.
 */
typedef unsigned short int BTSVendorIdSource;

/**
 * Type definition for version.
 */
typedef unsigned short int BTSVersion;

/**
 * Type definition for WBL WiFi setup id.
 */
typedef unsigned int BTSWblWiFiSetupId;

/**
 * Type definition for WBL healthiness indicator.
 */
typedef unsigned int BTSWblHealthinessIndicator;

/**
 * Type definition for MAS instance id (used for MAP connections).
 * HINT: according MAP spec this is Uint8
 * HINT: during CreateSession() the instance id has to be set as Int32
 */
typedef unsigned char BTSMasInstanceId;

/**
 * Type definition for MAS instance name (used for MAP connections). Empty name indicates that default instance shall be used.
 */
typedef ::std::string BTSMasInstanceName;

/**
 * Type definition for list of MAP SDP records.
 */
typedef ::std::vector< BTSMapSdpRecord > BTSMapSdpRecordList;

/**
 * Type definition for list of SPP service infos.
 */
typedef ::std::vector< BTSSppServiceInfo > BTSSppServiceInfoList;

/**
 * Type definition for status list of WBL restrictions.
 */
typedef ::std::vector< BTSWblRestriction > BTSWblRestrictionList;

/**
 * Type definition for status list of WBL supported restrictions.
 */
typedef ::std::vector< BTSWblSupportedRestriction > BTSWblSupportedRestrictionList;

/**
 * Type definition for status list of WBL supported technologies.
 */
typedef ::std::vector< BTSWblSupportedTechnology > BTSWblSupportedTechnologyList;

/**
 * Type definition for status list of WBL supported restrictions (supported frequencies) entries.
 */
typedef ::std::vector< BTSWblSupportedRestrictionEntry > BTSWblSupportedRestrictionEntryList;

/**
 * Type definition for status list of WBL WiFi setup status entries.
 */
typedef ::std::vector< BTSWblWiFiSetupStatusEntry > BTSWblWiFiSetupStatusEntryList;

/**
 * Type definition for status list of WBL last intended modes.
 */
typedef ::std::vector< BTSWblLastIntendedMode > BTSWblLastIntendedModeList;

/**
 * Possible Bluetooth stack interface implementations.
 */
enum BTSInterfaceType
{
   BTS_IF_UNDEFINED = 0, /**< interface type was not set */
   BTS_IF_SIMULATION, /**< interface implementation to be used for local simulation e.g. for testing */
   BTS_IF_ALPS_EVOLUTION_GENIVI, /**< interface implementation using ALPS Evolution stack with GENIVI interface */
   // -------------------------------------------------------------------------
   BTS_IF_LAST /**< end marker, must be the last one */
};

/**
 * Possible BtStackIf user modes.
 */
enum BTSUserMode
{
   BTS_USER_MODE_CONNECTION = 0, /**< mode for connection handling (includes agent functionality) */
   BTS_USER_MODE_TELEPHONY, /**< mode for phonecall handling */
   BTS_USER_MODE_PHONEBOOK, /**< mode for phonebook handling */
   BTS_USER_MODE_MESSAGING, /**< mode for messaging handling */
   BTS_USER_MODE_MEDIAPLAYER, /**< mode for media player handling */
   BTS_USER_MODE_CONN_WO_AGENT, /**< mode for debug/testing, includes connection handling but without agent functionality */
   BTS_USER_MODE_ALL_WI_AGENT, /**< mode for debug/testing, includes all including agent functionality */
   BTS_USER_MODE_ALL_WO_AGENT, /**< mode for debug/testing, includes all but without agent functionality */
   BTS_USER_MODE_BT_DISABLED, /**< mode for debug/testing, communication with BT stack is disabled */
   // -------------------------------------------------------------------------
   BTS_USER_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for function blocks in BtStackIf.
 * IF ENUM BELOW IS MODIFIED THEN FILE bt_stack_if_get_all_enums.py SHALL BE UPDATED!!!
 */
enum BTSFunctionBlock
{
   BTS_FB_NONE = 0, /**< none */
   BTS_FB_CONFIG, /**< configuration */
   BTS_FB_CONNECTION, /**< connection handling including pairing, connecting, disconnecting, inquiry, service search, BT on/off, production test, ... */
   BTS_FB_TELEPHONY, /**< phonecall handling */
   BTS_FB_PHONEBOOK, /**< phonebook handling */
   BTS_FB_MESSAGING, /**< messaging handling */
   BTS_FB_MEDIAPLAYER, /**< media player handling */
   BTS_FB_WBL, /**< handling for wifi business logic manager */
   BTS_FB_ECNR, /**< handling for ECNR */
   // -------------------------------------------------------------------------
   BTS_FB_LAST /**< end marker, must be the last one */
};

/**
 * Error codes in BT Stack IF implementation.
 */
enum BTSErrorCode
{
   BTS_OK = 0, /**< OK, must be the first code */
   BTS_ERROR, /**< general error */
   BTS_LIB_NOT_INITIALIZED, /**< library was not initialized */
   BTS_INVALID_PARAM, /**< given parameter is invalid */
   BTS_BUSY, /**< another action is ongoing */
   BTS_BT_CONNECTION_EXISTS, /**< at least one BT connection exists */
   BTS_LOCAL_REJECT, /**< request was rejected by BtStackIf layer itself TODO: recheck with BTS_BUSY */
   BTS_QUEUE_LIMIT_REACHED, /**< the maximum size of a queue was reached */
   BTS_NOT_IMPLEMENTED, /**< not implemented */
   BTS_NULL_POINTER, /**< NULL pointer */
   BTS_INIT_ERROR, /**< initialization error */
   BTS_NO_RESOURCES, /**< resources are empty */
   BTS_SEMOP_ERROR, /**< error occurred during semaphore handling */
   BTS_TIMEOUT_ERROR, /**< timeout error occurred */
   // -------------------------------------------------------------------------
   BTS_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for different protocols.
 */
enum BTSProtocolId
{
   BTS_PROTO_HFP = 0, /**< HFP */
   BTS_PROTO_AVP, /**< AVP */
   BTS_PROTO_PIM, /**< PIM */
   BTS_PROTO_MSG, /**< MSG */
   BTS_PROTO_DUN, /**< DUN */
   BTS_PROTO_PAN, /**< PAN */
   BTS_PROTO_SPP, /**< SPP */
   // -------------------------------------------------------------------------
   BTS_PROTO_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for boolean values. DO NOT USE! USE bool INSTEAD! KEEP FOR TRACING!
 */
enum BTSBool
{
   BTS_FALSE = 0, /**< false */
   BTS_TRUE /**< true */
};

/**
 * Type definition for common result of a request.
 */
enum BTSRequestResult
{
   BTS_REQ_SUCCESS = 0, /**< request was successful */
   BTS_REQ_FAILED, /**< request/operation failed */
   BTS_REQ_CONNECT_PROTOCOL_FAILED, /**< connection failed on protocol level */
   BTS_REQ_CONNECT_ACL_FAILED, /**< connection failed on ACL level */
   BTS_REQ_CONNECT_ABORTED, /**< connect was aborted */
   BTS_REQ_DISCONNECT_ABORTED, /**< disconnect was aborted */
   BTS_REQ_PROTOCOL_NOT_SUPPORTED, /**< protocol is not supported */
   BTS_REQ_DISCOVERY_ALREADY_STARTED, /**< discovery session was already started TODO: consider in implementation */
   BTS_REQ_PAIRING_ABORTED, /**< pairing procedure was aborted */
   BTS_REQ_PAIRING_CONN_LOST, /**< failed due to connection lost */
   BTS_REQ_PAIRING_AUTHENTICATION_ERROR, /**< failed due to authentication error */
   BTS_REQ_PAIRING_PAIRING_CANCELLED, /**< failed because cancelled */
   BTS_REQ_PAIRING_MISSING_PIN, /**< failed because PIN is missing */
   BTS_REQ_PAIRING_UNKNOWN, /**< failed due to unknown reason */
   BTS_REQ_BT_INIT_FAILED, /**< BT initialization failed */
   BTS_REQ_SEARCH_NOT_STARTED, /**< related service search was not started */
   BTS_REQ_SPP_UUID_NOT_FOUND, /**< SPP UUID not found in SPP capabilities */
   BTS_REQ_SPP_LIMIT_REACHED, /**< all possible SPP connections are in use */
   BTS_REQ_DBUS_ERROR, /**< request/operation failed due to DBUS error */
   BTS_REQ_CONNECT_TIMEOUT, /**< timeout occurred during connect; PBAP/MAP: timeout happened in Bluetooth stack; can happen if user ignores popup request on Android phone */
   BTS_REQ_CONNECT_REJECTED, /**< connect rejected; PBAP/MAP: connection request was rejected; can happen if user rejects popup request on Android phone or timeout happens on Android phone and a reject is triggered by phone itself */
   BTS_REQ_INVALID_PARAM, /**< given parameter is invalid */
   BTS_REQ_NOT_CONNECTED, /**< device or protocol is not connected */
   BTS_REQ_BUSY, /**< BtStackIf is busy (blocked due to other activities) */
   BTS_REQ_PARTIAL_UPDATE, /**< indicates that update contains only partial information, client shall only update specific information instead of all */
   // -------------------------------------------------------------------------
   BTS_REQ_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for connection status.
 */
enum BTSConnectionStatus
{
   BTS_CONN_CONNECTED = 0, /**< connection is established */
   BTS_CONN_DISCONNECTED, /**< connection is not established */
   // -------------------------------------------------------------------------
   BTS_CONN_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for disconnect(ed) reason.
 */
enum BTSDisconnectReason
{
   BTS_DISCONNECT_REASON_NOT_VALID = 0, /**< to be set while connection status is CONNECTED */
   BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL_OR_REMOTE, /**< disconnect was initiated by local or remote side */
   BTS_DISCONNECT_REASON_NORMAL_LOSS_LOCAL, /**< disconnect was initiated by local side */
   BTS_DISCONNECT_REASON_NORMAL_LOSS_REMOTE, /**< disconnect was initiated by remote side */
   BTS_DISCONNECT_REASON_ABNORMAL_LOSS, /**< disconnect was caused by link loss or exception */
   BTS_DISCONNECT_REASON_ACL_CONNECT_FAILED, /**< device/protocol connection failed; current state is device/protocol disconnected (connection failed on ACL level) */
   BTS_DISCONNECT_REASON_PROTO_CONNECT_FAILED, /**< protocol connection failed; current state is protocol disconnected */
   BTS_DISCONNECT_REASON_PROTO_CONNECT_ABORTED, /**< connect was aborted; current state is protocol disconnected */
   BTS_DISCONNECT_REASON_PROTO_CONNECT_TIMEOUT, /**< timeout occurred during connect; PBAP/MAP: timeout happened in Bluetooth stack; can happen if user ignores popup request on Android phone; current state is protocol disconnected */
   BTS_DISCONNECT_REASON_PROTO_CONNECT_REJECTED, /**< connect rejected; PBAP/MAP: connection request was rejected; can happen if user rejects popup request on Android phone or timeout happens on Android phone and a reject is triggered by phone itself; current state is protocol disconnected */
   BTS_DISCONNECT_REASON_PROTO_NOT_SUPPORTED, /**< protocol is not supported */
   BTS_DISCONNECT_REASON_SPP_UUID_NOT_FOUND, /**< SPP UUID not found in SPP capabilities */
   BTS_DISCONNECT_REASON_SPP_LIMIT_REACHED, /**< all possible SPP connections are in use */
   BTS_DISCONNECT_REASON_MISSING_LINK_KEY, /**< valid link key information is not available, pairing shall be repeated to create new link key */
   BTS_DISCONNECT_REASON_OUT_OF_RANGE, /**< device is not in range, connect failed with reason page timeout */
   BTS_DISCONNECT_REASON_CONFLICT_DIFFERENT_DEVICE, /**< internal disconnect reason; conflict happened during connect, different devices */
   BTS_DISCONNECT_REASON_CONFLICT_SAME_DEVICE, /**< internal disconnect reason; conflict happened during connect, same device */
   // -------------------------------------------------------------------------
   BTS_DISCONNECT_REASON_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for originator of pairing/connect request.
 */
enum BTSPairConnectOriginator
{
   BTS_PAIR_CONNECT_INITIATED_BY_LOCAL_DEVICE = 0, /**< pairing/connect was initiated by local device */
   BTS_PAIR_CONNECT_INITIATED_BY_REMOTE_DEVICE, /**< pairing/connect was initiated by remote device */
   // -------------------------------------------------------------------------
   BTS_PAIR_CONNECT_INITIATED_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for different SSP modes.
 */
enum BTSSspMode
{
   BTS_SSP_WITH_NUMERIC_COMPARISON = 0, /**< SSP with numeric comparison was requested, user shall accept or reject numeric value, remote SSP IO capability is set as 'Display and input yes/no' */
   BTS_SSP_WITH_JUST_WORK_MODE, /**< SSP with just work mode was requested, user shall accept or reject request, remote SSP IO capability is set as 'No input and no output' or 'Display only' */
   BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_REMOTE_DEVICE, /**< SSP with passkey entry mode (input by remote device) was requested, a passkey was given to be displayed, remote SSP IO capability is set as 'Keyboard only' */
   BTS_SSP_WITH_PASSKEY_ENTRY_MODE_INPUT_BY_LOCAL_DEVICE, /**< SSP with passkey entry mode (input by local device) was requested, user shall accept or reject request, user shall enter a passkey in case of accept, remote SSP IO capability is set as 'Display only', use case can only happen if local SSP IO capability is configured as 'Keyboard only') */
   // -------------------------------------------------------------------------
   BTS_SSP_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for confirmation of a request (general reject/accept). TODO: check for general accept/reject
 */
enum BTSConfirmationMode
{
   BTS_CONFIRM_REJECT = 0, /**< reject request */
   BTS_CONFIRM_ACCEPT, /**< accept request */
   // -------------------------------------------------------------------------
   BTS_CONFIRM_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for local adapter modes (discoverable, pairable, connectable).
 */
enum BTSLocalMode
{
   BTS_MODE_DISABLED = 0, /**< mode is disabled */
   BTS_MODE_ENABLED, /**< mode is enabled */
   BTS_MODE_UNCHANGED, /**< keep existing mode, only for request */
   // -------------------------------------------------------------------------
   BTS_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for discovery status.
 */
enum BTSDiscoveringStatus
{
   BTS_DISCOVERING_OFF = 0, /**< discovering is switched off */
   BTS_DISCOVERING_ON, /**< discovering is switched on */
   // -------------------------------------------------------------------------
   BTS_DISCOVERING_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for (switching) Bluetooth on/off.
 */
enum BTSBluetoothMode
{
   BTS_BT_MODE_OFF = 0, /**< (switch) Bluetooth off */
   BTS_BT_MODE_ON, /**< (switch) Bluetooth on */
   BTS_BT_MODE_RF_TEST_MODE, /**< (switch) Bluetooth into RF test mode (Bluetooth type approval tests) */
   // -------------------------------------------------------------------------
   BTS_BT_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Bluetooth off reason.
 */
enum BTSBluetoothOffReason
{
   BTS_BT_OFF_REASON_NOT_VALID = 0, /**< to be set while Bluetooth is switched on (also default value) */
   BTS_BT_OFF_REASON_APP_REQUESTED, /**< change of Bluetooth mode was requested by application */
   BTS_BT_OFF_REASON_STARTUP, /**< Bluetooth is switched on/off due to initial setting of Bluetooth stack during startup (normally Bluetooth is off after startup of Bluetooth stack) */
   BTS_BT_OFF_REASON_STACK_NOT_RUNNING, /**< Bluetooth is switched off due to Bluetooth stack is not running (error case) */
   BTS_BT_OFF_REASON_DBUS_ERROR, /**< Bluetooth is switched off due to failed DBUS access (error case) */
   // -------------------------------------------------------------------------
   BTS_BT_OFF_REASON_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for input and output capabilities of the agent (secure simple pairing).
 */
enum BTSAgentCapability
{
   BTS_AGENT_CAPABILITY_DISPLAY_YES_NO = 0, /**< capability display yes no (default) */
   BTS_AGENT_CAPABILITY_DISPLAY_ONLY, /**< display only */
   BTS_AGENT_CAPABILITY_KEYBOARD_ONLY, /**< keyboard only */
   BTS_AGENT_CAPABILITY_NO_INPUT_NO_OUTPUT, /**< no input no output */
   BTS_AGENT_CAPABILITY_KEYBOARD_DISPLAY, /**< keyboard display */
   // -------------------------------------------------------------------------
   BTS_AGENT_CAPABILITY_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for (switching) local Bluetooth adapter mode.
 */
enum BTSAdapterMode
{
   BTS_ADAPTER_MODE_APP = 0, /**< (switch) local Bluetooth adapter to APP (Application) mode */
   BTS_ADAPTER_MODE_HCI, /**< (switch) local Bluetooth adapter to HCI (Host Controller Interface) mode (RF test mode) */
   // -------------------------------------------------------------------------
   BTS_ADAPTER_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for link key type.
 * possible values (but not sure, taken from RPI and http://pfalcon-oe.blogspot.de/2011/12/bluetooth-link-key-types.html):
 * LINK_KEY_TYPE_COMBINATION_KEY = 0x00,
 * LINK_KEY_TYPE_LOCAL_UNIT_KEY = 0x01,
 * LINK_KEY_TYPE_REMOTE_UNIT_KEY = 0x02,
 * LINK_KEY_TYPE_DEBUG_COMBINATION_KEY = 0x03,
 * LINK_KEY_TYPE_UNAUTHENTICATED_COMBINATION_KEY = 0x04,
 * LINK_KEY_TYPE_AUTHENTICATED_COMBINATION_KEY = 0x05,
 * LINK_KEY_TYPE_CHANGED_COMBINATION_KEY = 0x06,
 */
enum BTSLinkKeyType
{
   BTS_LINK_KEY_TYPE_COMBINATION_KEY = 0, /**< LINK_KEY_TYPE_COMBINATION_KEY */
   BTS_LINK_KEY_TYPE_LOCAL_UNIT_KEY, /**< LINK_KEY_TYPE_LOCAL_UNIT_KEY */
   BTS_LINK_KEY_TYPE_REMOTE_UNIT_KEY, /**< LINK_KEY_TYPE_REMOTE_UNIT_KEY */
   BTS_LINK_KEY_TYPE_DEBUG_COMBINATION_KEY, /**< LINK_KEY_TYPE_DEBUG_COMBINATION_KEY */
   BTS_LINK_KEY_TYPE_UNAUTHENTICATED_COMBINATION_KEY, /**< LINK_KEY_TYPE_UNAUTHENTICATED_COMBINATION_KEY */
   BTS_LINK_KEY_TYPE_AUTHENTICATED_COMBINATION_KEY, /**< LINK_KEY_TYPE_AUTHENTICATED_COMBINATION_KEY */
   BTS_LINK_KEY_TYPE_CHANGED_COMBINATION_KEY, /**< LINK_KEY_TYPE_CHANGED_COMBINATION_KEY */
   BTS_LINK_KEY_TYPE_UNKNOWN, /**< LINK_KEY_TYPE_UNKNOWN */
   // -------------------------------------------------------------------------
   BTS_LINK_KEY_TYPE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for major service class bits (COD, bit definition).
 * See 'The General- and Device-Specific Inquiry Access Codes (DIACs)'.
 */
enum BTSMajorServiceClassBit
{
   BTS_COD_MSC_LIMITED_DISCOVERABLE_MODE = 0, /**< Limited Discoverable Mode */
   BTS_COD_MSC_POSITIONING = 3, /**< Positioning (Location identification) */
   BTS_COD_MSC_NETWORKING = 4, /**< Networking (LAN, Ad hoc, ...) */
   BTS_COD_MSC_RENDERING = 5, /**< Rendering (Printing, Speaker, ...) */
   BTS_COD_MSC_CAPTURING = 6, /**< Capturing (Scanner, Microphone, ...) */
   BTS_COD_MSC_OBJECT_TRANSFER = 7, /**< Object Transfer (v-Inbox, v-Folder, ...) */
   BTS_COD_MSC_AUDIO = 8, /**< Audio (Speaker, Microphone, Headset service, ...) */
   BTS_COD_MSC_TELEPHONY = 9, /**< Telephony (Cordless telephony, Modem, Headset service, ...) */
   BTS_COD_MSC_INFORMATION = 10, /**< Information (WEB-server, WAP-server, ...) */
   // -------------------------------------------------------------------------
   BTS_COD_MSC_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for major service class bitfield (COD).
 */
typedef BTSBitfield<unsigned short int, BTSMajorServiceClassBit, BTS_COD_MSC_LAST> BTSMajorServiceClass;

/**
 * Type definition for major device class (COD).
 * See 'The General- and Device-Specific Inquiry Access Codes (DIACs)'.
 */
enum BTSMajorDeviceClass
{
   BTS_COD_MAJORDC_MISCELLANEOUS = 0, /**< Miscellaneous */
   BTS_COD_MAJORDC_COMPUTER, /**< Computer (desktop,notebook, PDA, organizers, .... ) */
   BTS_COD_MAJORDC_PHONE, /**< Phone (cellular, cordless, payphone, modem, ...) */
   BTS_COD_MAJORDC_LAN, /**< LAN /Network Access point */
   BTS_COD_MAJORDC_AUDIO, /**< Audio/Video (headset,speaker,stereo, video display, vcr..... */
   BTS_COD_MAJORDC_PERIPHERAL, /**< Peripheral (mouse, joystick, keyboards, ..... ) */
   BTS_COD_MAJORDC_IMAGING, /**< Imaging (printing, scanner, camera, display, ...) */
   BTS_COD_MAJORDC_WEARABLE, /**< Wearable */
   BTS_COD_MAJORDC_TOY, /**< Toy */
   BTS_COD_MAJORDC_HEALTH, /**< Health */
   BTS_COD_MAJORDC_UNCATEGORIZED = 0x1F, /**< Uncategorized, specific device code not specified */
   // -------------------------------------------------------------------------
   BTS_COD_MAJORDC_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for minor device class (COD).
 * See 'The General- and Device-Specific Inquiry Access Codes (DIACs)'.
 */
enum BTSMinorDeviceClass
{
   BTS_COD_MINORDC_UNCATEGORIZED = 0, /**< Uncategorized, code for device not assigned */
   BTS_COD_MINORDC_DESKTOP, /**< Desktop workstation */
   BTS_COD_MINORDC_SERVER, /**< Server-class computer */
   BTS_COD_MINORDC_LAPTOP, /**< Laptop */
   BTS_COD_MINORDC_HANDHELD, /**< Handheld PC/PDA (clam shell) */
   BTS_COD_MINORDC_PALM, /**< Palm sized PC/PDA */
   BTS_COD_MINORDC_WEARABLE, /**< Wearable computer (Watch sized) */
   // -------------------------------------------------------------------------
   BTS_COD_MINORDC_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for service search.
 */
enum BTSSearchType
{
   BTS_SEARCH_ALL = 0, /**< ALL */
   BTS_SEARCH_SPP, /**< SPP */
   BTS_SEARCH_DID, /**< DID (PnPInformation) */
   BTS_SEARCH_PBAP, /**< PBAP (internal use only) */
   BTS_SEARCH_MAP, /**< MAP (internal use only) */
   BTS_SEARCH_PAN, /**< PAN (internal use only) */
   // -------------------------------------------------------------------------
   BTS_SEARCH_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for supported services bits.
 */
enum BTSSupportedServicesBit
{
   BTS_SUPP_SRV_HFP = 0, /**< HFP */
   BTS_SUPP_SRV_A2DP = 1, /**< A2DP */
   BTS_SUPP_SRV_AVRCP = 2, /**< AVRCP */
   BTS_SUPP_SRV_PBAP = 3, /**< PBAP */
   BTS_SUPP_SRV_MAP = 4, /**< MAP */
   BTS_SUPP_SRV_DUN = 5, /**< DUN */
   BTS_SUPP_SRV_PAN = 6, /**< PAN */
   BTS_SUPP_SRV_SPP = 7, /**< SPP */
   BTS_SUPP_SRV_PNP = 8, /**< PNP (Portable Navigation Profile) */
   BTS_SUPP_SRV_DID = 9, /**< DID (PnPInformation) */
   // -------------------------------------------------------------------------
   BTS_SUPP_SRV_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for supported services bitfield.
 */
typedef BTSBitfield<unsigned short int, BTSSupportedServicesBit, BTS_SUPP_SRV_LAST> BTSSupportedServices;

/**
 * Type definition for discoverable/connectable off reason.
 */
enum BTSLocalModeOffReason
{
   BTS_MODE_OFF_REASON_NOT_VALID = 0, /**< to be set while local mode is set to on (also default value) */
   BTS_MODE_OFF_REASON_APP_REQUESTED, /**< change of local mode was requested by application */
   BTS_MODE_OFF_REASON_TIMEOUT, /**< mode was switched back to off because timeout happened */
   BTS_MODE_OFF_REASON_INTERNAL, /**< mode was switched back to off due to BtStackIf internal reasons; includes also spontaneous change from Bluetooth stack */
   // -------------------------------------------------------------------------
   BTS_MODE_OFF_REASON_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for BT core version.
 */
enum BTSBtCoreVersion
{
   BTS_BT_CORE_1_0_B = 0, /**< Bluetooth Core Specification 1.0b */
   BTS_BT_CORE_1_1, /**< Bluetooth Core Specification 1.1 */
   BTS_BT_CORE_1_2, /**< Bluetooth Core Specification 1.2 */
   BTS_BT_CORE_2_0_EDR, /**< Bluetooth Core Specification 2.0 + EDR */
   BTS_BT_CORE_2_1_EDR, /**< Bluetooth Core Specification 2.1 + EDR */
   BTS_BT_CORE_3_0_HS, /**< Bluetooth Core Specification 3.0 + HS */
   BTS_BT_CORE_4_0, /**< Bluetooth Core Specification 4.0 */
   BTS_BT_CORE_4_1, /**< Bluetooth Core Specification 4.1 */
   BTS_BT_CORE_4_2, /**< Bluetooth Core Specification 4.2 */
   BTS_BT_CORE_5_0, /**< Bluetooth Core Specification 5.0 */
   // -------------------------------------------------------------------------
   BTS_BT_CORE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL WIFI frequency.
 */
enum BTSWblWifiFrequency
{
   BTS_WBL_WIFI_FREQUENCY_2_4_GHZ = 0, /**< 2.4 GHz */
   BTS_WBL_WIFI_FREQUENCY_5_GHZ, /**< 5 GHz */
   // -------------------------------------------------------------------------
   BTS_WBL_WIFI_FREQUENCY_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL access point configuration type.
 */
enum BTSWblAccessPointConfigType
{
   BTS_WBL_AP_CONFIG_NORMAL = 0, /**< normal */
   BTS_WBL_AP_CONFIG_CPW, /**< car play wireless */
   BTS_WBL_AP_CONFIG_AA, /**< android auto */
   BTS_WBL_AP_CONFIG_UNKNOWN, /**< unknown */
   BTS_WBL_AP_CONFIG_RESTRICTED, /**< restricted */
   // -------------------------------------------------------------------------
   BTS_WBL_AP_CONFIG_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL WIFI power state.
 */
enum BTSWblWifiPowerState
{
   BTS_WBL_WIFI_STATE_DEVICE_UNAVAILABLE = 0, /**< The WIFI device is not available yet. */
   BTS_WBL_WIFI_STATE_DEVICE_AVAILABLE, /**< The WIFI device is available to be turned ON. */
   BTS_WBL_WIFI_STATE_POWERING_ON, /**<  In the process of turning ON. */
   BTS_WBL_WIFI_STATE_POWERED_ON, /**< WIFI AP/STA/P2P mode is turned ON and the device is UP. */
   BTS_WBL_WIFI_STATE_POWERING_OFF, /**< In the process of turning OFF. */
   BTS_WBL_WIFI_STATE_POWERED_OFF, /**< WIFI AP/STA/P2P mode is turned OFF and the device is DOWN. */
   BTS_WBL_WIFI_STATE_POWER_FAILURE, /**< An unexpected error occurred while turning ON/OFF WIFI. */
   // -------------------------------------------------------------------------
   BTS_WBL_WIFI_STATE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL WIFI mode.
 */
enum BTSWblWifiMode
{
   BTS_WBL_WIFI_MODE_STA = 0, /**< station */
   BTS_WBL_WIFI_MODE_AP, /**< access point */
   BTS_WBL_WIFI_MODE_P2P, /**< peer to peer */
   BTS_WBL_WIFI_MODE_ALL, /**< all */
   // -------------------------------------------------------------------------
   BTS_WBL_WIFI_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL supported setting (w.r.t. SPI technologies).
 */
enum BTSWblSupportedSetting
{
   BTS_WBL_SUPPORTED_SETTING_NOT_ALLOWED = 0, /**< not allowed */
   BTS_WBL_SUPPORTED_SETTING_ALLOWED, /**< allowed */
   BTS_WBL_SUPPORTED_SETTING_ALLOWED_WITH_LIMITATIONS, /**< allowed with limitations */
   // -------------------------------------------------------------------------
   BTS_WBL_SUPPORTED_SETTING_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL restriction action.
 */
enum BTSWblRestrictionAction
{
   BTS_WBL_RESTRICTION_ACTION_DEACTIVATE = 0, /**< deactivate */
   BTS_WBL_RESTRICTION_ACTION_ACTIVATE, /**< activate */
   // -------------------------------------------------------------------------
   BTS_WBL_RESTRICTION_ACTION_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for WBL restriction state.
 */
enum BTSWblRestrictionState
{
   BTS_WBL_RESTRICTION_STATE_REQUESTED = 0, /**< requested */
   BTS_WBL_RESTRICTION_STATE_ACTIVATING, /**< activating */
   BTS_WBL_RESTRICTION_STATE_ACTIVATED, /**< activated */
   BTS_WBL_RESTRICTION_STATE_DEACTIVATING, /**< deactivating */
   BTS_WBL_RESTRICTION_STATE_DEACTIVATED, /**< deactivated */
   BTS_WBL_RESTRICTION_STATE_FAILED, /**< failed */
   // -------------------------------------------------------------------------
   BTS_WBL_RESTRICTION_STATE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for PBAP supported repositories (bits).
 */
enum BTSPbapSupportedRepositoriesBit
{
   BTS_PBAP_REPO_LOCAL_PHONEBOOK = 0, /**< local phonebook (me) */
   BTS_PBAP_REPO_SIM_CARD, /**< SIM card (sim) */
   BTS_PBAP_REPO_SPEED_DIAL, /**< speed dial (sdp) */
   BTS_PBAP_REPO_FAVORITES, /**< favorites (fav) */
   // -------------------------------------------------------------------------
   BTS_PBAP_REPO_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for PBAP supported repositories bitfield.
 */
typedef BTSBitfield<unsigned char, BTSPbapSupportedRepositoriesBit, BTS_PBAP_REPO_LAST> BTSPbapSupportedRepositories;

/**
 * Type definition for PBAP supported features (bits).
 * Note from PBAP specification: Backwards compatibility: If the PbapSupportedFeatures attribute is not present 0x00000003 shall be assumed for a remote PSE.
 */
enum BTSPbapSupportedFeaturesBit
{
   BTS_PBAP_FEA_DOWNLOAD = 0, /**< Download */
   BTS_PBAP_FEA_BROWSING, /**< Browsing */
   BTS_PBAP_FEA_DATABASE_IDENTIFIER, /**< Database Identifier */
   BTS_PBAP_FEA_FOLDER_VERSION_COUNTERS, /**< Folder Version Counters */
   BTS_PBAP_FEA_VCARD_SELECTING, /**< vCard Selecting */
   BTS_PBAP_FEA_ENHANCED_MISSED_CALLS, /**< Enhanced Missed Calls */
   BTS_PBAP_FEA_X_BT_UCI_VCARD_PROPERTY, /**< X-BT-UCI vCard Property */
   BTS_PBAP_FEA_X_BT_UID_VCARD_PROPERTY, /**< X-BT-UID vCard Property */
   BTS_PBAP_FEA_CONTACT_REFERENCING, /**< Contact Referencing */
   BTS_PBAP_FEA_DEFAULT_CONTACT_IMAGE_FORMAT, /**< Default Contact Image Format */
   // -------------------------------------------------------------------------
   BTS_PBAP_FEA_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for PBAP supported features bitfield.
 */
typedef BTSBitfield<unsigned int, BTSPbapSupportedFeaturesBit, BTS_PBAP_FEA_LAST> BTSPbapSupportedFeatures;

/**
 * Type definition for MAP supported message types (bits).
 */
enum BTSMapSupportedMessageTypesBit
{
   BTS_MAP_MSG_TYPE_EMAIL = 0, /**< EMAIL: emails on RFC2822 or MIME type basis */
   BTS_MAP_MSG_TYPE_SMS_GSM, /**< SMS: short messages for cellular or other networks (GSM network) */
   BTS_MAP_MSG_TYPE_SMS_CDMA, /**< SMS: short messages for cellular or other networks (CDMA network) */
   BTS_MAP_MSG_TYPE_MMS, /**< MMS: 3GPP MMS messages */
   BTS_MAP_MSG_TYPE_IM, /**< IM: Instant Messages on MIME type basis (MAP 1.3) */
   // -------------------------------------------------------------------------
   BTS_MAP_MSG_TYPE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for MAP supported message types bitfield.
 */
typedef BTSBitfield<unsigned char, BTSMapSupportedMessageTypesBit, BTS_MAP_MSG_TYPE_LAST> BTSMapSupportedMessageTypes;

/**
 * Type definition for MAP supported features (bits).
 * Note from MAP 1.3 specification: Backwards compatibility: If the MapSupportedFeatures attribute is not present, 0x0000001F shall be assumed for a remote MSE.
 */
enum BTSMapSupportedFeaturesBit
{
   BTS_MAP_FEA_NOTIFICATION_REGISTRATION = 0, /**< Notification Registration Feature */
   BTS_MAP_FEA_NOTIFICATION, /**< Notification Feature */
   BTS_MAP_FEA_BROWSING, /**< Browsing Feature */
   BTS_MAP_FEA_UPLOADING, /**< Uploading Feature */
   BTS_MAP_FEA_DELETE, /**< Delete Feature */
   BTS_MAP_FEA_INSTANCE_INFORMATION, /**< Instance Information Feature */
   BTS_MAP_FEA_EXTENDED_EVENT_REPORT_1_1, /**< Extended Event Report 1.1 */
   BTS_MAP_FEA_EVENT_REPORT_VERSION_1_2, /**< Event Report Version 1.2 (MAP 1.3) */
   BTS_MAP_FEA_MESSAGE_FORMAT_VERSION_1_1, /**< Message Format Version 1.1 (MAP 1.3) */
   BTS_MAP_FEA_MESSAGES_LISTING_FORMAT_VERSION_1_1, /**< Messages-Listing Format Version 1.1 (MAP 1.3) */
   BTS_MAP_FEA_PERSISTENT_MESSAGE_HANDLES, /**< Persistent Message Handles (MAP 1.3) */
   BTS_MAP_FEA_DATABASE_IDENTIFIER, /**< Database Identifier (MAP 1.3) */
   BTS_MAP_FEA_FOLDER_VERSION_COUNTER, /**< Folder Version Counter (MAP 1.3) */
   BTS_MAP_FEA_CONVERSATION_VERSION_COUNTERS, /**< Conversation Version Counters (MAP 1.3) */
   BTS_MAP_FEA_PARTICIPANT_PRESENCE_CHANGE_NOTIFICATION, /**< Participant Presence Change Notification (MAP 1.3) */
   BTS_MAP_FEA_PARTICIPANT_CHAT_STATE_CHANGE_NOTIFICATION, /**< Participant Chat State Change Notification (MAP 1.3) */
   BTS_MAP_FEA_PBAP_CONTACT_CROSS_REFERENCE, /**< PBAP Contact Cross Reference (MAP 1.3) */
   BTS_MAP_FEA_NOTIFICATION_FILTERING, /**< Notification Filtering (MAP 1.3) */
   BTS_MAP_FEA_UTC_OFFSET_TIMESTAMP_FORMAT, /**< UTC Offset Timestamp Format (MAP 1.3) */
   BTS_MAP_FEA_MAP_SUPPORTED_FEATURES_IN_CONNECT_REQUEST, /**< MapSupportedFeatures in Connect Request (MAP 1.3) */
   BTS_MAP_FEA_CONVERSATION_LISTING, /**< Conversation listing (MAP 1.3) */
   BTS_MAP_FEA_OWNER_STATUS, /**< Owner Status (MAP 1.3) */
   // -------------------------------------------------------------------------
   BTS_MAP_FEA_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for MAP supported features bitfield.
 */
typedef BTSBitfield<unsigned int, BTSMapSupportedFeaturesBit, BTS_MAP_FEA_LAST> BTSMapSupportedFeatures;

/**
 * Type definition for HCI chip vendor IDs.
 */
enum BTSHciChipVendorId
{
   BTS_HCI_CHIP_VENDOR_UNKNOWN = 0, /**< unknown */
   BTS_HCI_CHIP_VENDOR_MARVELL, /**< Marvell */
   BTS_HCI_CHIP_VENDOR_CSR, /**< CSR */
   // -------------------------------------------------------------------------
   BTS_HCI_CHIP_VENDOR_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for BT module IDs.
 */
enum BTSBtModuleId
{
   BTS_BT_MODULE_UNKNOWN = 0, /**< unknown */
   BTS_BT_MODULE_UGXZE_104A, /**< UGXZE_104A, CSR */
   BTS_BT_MODULE_UGXZE_304A, /**< UGXZE_304A, CSR */
   BTS_BT_MODULE_UGKZ2_H, /**< UGKZ2-H, CSR */
   BTS_BT_MODULE_UGKZ2_E, /**< UGKZ2-E, CSR */
   BTS_BT_MODULE_UGKZ2_2, /**< UGKZ2-2, CSR */
   BTS_BT_MODULE_UGKZ7, /**< UGKZ7, Marvell */
   // -------------------------------------------------------------------------
   BTS_BT_MODULE_LAST /**< end marker, must be the last one */
};

/**
 * Default RSSI value.
 */
extern const BTSRssi defaultRssiValue;

/**
 * Default link quality value.
 */
extern const BTSLinkQuality defaultLinkQualityValue;

/**
 * Value for empty profile version.
 */
extern const BTSProfileVersion emptyProtocolVersion;

/**
 * Value for default profile version (unknown).
 */
extern const BTSProfileVersion defaultProtocolVersion;

/**
 * Type definition for bitfields.
 */
template < typename TUInt, typename TEnum, int enumMax >
class BTSBitfield
{
public:
   /**
    * Constructor.
    */
   BTSBitfield()
   {
      _data = 0;
      _bitMask = (TUInt)(((sizeof(TUInt)) << 3) - 1);
      _dataMask = 0;
      for(TUInt i = 0; i < (TUInt)enumMax; i++)
      {
         // old: _dataMask |= (TUInt)(1 << (i & _bitMask));
         _dataMask = (TUInt)(_dataMask | (1 << (i & _bitMask)));
      }
   }

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   BTSBitfield(IN const BTSBitfield& ref)
   {
      _data = ref._data;
      _bitMask = ref._bitMask;
      _dataMask = ref._dataMask;
   }

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    *
    * @return = reference of message
    */
   BTSBitfield& operator=(IN const BTSBitfield& ref)
   {
      if(this == &ref)
      {
         return *this;
      }

      _data = ref._data;
      _bitMask = ref._bitMask;
      _dataMask = ref._dataMask;

      return *this;
   }

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: equal
    * @return = false: not equal
    */
   bool operator==(IN const BTSBitfield& ref) const
   {
      bool result = true;

      result = (true == result) && (_data == ref._data);
      result = (true == result) && (_bitMask == ref._bitMask);
      result = (true == result) && (_dataMask == ref._dataMask);

      return result;
   }

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: not equal
    * @return = false: equal
    */
   bool operator!=(IN const BTSBitfield& ref) const
   {
      return !(operator==(ref));
   }

   /**
    * Destructor.
    */
   virtual ~BTSBitfield()
   {
   }

   /**
    * Set data.
    *
    * @param[in] data: data as integer
    */
   inline void setData(IN const TUInt data)
   {
      _data = data & _dataMask;
   }

   /**
    * Get data.
    *
    * @return = data as integer
    */
   inline TUInt getData(void) const
   {
      return _data;
   }

   /**
    * Set bit.
    *
    * @param[in] bit: bit
    */
   inline void setBit(IN const TEnum bit)
   {
      if(enumMax > (int)bit)
      {
         // old:     _data |= (TUInt)(1 << ((TUInt)bit & _bitMask));
         _data = (TUInt)(_data | (1 << ((TUInt)bit & _bitMask)));
      }
   }

   /**
    * Reset bit.
    *
    * @param[in] bit: bit
    */
   inline void resetBit(IN const TEnum bit)
   {
      if(enumMax > (int)bit)
      {
         _data &= (TUInt)(~((TUInt)(1 << ((TUInt)bit & _bitMask))));
      }
   }

   /**
    * Get bit.
    *
    * @param[in] bit: bit
    *
    * @return = true: bit is set
    * @return = false: bit is not set
    */
   inline bool getBit(IN const TEnum bit) const
   {
      if(enumMax > (int)bit)
      {
         return (0 != (_data & (TUInt)(1 << ((TUInt)bit & _bitMask))));
      }
      return false;
   }

   /**
    * Get data mask. For testing.
    *
    * @return = data mask
    */
   inline TUInt getDataMask(void) const
   {
      return _dataMask;
   }

   /**
    * Get bit mask. For testing.
    *
    * @return = bit mask
    */
   inline TUInt getBitMask(void) const
   {
      return _bitMask;
   }

private:
   TUInt _data; /**< data container */
   TUInt _bitMask; /**< bit mask */
   TUInt _dataMask; /**< data mask */
};

/**
 * Type definition for status information of a discovered device.
 */
class BTSServiceInfo
{
public:
   BTSServiceInfo();
   BTSServiceInfo(IN const BTSServiceInfo& ref);
   BTSServiceInfo& operator=(IN const BTSServiceInfo& ref);
   bool operator==(IN const BTSServiceInfo& ref) const;
   bool operator!=(IN const BTSServiceInfo& ref) const;
   virtual ~BTSServiceInfo();
   // member with public access
   BTSSupportedServicesBit service; /**< supported service (profile) */
   BTSProfileVersion version; /**< profile version, 0xFFFF means unknown */
};

/**
 * Type definition for status information of a discovered device.
 */
class BTSDiscoveredDeviceInfo
{
public:
   BTSDiscoveredDeviceInfo();
   BTSDiscoveredDeviceInfo(IN const BTSDiscoveredDeviceInfo& ref);
   BTSDiscoveredDeviceInfo& operator=(IN const BTSDiscoveredDeviceInfo& ref);
   bool operator==(IN const BTSDiscoveredDeviceInfo& ref) const;
   bool operator!=(IN const BTSDiscoveredDeviceInfo& ref) const;
   virtual ~BTSDiscoveredDeviceInfo();
   // member with public access
   BTSBDAddress address; /**< BT device address */
   BTSDeviceName name; /**< BT device name */
   BTSRssi rssi; /**< RSSI of ACL connection to BT device */
   BTSMajorServiceClass majorServiceClass; /**< parameter major service class */
   BTSMajorDeviceClass majorDeviceClass; /**< parameter major device class */
   BTSMinorDeviceClass minorDeviceClass; /**< parameter minor device class */
   BTSSupportedServices services; /**< supported services */
   BTSSppCapabilityList sppCapabilities; /**< SPP capabilities */
};

/**
 * Type definition for Device ID Service Record.
 */
class BTSDeviceIdServiceRecord
{
public:
   BTSDeviceIdServiceRecord();
   BTSDeviceIdServiceRecord(IN const BTSDeviceIdServiceRecord& ref);
   BTSDeviceIdServiceRecord& operator=(IN const BTSDeviceIdServiceRecord& ref);
   bool operator==(IN const BTSDeviceIdServiceRecord& ref) const;
   bool operator!=(IN const BTSDeviceIdServiceRecord& ref) const;
   virtual ~BTSDeviceIdServiceRecord();
   // member with public access
   unsigned short int specificationID; /**< version number of the Bluetooth Device ID Profile specification supported by the device, should be always 0x0103 */
   BTSVendorId vendorID; /**< vendor of the device; VendorID 0xFFFF is reserved as the default VendorID when no Device ID Service Record is present in the device */
   BTSProductId productID; /**< indicates different products made by the vendor above; managed by the vendors themselves */
   BTSVersion version; /**< numeric expression identifying the device release number in Binary-Coded Decimal; vendor-assigned field; value of the field is 0xJJMN for version JJ.M.N (JJ - major version number, M - minor version number, N - sub-minor version number) */
   bool primaryRecord; /**< intended to designate one particular Device ID Service Record (in case multiple Device ID Service Records exist on a device) as the primary Device ID Service Record for the device; true: the primary record, false: not the primary record */
   BTSVendorIdSource vendorIDSource; /**< designates which organization assigned the VendorID attribute; 0x0001: Bluetooth SIG, 0x0002: USB Implementer's Forum, 0x0000 and 0x0003 ~ 0xFFFF: Reserved */
   ::std::string clientExecutableURL; /**< client executable URL */
   ::std::string serviceDescription; /**< service description */
   ::std::string clientDocumentationURL; /**< client documentation URL */
};

/**
 * Type definition for SPP service search result.
 */
class BTSSppCapability
{
public:
   BTSSppCapability();
   BTSSppCapability(IN const BTSSppCapability& ref);
   BTSSppCapability& operator=(IN const BTSSppCapability& ref);
   bool operator==(IN const BTSSppCapability& ref) const;
   bool operator!=(IN const BTSSppCapability& ref) const;
   virtual ~BTSSppCapability();
   // member with public access
   BTSUuidList uuidList; /**< uuid list */
   ::std::string serviceName; /**< service name */
};

/**
 * Type definition for local stack configuration.
 */
class BTSLocalStackConfiguration
{
public:
   BTSLocalStackConfiguration();
   BTSLocalStackConfiguration(IN const BTSLocalStackConfiguration& ref);
   BTSLocalStackConfiguration& operator=(IN const BTSLocalStackConfiguration& ref);
   bool operator==(IN const BTSLocalStackConfiguration& ref) const;
   bool operator!=(IN const BTSLocalStackConfiguration& ref) const;
   virtual ~BTSLocalStackConfiguration();
   // member with public access
   BTSDeviceName name; /**< local device name (user friendly name of the Bluetooth adapter) */
   BTSSupportedServices supportedServices; /**< supported services of local Bluetooth adapter */
   BTSDeviceIdServiceRecord deviceIdServiceRecord; /**< device id service record of local Bluetooth adapter */
   bool wbsEnabled; /**< wide band speech enabled flag: default=true */
   bool pbdlViaPbapEnabled; /**< flag for phonebook download via PBAP: default=true */
   bool pbdlViaHfpEnabled; /**< flag for phonebook download via HFP: default=false */
   bool pbdlViaSppEnabled; /**< flag for phonebook download via SPP: default=false */
   bool pbdlViaSyncMlEnabled; /**< flag for phonebook download via SyncML: default=false */
   BTSLocalMode initialDiscoverableMode; /**< initial discoverable mode: default=BTS_MODE_DISABLED */
   BTSLocalMode initialConnectableMode; /**< initial connectable mode: default=BTS_MODE_DISABLED */
   bool voiceRecognitionEnabled; /**< voice recognition enabled flag: default=true */
   bool enhancedCallControlEnabled; /**< enhanced call control enabled flag: default=true */
   bool avrcpBrowsingEnabled; /**< AVRCP browsing enabled flag: default=true */
   bool audioCodecMp3Enabled; /**< audio codec MP3 enabled flag: default=false */
   bool audioCodecAacEnabled; /**< audio codec AAC enabled flag: default=false */
   bool mapMmsEnabled; /**< MMS feature enabled flag: default=true */
   bool mapEmailEnabled; /**< EMAIL feature enabled flag: default=false */
   bool bluetoothLowEnergyEnabled; /**< Bluetooth low energy feature enabled flag: default=false */
   bool carPlayWirelessEnabled; /**< CarPlay wireless feature enabled flag: default=false TODO: will be replaced by sppServiceInfoList */
   bool enableServiceSearchAfterFirstProtocol; /**< flag to enable service search after first connected protocol: default=true */
   bool stopInquiryDuringIncomingPairing; /**< flag to stop inquiry during incoming pairing request: default=true */
   unsigned char maxNumberConnectedDevices; /**< maximum number of connected devices: default=3 */
   bool sppServerEnabled; /**< SPP server feature enabled flag: default=false TODO: will be replaced by sppServiceInfoList */
   bool sppInstanceTestEnabled; /**< SPP instance test enabled flag: default=false */
   BTSSppServiceInfoList sppServiceInfoList; /**< SPP service info list: default=empty */
   BTSTimeValue defaultPageTimeout; /**< default page timeout */
   BTSAgentCapability agentCapability; /**< default agent capability */
};

/**
 * Type definition for local config configuration.
 */
class BTSLocalConfigConfiguration
{
public:
   BTSLocalConfigConfiguration();
   BTSLocalConfigConfiguration(IN const BTSLocalConfigConfiguration& ref);
   BTSLocalConfigConfiguration& operator=(IN const BTSLocalConfigConfiguration& ref);
   bool operator==(IN const BTSLocalConfigConfiguration& ref) const;
   bool operator!=(IN const BTSLocalConfigConfiguration& ref) const;
   virtual ~BTSLocalConfigConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local connection configuration.
 */
class BTSLocalConnectionConfiguration
{
public:
   BTSLocalConnectionConfiguration();
   BTSLocalConnectionConfiguration(IN const BTSLocalConnectionConfiguration& ref);
   BTSLocalConnectionConfiguration& operator=(IN const BTSLocalConnectionConfiguration& ref);
   bool operator==(IN const BTSLocalConnectionConfiguration& ref) const;
   bool operator!=(IN const BTSLocalConnectionConfiguration& ref) const;
   virtual ~BTSLocalConnectionConfiguration();
   // member with public access
   BTSLocalStackConfiguration configuration; /**< configuration for connection management */
   BTSBDAddressList pairedDevices; /**< list of paired devices */
};

/**
 * Type definition for local telephony configuration.
 */
class BTSLocalTelephonyConfiguration
{
public:
   BTSLocalTelephonyConfiguration();
   BTSLocalTelephonyConfiguration(IN const BTSLocalTelephonyConfiguration& ref);
   BTSLocalTelephonyConfiguration& operator=(IN const BTSLocalTelephonyConfiguration& ref);
   bool operator==(IN const BTSLocalTelephonyConfiguration& ref) const;
   bool operator!=(IN const BTSLocalTelephonyConfiguration& ref) const;
   virtual ~BTSLocalTelephonyConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local phonebook configuration.
 */
class BTSLocalPhonebookConfiguration
{
public:
   BTSLocalPhonebookConfiguration();
   BTSLocalPhonebookConfiguration(IN const BTSLocalPhonebookConfiguration& ref);
   BTSLocalPhonebookConfiguration& operator=(IN const BTSLocalPhonebookConfiguration& ref);
   bool operator==(IN const BTSLocalPhonebookConfiguration& ref) const;
   bool operator!=(IN const BTSLocalPhonebookConfiguration& ref) const;
   virtual ~BTSLocalPhonebookConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local messaging configuration.
 */
class BTSLocalMessagingConfiguration
{
public:
   BTSLocalMessagingConfiguration();
   BTSLocalMessagingConfiguration(IN const BTSLocalMessagingConfiguration& ref);
   BTSLocalMessagingConfiguration& operator=(IN const BTSLocalMessagingConfiguration& ref);
   bool operator==(IN const BTSLocalMessagingConfiguration& ref) const;
   bool operator!=(IN const BTSLocalMessagingConfiguration& ref) const;
   virtual ~BTSLocalMessagingConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local media player configuration.
 */
class BTSLocalMediaPlayerConfiguration
{
public:
   BTSLocalMediaPlayerConfiguration();
   BTSLocalMediaPlayerConfiguration(IN const BTSLocalMediaPlayerConfiguration& ref);
   BTSLocalMediaPlayerConfiguration& operator=(IN const BTSLocalMediaPlayerConfiguration& ref);
   bool operator==(IN const BTSLocalMediaPlayerConfiguration& ref) const;
   bool operator!=(IN const BTSLocalMediaPlayerConfiguration& ref) const;
   virtual ~BTSLocalMediaPlayerConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local wbl configuration.
 */
class BTSLocalWblConfiguration
{
public:
   BTSLocalWblConfiguration();
   BTSLocalWblConfiguration(IN const BTSLocalWblConfiguration& ref);
   BTSLocalWblConfiguration& operator=(IN const BTSLocalWblConfiguration& ref);
   bool operator==(IN const BTSLocalWblConfiguration& ref) const;
   bool operator!=(IN const BTSLocalWblConfiguration& ref) const;
   virtual ~BTSLocalWblConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local ecnr configuration.
 */
class BTSLocalEcnrConfiguration
{
public:
   BTSLocalEcnrConfiguration();
   BTSLocalEcnrConfiguration(IN const BTSLocalEcnrConfiguration& ref);
   BTSLocalEcnrConfiguration& operator=(IN const BTSLocalEcnrConfiguration& ref);
   bool operator==(IN const BTSLocalEcnrConfiguration& ref) const;
   bool operator!=(IN const BTSLocalEcnrConfiguration& ref) const;
   virtual ~BTSLocalEcnrConfiguration();
   // member with public access
   bool dummy; /**< dummy parameter: to be removed after first valid parameter is added */
};

/**
 * Type definition for local configuration container.
 */
class BTSLocalConfigurationContainer
{
public:
   BTSLocalConfigurationContainer();
   BTSLocalConfigurationContainer(IN const BTSLocalConfigurationContainer& ref);
   BTSLocalConfigurationContainer& operator=(IN const BTSLocalConfigurationContainer& ref);
   bool operator==(IN const BTSLocalConfigurationContainer& ref) const;
   bool operator!=(IN const BTSLocalConfigurationContainer& ref) const;
   virtual ~BTSLocalConfigurationContainer();
   // member with public access
   BTSLocalConfigConfiguration configConfiguration; /**< config configuration */
   bool configActive; /**< active flag for config configuration */
   BTSLocalConnectionConfiguration connectionConfiguration; /**< connection configuration */
   bool connectionActive; /**< active flag for connection configuration */
   BTSLocalTelephonyConfiguration telephonyConfiguration; /**< telephony configuration */
   bool telephonyActive; /**< active flag for telephony configuration */
   BTSLocalPhonebookConfiguration phonebookConfiguration; /**< phonebook configuration */
   bool phonebookActive; /**< active flag for phonebook configuration */
   BTSLocalMessagingConfiguration messagingConfiguration; /**< messaging configuration */
   bool messagingActive; /**< active flag for messaging configuration */
   BTSLocalMediaPlayerConfiguration mediaPlayerConfiguration; /**< media player configuration */
   bool mediaPlayerActive; /**< active flag for media player configuration */
   BTSLocalWblConfiguration wblConfiguration; /**< wbl configuration */
   bool wblActive; /**< active flag for wbl configuration */
   BTSLocalEcnrConfiguration ecnrConfiguration; /**< ecnr configuration */
   bool ecnrActive; /**< active flag for ecnr configuration */
};

/**
 * Type definition for local version information.
 */
class BTSLocalVersionInfo
{
public:
   BTSLocalVersionInfo();
   BTSLocalVersionInfo(IN const BTSLocalVersionInfo& ref);
   BTSLocalVersionInfo& operator=(IN const BTSLocalVersionInfo& ref);
   bool operator==(IN const BTSLocalVersionInfo& ref) const;
   bool operator!=(IN const BTSLocalVersionInfo& ref) const;
   virtual ~BTSLocalVersionInfo();
   // member with public access
   BTSInterfaceType stackType; /**< used stack type */
   ::std::string stackVersion; /**< stack version: includes stack app code, stack app version, stack core version */
   BTSSupportedServices supportedServices; /**< enabled services of local Bluetooth adapter */
   ::std::string moduleName; /**< name of HW module */
   BTSBtCoreVersion btVersion; /**< BT version provided by HW module */
   ::std::string fwVersion; /**< firmware version of HW module: includes firmware version and ROM version */
};

/**
 * Type definition for PBAP SDP record.
 */
class BTSPbapSdpRecord
{
public:
   BTSPbapSdpRecord();
   BTSPbapSdpRecord(IN const BTSPbapSdpRecord& ref);
   BTSPbapSdpRecord& operator=(IN const BTSPbapSdpRecord& ref);
   bool operator==(IN const BTSPbapSdpRecord& ref) const;
   bool operator!=(IN const BTSPbapSdpRecord& ref) const;
   virtual ~BTSPbapSdpRecord();
   // member with public access
   BTSProfileVersion version; /**< profile version */
   BTSPbapSupportedRepositories repositories; /**< supported repositories */
   BTSPbapSupportedFeatures features; /**< supported features */
};

/**
 * Type definition for MAP SDP record.
 */
class BTSMapSdpRecord
{
public:
   BTSMapSdpRecord();
   BTSMapSdpRecord(IN const BTSMapSdpRecord& ref);
   BTSMapSdpRecord& operator=(IN const BTSMapSdpRecord& ref);
   bool operator==(IN const BTSMapSdpRecord& ref) const;
   bool operator!=(IN const BTSMapSdpRecord& ref) const;
   virtual ~BTSMapSdpRecord();
   // member with public access
   BTSProfileVersion version; /**< profile version */
   BTSMasInstanceId instanceId; /**< MAS instance ID */
   ::std::string name; /**< service name */
   BTSMapSupportedMessageTypes messageTypes; /**< supported message types */
   BTSMapSupportedFeatures features; /**< supported features */
};

/**
 * Type definition for device capabilities.
 */
class BTSDeviceCapabilities
{
public:
   BTSDeviceCapabilities();
   BTSDeviceCapabilities(IN const BTSDeviceCapabilities& ref);
   BTSDeviceCapabilities& operator=(IN const BTSDeviceCapabilities& ref);
   bool operator==(IN const BTSDeviceCapabilities& ref) const;
   bool operator!=(IN const BTSDeviceCapabilities& ref) const;
   virtual ~BTSDeviceCapabilities();
   // member with public access
   bool inbandRingtoneSupported; /**< flag for inband ringtone support */
};

/**
 * Type definition for SPP service information (local and remote).
 * This structure shall be used by client of BtStackIf layer to configure the local SPP services.
 * The structure shall also provide a mapping of local and remote UUIDs if both scenarios, local and remote SPP connect are used.
 * The element serviceName is used for tracing functionality (debug purpose) and SPP capabilities interface.
 * Table below shows the usage of parameters localUuid and remoteUuid:
 * - localUuid=invalid and remoteUuid=invalid : entry is ignored
 * - localUuid=valid and remoteUuid=invalid : localUuid is used activate a SPP server instance with this UUID, localUuid is used for connect interface e.g. protocol connection status update
 * - localUuid=valid and remoteUuid=valid : localUuid is used activate a SPP server instance with this UUID, remoteUuid is used for connect interface e.g. protocol connection status update
 * - localUuid=invalid and remoteUuid=valid : entry is ignored
 */
class BTSSppServiceInfo
{
public:
   BTSSppServiceInfo();
   BTSSppServiceInfo(IN const BTSSppServiceInfo& ref);
   BTSSppServiceInfo& operator=(IN const BTSSppServiceInfo& ref);
   bool operator==(IN const BTSSppServiceInfo& ref) const;
   bool operator!=(IN const BTSSppServiceInfo& ref) const;
   virtual ~BTSSppServiceInfo();
   // member with public access
   ::std::string serviceName; /**< SPP service name */
   BTSUuid localUuid; /**< local UUID */
   BTSUuid remoteUuid; /**< remote UUID */
};

/**
 * Type definition for component entry.
 */
class BTSComponentEntry
{
public:
   BTSComponentEntry();
   BTSComponentEntry(IN const BTSFunctionBlock inputComponent, IN const BTSInterfaceType inputStackInterface, IN const BTSFunctionBlock inputSubComponent);
   BTSComponentEntry(IN const BTSComponentEntry& ref);
   BTSComponentEntry& operator=(IN const BTSComponentEntry& ref);
   bool operator==(IN const BTSComponentEntry& ref) const;
   bool operator!=(IN const BTSComponentEntry& ref) const;
   bool operator<(const BTSComponentEntry& ref) const;
   bool operator>(const BTSComponentEntry& ref) const;
   virtual ~BTSComponentEntry();
   // member with public access
   BTSFunctionBlock component; /**< component */
   BTSInterfaceType stackInterface; /**< stack interface */
   BTSFunctionBlock subComponent; /**< sub component */

protected:
   int compare(const BTSComponentEntry& ref) const;
};

/**
 * Type definition for WBL WiFi setup status.
 */
class BTSWblWiFiSetupStatus
{
public:
   BTSWblWiFiSetupStatus();
   BTSWblWiFiSetupStatus(IN const BTSWblWiFiSetupStatus& ref);
   BTSWblWiFiSetupStatus& operator=(IN const BTSWblWiFiSetupStatus& ref);
   bool operator==(IN const BTSWblWiFiSetupStatus& ref) const;
   bool operator!=(IN const BTSWblWiFiSetupStatus& ref) const;
   virtual ~BTSWblWiFiSetupStatus();
   void reset(void);
   // member with public access
   BTSWblWifiMode mode; /**< mode */
   BTSWblWifiPowerState powerState; /**< power state */
   BTSWblWifiFrequency frequency; /**< frequency */
   BTSWblAccessPointConfigType type; /**< access point configuration type */
};

/**
 * Type definition for WBL restriction.
 */
class BTSWblRestriction
{
public:
   BTSWblRestriction();
   BTSWblRestriction(IN const BTSWblRestriction& ref);
   BTSWblRestriction& operator=(IN const BTSWblRestriction& ref);
   bool operator==(IN const BTSWblRestriction& ref) const;
   bool operator!=(IN const BTSWblRestriction& ref) const;
   virtual ~BTSWblRestriction();
   // member with public access
   BTSWblAccessPointConfigType type; /**< access point configuration type */
   BTSWblRestrictionState state; /**< restriction type */
};

/**
 * Type definition for WBL supported technology.
 */
class BTSWblSupportedTechnology
{
public:
   BTSWblSupportedTechnology();
   BTSWblSupportedTechnology(IN const BTSWblSupportedTechnology& ref);
   BTSWblSupportedTechnology& operator=(IN const BTSWblSupportedTechnology& ref);
   bool operator==(IN const BTSWblSupportedTechnology& ref) const;
   bool operator!=(IN const BTSWblSupportedTechnology& ref) const;
   virtual ~BTSWblSupportedTechnology();
   // member with public access
   BTSWblAccessPointConfigType type; /**< access point configuration type */
   BTSWblSupportedSetting setting; /**< supported settings */
};

/**
 * Type definition for WBL supported restriction (supported frequencies).
 */
class BTSWblSupportedRestriction
{
public:
   BTSWblSupportedRestriction();
   BTSWblSupportedRestriction(IN const BTSWblSupportedRestriction& ref);
   BTSWblSupportedRestriction& operator=(IN const BTSWblSupportedRestriction& ref);
   bool operator==(IN const BTSWblSupportedRestriction& ref) const;
   bool operator!=(IN const BTSWblSupportedRestriction& ref) const;
   virtual ~BTSWblSupportedRestriction();
   BTSWblWifiFrequency frequency; /**< frequency */
   BTSWblSupportedTechnologyList technologies; /**< supported technologies */
};

/**
 * Type definition for WBL supported restrictions (supported frequencies) entry.
 */
class BTSWblSupportedRestrictionEntry
{
public:
   BTSWblSupportedRestrictionEntry();
   BTSWblSupportedRestrictionEntry(IN const BTSWblSupportedRestrictionEntry& ref);
   BTSWblSupportedRestrictionEntry& operator=(IN const BTSWblSupportedRestrictionEntry& ref);
   bool operator==(IN const BTSWblSupportedRestrictionEntry& ref) const;
   bool operator!=(IN const BTSWblSupportedRestrictionEntry& ref) const;
   virtual ~BTSWblSupportedRestrictionEntry();
   // member with public access
   BTSWblWiFiSetupId id; /**< id */
   BTSWblSupportedRestrictionList supportedRestrictions; /**< supported restrictions (supported frequencies) */
};

/**
 * Type definition for WBL WiFi setup status entry.
 */
class BTSWblWiFiSetupStatusEntry
{
public:
   BTSWblWiFiSetupStatusEntry();
   BTSWblWiFiSetupStatusEntry(IN const BTSWblWiFiSetupStatusEntry& ref);
   BTSWblWiFiSetupStatusEntry& operator=(IN const BTSWblWiFiSetupStatusEntry& ref);
   bool operator==(IN const BTSWblWiFiSetupStatusEntry& ref) const;
   bool operator!=(IN const BTSWblWiFiSetupStatusEntry& ref) const;
   virtual ~BTSWblWiFiSetupStatusEntry();
   // member with public access
   BTSWblWiFiSetupId id; /**< id */
   BTSWblWiFiSetupStatus status; /**< status */
};

/**
 * Type definition for WBL last intended mode.
 */
class BTSWblLastIntendedMode
{
public:
   BTSWblLastIntendedMode();
   BTSWblLastIntendedMode(IN const BTSWblLastIntendedMode& ref);
   BTSWblLastIntendedMode& operator=(IN const BTSWblLastIntendedMode& ref);
   bool operator==(IN const BTSWblLastIntendedMode& ref) const;
   bool operator!=(IN const BTSWblLastIntendedMode& ref) const;
   virtual ~BTSWblLastIntendedMode();
   // member with public access
   BTSWblWiFiSetupId id; /**< id */
   BTSWblWiFiSetupStatus status; /**< status */
};

/**
 * Check for valid stack interface type.
 *
 * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
 *
 * @return = true: valid stack interface type,
 * @return = false: invalid stack interface type
 */
bool isValidInterfaceType(IN const BTSInterfaceType stackInterface);

/**
 * Check for valid component type.
 *
 * @param[in] component: component
 *
 * @return = true: valid component type,
 * @return = false: invalid component type
 */
bool isValidComponentType(IN const BTSFunctionBlock component);

} //btstackif

#endif //_BT_STACK_IF_TYPES_H_
