/**
 * @file GenericStackIf.cpp
 *
 * @par SW-Component
 * Interface
 *
 * @brief Generic stack interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of generic stack interface.
 * This file contains only the general implementation.
 */

#include "GenericStackIf.h"
#include "App2Bts_BaseMessage.h"
#include "Bts2App_BaseMessage.h"
#include "Bts2Ipc_BaseMessage.h"
#include "Ipc2Bts_BaseMessage.h"
#include "InstanceFactory.h"
#include "IMainControl.h"
#include "IBasicControl.h"
#include "IDbusBase.h"
#include "BtsUtils.h"
#include "BtsTimerBase.h"
#include "BtsTimerPool.h"
#include "TraceBase.h"
#include "FwTestCommandServer.h"
#include "FwFormattedDataPrint.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/GenericStackIf.cpp.trc.h"
#endif
#endif

#include "cc_dbus_if/CcDbusIfControllerFactory.h"
#include "cc_dbus_if/CcDbusIfLaunchType.h"
#include "cc_dbus_if/ICcDbusIfController.h"

namespace btstackif {

GenericStackIf::GenericStackIf() :
_userList(),
_lock(),
_mainControl(0),
_controlIfList(),
_dbusIfList(),
_intInitDone(false),
_intInitSuccess(false),
_extInitDone(false),
_extInitSuccess(false),
_ifSetupDoneList(),
_ifSetupSuccessList(),
_userMode(BTS_USER_MODE_CONNECTION), // default value, not used for simulation
_asfConfigFile(),
_dbusController(0)
{
   TimeInfo timeInfo(this);
   ETG_TRACE_USR1((" GenericStackIf(): %s", timeInfo.getInfo()));

   // call after member default initialization
   initGenericStackIf();
}

GenericStackIf::GenericStackIf(IN const BTSUserMode userMode, IN const ::std::string& file) :
_userList(),
_lock(),
_mainControl(0),
_controlIfList(),
_dbusIfList(),
_intInitDone(false),
_intInitSuccess(false),
_extInitDone(false),
_extInitSuccess(false),
_ifSetupDoneList(),
_ifSetupSuccessList(),
_userMode(userMode),
_asfConfigFile(file),
_dbusController(0)
{
   TimeInfo timeInfo(this);
   ETG_TRACE_USR1((" GenericStackIf(): %s", timeInfo.getInfo()));

   // call after member default initialization
   initGenericStackIf();
}

GenericStackIf::~GenericStackIf()
{
   // call setter
   for(::std::map< BTSComponentEntry, IDbusBase* >::iterator it = _dbusIfList.begin(); it != _dbusIfList.end(); ++it)
   {
      if(it->second)
      {
         it->second->setMainControl(0);
      }
   }

   for(::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _controlIfList.begin(); it != _controlIfList.end(); ++it)
   {
      if(it->second)
      {
         it->second->setMainControl(0);
      }
   }

   if(_mainControl)
   {
      _mainControl->resetControlAndDbusIf();
      _mainControl->resetGenericStackIf();
   }

   // destroy instances
   for(::std::map< BTSComponentEntry, IDbusBase* >::iterator it = _dbusIfList.begin(); it != _dbusIfList.end(); ++it)
   {
      if(it->second)
      {
         InstanceFactory::getInstance().destroyDbusAccessInstance(it->second);
      }
   }

   for(::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _controlIfList.begin(); it != _controlIfList.end(); ++it)
   {
      if(it->second)
      {
         InstanceFactory::getInstance().destroyBasicControlInstance(it->second);
      }
   }

   InstanceFactory::getInstance().destroyMainControlInstance();
   _mainControl = 0;

   _dbusController = 0;

   TimerPool::getInstance().destroy();
}

void GenericStackIf::printStatistics(void)
{
   ETG_TRACE_USR1((" printStatistics(): start"));
   ETG_TRACE_USR1((" ***App2BtsMsg=%u", App2Bts_BaseMessage::getApp2BtsMessageCount()));
   ETG_TRACE_USR1((" ***Bts2AppMsg=%u", Bts2App_BaseMessage::getBts2AppMessageCount()));
   ETG_TRACE_USR1((" ***Bts2IpcMsg=%u", Bts2Ipc_BaseMessage::getBts2IpcMessageCount()));
   ETG_TRACE_USR1((" ***Ipc2BtsMsg=%u", Ipc2Bts_BaseMessage::getIpc2BtsMessageCount()));

   if(_mainControl)
   {
      _mainControl->printQueueStatistics();
   }

   for(::std::map< BTSComponentEntry, IBasicControl* >::iterator it = _controlIfList.begin(); it != _controlIfList.end(); ++it)
   {
      if(it->second)
      {
         it->second->printQueueStatistics();
      }
   }

   for(::std::map< BTSComponentEntry, IDbusBase* >::iterator it = _dbusIfList.begin(); it != _dbusIfList.end(); ++it)
   {
      if(it->second)
      {
         it->second->printQueueStatistics();
      }
   }

   unsigned int nmbTimer(0);
   unsigned int nmbUsedTimer(0);
   unsigned int nmbExtendedTimer(0);
   unsigned int nmbUsedExtendedTimer(0);

   TimerPool::getInstance().getTimerCount(nmbTimer, nmbUsedTimer, nmbExtendedTimer, nmbUsedExtendedTimer);
   ETG_TRACE_USR1((" ***nmbTimer=%u", nmbTimer));
   ETG_TRACE_USR1((" ***nmbUsedTimer=%u", nmbUsedTimer));
   ETG_TRACE_USR1((" ***nmbExtendedTimer=%u", nmbExtendedTimer));
   ETG_TRACE_USR1((" ***nmbUsedExtendedTimer=%u", nmbUsedExtendedTimer));

   ETG_TRACE_USR1((" printStatistics(): end"));
}

BTSErrorCode GenericStackIf::init(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);

   const BTSComponentEntry entry(BTS_FB_NONE, stackInterface, subComponent);

   ETG_TRACE_USR1((" init(): _intInitDone=%d _intInitSuccess=%d _extInitDone=%d _extInitSuccess=%d _ifSetupDone=%d pairedDevices.size()=%u", _intInitDone, _intInitSuccess, _extInitDone, _extInitSuccess, _ifSetupSuccessList[entry], configuration.connectionConfiguration.pairedDevices.size()));

   if((true == _intInitDone) && (true == _intInitSuccess))
   {
      if((true == _extInitDone) && (true == _extInitSuccess))
      {
         // already done
      }
      else
      {
         // initialize and start all common stuff:
         // start CcDbusIf interface (independent of used stack interfaces) => check user mode only
         // note: in case of support for parallel stack interface usage is needed then ASF application (cma) shall be checked/updated (best case: all interfaces are defined as "wired at runtime")
         _dbusController = ::ccdbusif::CcDbusIfControllerFactory::getInstance().getControllerInstance();
         if(0 != _dbusController)
         {
            ::ccdbusif::LaunchType launchType = ::ccdbusif::PROXIES_ONLY; // default

            if((BTS_USER_MODE_CONNECTION == _userMode) ||
               (BTS_USER_MODE_ALL_WI_AGENT == _userMode))
            {
               // agent functionality needed
               // if(BTS_IF_ALPS_EVOLUTION_GENIVI == _interfaceType)
               {
                  launchType = ::ccdbusif::WITH_EVOLUTION_AGENT;
               }
            }

            _dbusController->start();
            _dbusController->startDbusLauncher(launchType, _asfConfigFile);
         }
         else
         {
            // should never happen
            FW_NORMAL_ASSERT_ALWAYS();
         }

         // start main control worker thread
         if(_mainControl)
         {
            _mainControl->run();
         }

         // register test command callback
         ::fw::TestCommandServer::getInstance().create(94); // #define TR_TTFIS_CONNECTIVITY     ((TR_tenTraceChan)94) TODO: shall be done by CCA layer
         ::fw::TestCommandServer::getInstance().registerClient(this);

         _extInitDone = true;
         _extInitSuccess = true;
      }

      if(_mainControl)
      {
         _mainControl->triggerRequestCurrentTime();
      }

      if(true == _ifSetupDoneList[entry])
      {
         // already done => trigger initialized callback
         IBasicControl* control(getControlHandler(component, stackInterface, subComponent));
         if(control)
         {
            control->triggerInitializedCallback();
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         if(true == _ifSetupSuccessList[entry])
         {
            return BTS_OK;
         }
         else
         {
            return BTS_ERROR;
         }
      }
      else
      {
         // initialize and start all specific stuff:
         BTSErrorCode errCode(BTS_ERROR);
         ::std::vector< BTSDbusInterfaceItem > dbusInterfaceList;
         dbusInterfaceList.reserve(50);

         // get stack configuration
         IBasicControl* control(getControlHandler(component, stackInterface, subComponent));
         if(control)
         {
            control->setStackConfiguration(component, stackInterface, subComponent, _userMode, dbusInterfaceList, configuration);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         // start DBUS handling worker thread
         IDbusBase* dbusIf(getDbusIf(component, stackInterface, subComponent));
         if(dbusIf)
         {
            // set DBUS controller interface
            errCode = dbusIf->setCcDbusIfControllerIf(component, stackInterface, subComponent, _userMode, _dbusController, dbusInterfaceList, configuration);

            // start DBUS handling worker thread
            dbusIf->run();

            // enable timer handling
            dbusIf->setTimerHandling(true);
         }
         else
         {
            FW_NORMAL_ASSERT_ALWAYS();
         }

         _ifSetupDoneList[entry] = true;

         if(BTS_OK == errCode)
         {
            // initialized callback is triggered after initialization is finished
            _ifSetupSuccessList[entry] = true;
            return BTS_OK;
         }
         else
         {
            _ifSetupSuccessList[entry] = false;
         }
      }
   }

   // no initialized callback needed
   return BTS_ERROR;
}

BTSErrorCode GenericStackIf::deInit(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const bool force /*= false*/)
{
   const unsigned int nmb(getNumberRegisteredUsers());

   ETG_TRACE_USR1((" deInit(): force=%d nmb users=%u _extInitDone=%d _extInitSuccess=%d", force, nmb, _extInitDone, _extInitSuccess));

   if(false == force) // TODO: to be tested
   {
      // do specific stuff each time:

      IDbusBase* dbusIf(getDbusIf(component, stackInterface, subComponent));

      // disable timer handling
      if(dbusIf)
      {
         dbusIf->setTimerHandling(false);
      }

      // stop DBUS handling worker thread
      if(dbusIf)
      {
         dbusIf->stop();
      }

      // reset CcDbusIf controller interface
      if(dbusIf)
      {
         dbusIf->resetCcDbusIfControllerIf();
      }
   }
   else
   {
      // do common stuff only once:
      // stop main control worker thread
      if(_mainControl)
      {
         _mainControl->stop();
      }

      // never reset _intInitDone
      // never reset _intInitSuccess

      _extInitDone = false;
      _extInitSuccess = false;

      // stop CcDbusIf interface
      if(0 != _dbusController)
      {
         _dbusController->prepareStop();
         _dbusController->stopDbusLauncher();
         _dbusController->stop();
         ccdbusif::CcDbusIfControllerFactory::getInstance().destroyControllerInstance();
         _dbusController = 0;
      }

      // deregister test command callback
      ::fw::TestCommandServer::getInstance().deregisterClient(this);
      ::fw::TestCommandServer::getInstance().destroy(); // TODO: shall be done by CCA layer
   }

   // return always OK
   return BTS_OK;
}

void GenericStackIf::createEnvironment(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   internalCreateEnvironment(component, stackInterface, subComponent);
}

void GenericStackIf::registerUser(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user)
{
   /*
    * note: limitation of sub component instances per stack interface is done outside
    * note: we have to remember the assignment of user vs. stack interface + sub component
    * note: component information is for tracing only
    * note: sub component information is part of message opcode
    */

   (void)(component);

   if(user)
   {
      LockAccess();

      ::fw::FormattedOutputPtr userPtr((uintptr_t)user);
      ETG_TRACE_USR1((" registerUser(): component=%d stackInterface=%d subComponent=%d: user=%s", component, stackInterface, subComponent, userPtr.c_str()));

      if(_userList.end() != _userList.find(user))
      {
         // already registered
         FW_NORMAL_ASSERT_ALWAYS();
      }
      else
      {
         BTSUserCallbackEntry entry(0, stackInterface, subComponent); // callback will be set later
         _userList[user] = entry;
      }

      ETG_TRACE_USR1((" registerUser(): component=%d stackInterface=%d subComponent=%d: _userList.size=%u", component, stackInterface, subComponent, _userList.size()));

      UnlockAccess();
   }
}

void GenericStackIf::unregisterUser(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);

   if(user)
   {
      LockAccess();

      ::fw::FormattedOutputPtr userPtr((uintptr_t)user);
      ETG_TRACE_USR1((" unregisterUser(): component=%d stackInterface=%d subComponent=%d: user=%s", component, stackInterface, subComponent, userPtr.c_str()));

      ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >::iterator it = _userList.find(user);
      if(_userList.end() != it)
      {
         _userList.erase(it);
      }
      else
      {
         // already erased
         FW_NORMAL_ASSERT_ALWAYS();
      }

      ETG_TRACE_USR1((" unregisterUser(): component=%d stackInterface=%d subComponent=%d: _userList.size=%u", component, stackInterface, subComponent, _userList.size()));

      UnlockAccess();
   }
}

void GenericStackIf::registerCallback(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user, IN BtStackIfCallback* callback)
{
   (void)(component);
   (void)(subComponent);

   if((user) && (callback))
   {
      LockAccess();

      ::fw::FormattedOutputPtr userPtr((uintptr_t)user);
      ::fw::FormattedOutputPtr callbackPtr((uintptr_t)callback);
      ETG_TRACE_USR1((" registerCallback(): component=%d stackInterface=%d subComponent=%d: user=%20s callback=%s", component, stackInterface, subComponent, userPtr.c_str(), callbackPtr.c_str()));

      ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >::iterator it = _userList.find(user);
      if(_userList.end() != it)
      {
         if(it->second.callback)
         {
            // callback already registered
            FW_NORMAL_ASSERT_ALWAYS();
         }
         else
         {
            it->second.callback = callback;
            FW_NORMAL_ASSERT(stackInterface == it->second.stackInterface);
            FW_NORMAL_ASSERT(subComponent == it->second.subComponent);
         }
      }
      else
      {
         // user not registered
         FW_NORMAL_ASSERT_ALWAYS();
      }

      UnlockAccess();
   }
}

unsigned int GenericStackIf::getNumberRegisteredUsers(void)
{
   unsigned int size;

   LockAccess();

   size = (unsigned int)_userList.size();

   UnlockAccess();

   return size;
}

void GenericStackIf::processApp2BtsMsg(IN App2Bts_BaseMessage* message)
{
   FW_IF_NULL_PTR_RETURN(message);

   // do trace
   message->doInputTrace();

   if(_mainControl)
   {
      // if(BTS_OK == isInitSuccess(message->getComponent(), message->getStackInterface(), extractSubComponentFromApp2BtsOpCode(message->getOpCode())))
      if(BTS_OK == isInitSuccess(message->getComponent(), message->getStackInterface(), message->getFunctionBlock()))
      {
         // push to queue
         _mainControl->pushApp2BtsMessage(message);
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
         delete message;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
      delete message;
   }
}

void GenericStackIf::setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData)
{
   IDbusBase* dbusIf(getDbusIf(component, stackInterface, subComponent));
   FW_IF_NULL_PTR_RETURN(dbusIf);
   dbusIf->setSimulationTestCommand(testCommand, testData);
}

void GenericStackIf::setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned char* testData)
{
   IDbusBase* dbusIf(getDbusIf(component, stackInterface, subComponent));
   FW_IF_NULL_PTR_RETURN(dbusIf);
   dbusIf->setSimulationTestCommand(testCommand, testData);
}

void GenericStackIf::setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   IDbusBase* dbusIf(getDbusIf(component, stackInterface, subComponent));
   FW_IF_NULL_PTR_RETURN(dbusIf);
   dbusIf->setSimulationTestCommand(testCommand, testData);
}

void GenericStackIf::setMainControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);

   if(_mainControl)
   {
      _mainControl->setMainControlTestCommand(testCommand, testData);
   }
}

void GenericStackIf::setConnectionControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData)
{
   (void)(subComponent);
   IBasicControl* control(getControlHandler(component, stackInterface, BTS_FB_CONNECTION));
   FW_IF_NULL_PTR_RETURN(control);
   control->setSubControlTestCommand(testCommand, testData);
}

void GenericStackIf::setConnectionControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned char* testData)
{
   (void)(subComponent);
   IBasicControl* control(getControlHandler(component, stackInterface, BTS_FB_CONNECTION));
   FW_IF_NULL_PTR_RETURN(control);
   control->setSubControlTestCommand(testCommand, testData);
}

void GenericStackIf::handleTestCommand(const unsigned char* data)
{
   if(0 == data)
   {
      return;
   }

   // data[0]: ignore
   // data[1]: component id (CCA layer, BM Core, BtStackIf, ...); use same ranges as used for tracing; check https://hi-dms.de.bosch.com/docushare/dsweb/ServicesLib/Document-770581/History
   // data[2]: internal test command group
   // data[3]: command itself

   const unsigned char start = (unsigned char)(BTS_OFFSET + BTS_RANGE_START);
   const unsigned char end = (unsigned char)(BTS_OFFSET + BTS_RANGE_END);

   if((start <= data[1]) && (data[1] <= end))
   {
      // valid for BtStackIf
      switch(data[2])
      {
         case (unsigned char)BTS_TEST_COMMAND_GROUP_GENERIC:
            processTestCommand(&data[3]);
            break;
         case (unsigned char)BTS_TEST_COMMAND_GROUP_MAIN:
            // add if necessary
            break;
         case (unsigned char)BTS_TEST_COMMAND_GROUP_CONNECTION:
            // note: if needed extend to pass to all connection related control handler, maybe TTFis commands need to be extended
            setConnectionControlTestCommand(BTS_FB_CONNECTION, BTS_IF_ALPS_EVOLUTION_GENIVI, BTS_FB_CONNECTION, BTS_TTFIS_COMMAND, &data[3]);
            break;
         default:
            break;
      }
   }
}

void GenericStackIf::initGenericStackIf(void)
{
   if(true == _intInitDone)
   {
      // already done
      return;
   }

   // initialize and create basic stuff e.g. timer pool
   TimerBase::init();
   TimerPool::getInstance().create();

   // create main control instance (exact 1 instance)
   _mainControl = InstanceFactory::getInstance().getMainControlInstance();
   FW_NORMAL_ASSERT(0 != _mainControl);

   // call setter
   if(_mainControl)
   {
      _mainControl->setGenericStackIf(this);
   }

   // check if all handlers were created successfully
   if(_mainControl)
   {
      _intInitSuccess = true;
   }

   _intInitDone = true;
}

void GenericStackIf::internalCreateEnvironment(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   FW_IF_NULL_PTR_RETURN(_mainControl);

   // create stack interface dependent control handler
   IBasicControl* controlInstance = InstanceFactory::getInstance().getBasicControlInstance(stackInterface, subComponent);
   FW_IF_NULL_PTR_RETURN(controlInstance);

   // add only if not in list
   BTSComponentEntry controlEntry;
   controlEntry = getControlComponentKey(controlEntry, component, stackInterface, subComponent);
   if(_controlIfList.end() == _controlIfList.find(controlEntry))
   {
      _controlIfList[controlEntry] = controlInstance;

      // call setter
      controlInstance->setMainControl(_mainControl);
      controlInstance->setComponent(component);
      controlInstance->setStackInterface(stackInterface);
   }

   // create stack interface dependent DBUS interface
   IDbusBase* dbusInstance = InstanceFactory::getInstance().getDbusAccessInstance(stackInterface, subComponent);
   FW_IF_NULL_PTR_RETURN(dbusInstance);

   // add only if not in list
   BTSComponentEntry dbusEntry;
   dbusEntry = getDbusComponentKey(dbusEntry, component, stackInterface, subComponent);
   if(_dbusIfList.end() == _dbusIfList.find(dbusEntry))
   {
      _dbusIfList[dbusEntry] = dbusInstance;

      // call setter
      dbusInstance->setMainControl(_mainControl);
      dbusInstance->setComponent(component);
      dbusInstance->setStackInterface(stackInterface);
   }

   // call setter
   _mainControl->setControlAndDbusIf(component, stackInterface, subComponent, controlInstance, dbusInstance);
}

BTSErrorCode GenericStackIf::isInitSuccess(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   (void)(component);

   const BTSComponentEntry entry(BTS_FB_NONE, stackInterface, subComponent);

   if((true == _intInitDone) && (true == _intInitSuccess) && (true == _extInitDone) && (true == _extInitSuccess) && (true == _ifSetupDoneList[entry]) && (true == _ifSetupSuccessList[entry]))
   {
      return BTS_OK;
   }
   else
   {
      return BTS_ERROR;
   }
}

IBasicControl* GenericStackIf::getControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const
{
   IBasicControl* returnIf(0);

   BTSComponentEntry controlEntry;
   controlEntry = getControlComponentKey(controlEntry, component, stackInterface, subComponent);
   ::std::map< BTSComponentEntry, IBasicControl* >::const_iterator it = _controlIfList.find(controlEntry);
   if(_controlIfList.end() != it)
   {
      returnIf = it->second;
   }

   return returnIf;
}

IDbusBase* GenericStackIf::getDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const
{
   IDbusBase* returnIf(0);

   BTSComponentEntry dbusEntry;
   dbusEntry = getDbusComponentKey(dbusEntry, component, stackInterface, subComponent);
   ::std::map< BTSComponentEntry, IDbusBase* >::const_iterator it = _dbusIfList.find(dbusEntry);
   if(_dbusIfList.end() != it)
   {
      returnIf = it->second;
   }

   return returnIf;
}

void GenericStackIf::processTestCommand(IN const unsigned char* testData)
{
   if(0 == testData)
   {
      return;
   }

   ETG_TRACE_USR1((" TTFis command (GENERIC): opcode=0x%02X", testData[0]));

   switch(testData[0])
   {
      case 0x00:
         printStatistics();
         break;
      case 0x01:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
      case 0x02:
         FW_FATAL_ASSERT_ALWAYS();
         break;
      default:
         break;
   }
}

} //btstackif
