/**
 * @file IGenericStackIf.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Generic stack interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for generic stack interface.
 */

#ifndef _I_GENERIC_STACK_IF_H_
#define _I_GENERIC_STACK_IF_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class BtStackIfBaseRequest;
class BtStackIfCallback;
class Ipc2Bts_BaseMessage;

/**
 *
 */
class IGenericStackIf
{
public:
   virtual ~IGenericStackIf() {}

   /**
    * Print statistics.
    */
   virtual void printStatistics(void) = 0;

   /**
    * Init the BT stack interface layer.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] configuration: local configuration data (container)
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode init(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSLocalConfigurationContainer& configuration) = 0;

   /**
    * DeInit the BT stack interface layer.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] force: force flag
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode deInit(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const bool force = false) = 0;

   /**
    * Create environment needed for given stack and interface and sub component.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    */
   virtual void createEnvironment(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) = 0;

   /**
    * Register user.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] user: user
    */
   virtual void registerUser(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user) = 0;

   /**
    * Unregister user.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] user: user
    */
   virtual void unregisterUser(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user) = 0;

   /**
    * Register callback.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] user: user
    * @param[in] callback: callback
    */
   virtual void registerCallback(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user, IN BtStackIfCallback* callback) = 0;

   /**
    * Get number of registered users.
    *
    * @return = number of registered users
    */
   virtual unsigned int getNumberRegisteredUsers(void) = 0;

   /**
    * Get list of registered users.
    *
    * @return = list of registered users
    */
   virtual ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >& getUserList(void) = 0;

   /**
    * Lock access to user list.
    */
   virtual void LockAccess(void) = 0;

   /**
    * Unlock access to user list.
    */
   virtual void UnlockAccess(void) = 0;

   /**
    * Process App2Bts message.
    *
    * @param[in] message: App2Bts message
    */
   virtual void processApp2BtsMsg(IN App2Bts_BaseMessage* message) = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned char* testData) = 0;

   /**
    * Sets a simulation test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData) = 0;

   /**
    * Sets a main control test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setMainControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Sets a connection control test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData) = 0;

   /**
    * Sets a connection control test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned char* testData) = 0;
};

} //btstackif

#endif //_I_GENERIC_STACK_IF_H_
