/**
 * @file GenericStackIf.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief Generic stack interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Class for generic stack interface.
 */

#ifndef _GENERIC_STACK_IF_H_
#define _GENERIC_STACK_IF_H_

#include "IGenericStackIf.h"
#include "FwITestCommandClient.h"
#include "FwLock.h"

// class forward declarations
namespace ccdbusif {
class ICcDbusIfController;
} //ccdbusif

namespace btstackif {

// class forward declarations
class BtStackIfBaseRequest;
class BtStackIfCallback;
class App2Bts_BaseMessage;
class IMainControl;
class IBasicControl;
class IDbusBase;

/**
 *
 */
class GenericStackIf : public IGenericStackIf, public ::fw::ITestCommandClient
{
protected:
   /**************************************************************************
    Constructor / destructor
    **************************************************************************/
   /**
    * Default constructor. Do not use!!!
    */
   GenericStackIf();

public:
   /**
    * Constructor.
    *
    * @param[in] userMode: stack user mode to be used
    * @param[in] file: configuration file
    */
   GenericStackIf(IN const BTSUserMode userMode, IN const ::std::string& file);

   /**
    * Destructor.
    */
   virtual ~GenericStackIf();

   /**
    * Print statistics.
    */
   virtual void printStatistics(void);

   /**
    * Init the BT stack interface layer.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] configuration: local configuration data (container)
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode init(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSLocalConfigurationContainer& configuration);

   /**
    * DeInit the BT stack interface layer.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] force: force flag
    *
    * @return != BTS_OK: error,
    * @return  = BTS_OK: OK
    */
   virtual BTSErrorCode deInit(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const bool force = false);

   /**
    * Create environment needed for given stack and interface and sub component.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    */
   virtual void createEnvironment(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

   /**
    * Register user.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] user: user
    */
   virtual void registerUser(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user);

   /**
    * Unregister user.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] user: user
    */
   virtual void unregisterUser(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user);

   /**
    * Register callback.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] user: user
    * @param[in] callback: callback
    */
   virtual void registerCallback(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BtStackIfBaseRequest* user, IN BtStackIfCallback* callback);

   /**
    * Get number of registered users.
    *
    * @return = number of registered users
    */
   virtual unsigned int getNumberRegisteredUsers(void);

   /**
    * Get list of registered users.
    *
    * @return = list of registered users
    */
   virtual ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >& getUserList(void);

   /**
    * Lock access to user list.
    */
   virtual void LockAccess(void);

   /**
    * Unlock access to user list.
    */
   virtual void UnlockAccess(void);

   /**
    * Process App2Bts message.
    *
    * @param[in] message: App2Bts message
    */
   virtual void processApp2BtsMsg(IN App2Bts_BaseMessage* message);

   /**
    * Sets a simulation test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData);

   /**
    * Sets a simulation test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Sets a simulation test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setSimulationTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData);

   /**
    * Sets a main control test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setMainControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData);

   /**
    * Sets a connection control test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned int testData);

   /**
    * Sets a connection control test command.
    *
    * @param[in] component: component
    * @param[in] stackInterface: stack interface
    * @param[in] subComponent: sub component
    * @param[in] testCommand: test command as string
    * @param[in] testData: test data
    */
   virtual void setConnectionControlTestCommand(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const char* testCommand, IN const unsigned char* testData);

   /**
    * Handle received test command.
    *
    * @param[in] data: test command
    */
   virtual void handleTestCommand(const unsigned char* data);

private:
   ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry > _userList;
   LockForeverAndNonReentrant _lock;
   IMainControl* _mainControl;
   ::std::map< BTSComponentEntry, IBasicControl* > _controlIfList;
   ::std::map< BTSComponentEntry, IDbusBase* > _dbusIfList;
   bool _intInitDone; // internal initialization flag
   bool _intInitSuccess; // internal initialization flag
   bool _extInitDone; // external initialization flag
   bool _extInitSuccess; // external initialization flag
   ::std::map< BTSComponentEntry, bool > _ifSetupDoneList;
   ::std::map< BTSComponentEntry, bool > _ifSetupSuccessList;
   BTSUserMode _userMode; /**< stack user mode */
   ::std::string _asfConfigFile; /**< ASF log configuration file */
   ::ccdbusif::ICcDbusIfController* _dbusController; /**< CcDbusIf controller */

   void initGenericStackIf(void);
   void internalCreateEnvironment(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);
   BTSErrorCode isInitSuccess(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);
   IBasicControl* getControlHandler(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const;
   IDbusBase* getDbusIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent) const;
   void processTestCommand(IN const unsigned char* testData);
};

/*
 * inline methods
 */
inline ::std::map< const BtStackIfBaseRequest*, BTSUserCallbackEntry >& GenericStackIf::getUserList(void) { return _userList; }
inline void GenericStackIf::LockAccess(void) { _lock.lock(); }
inline void GenericStackIf::UnlockAccess(void) { _lock.unlock(); }

} //btstackif

#endif //_GENERIC_STACK_IF_H_
