/**
 * @file DbusBase.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS basic handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of basic DBUS handling.
 */

#include "DbusBase.h"
#include "Bts2Ipc_BaseMessage.h"
#include "Ipc2Bts_BaseMessage.h"
#include "IMainControl.h"
#include "IDbusIfHandler.h"
#include "DbusTimeoutControl.h"
#include "BtsTimer.h"
#include "BtsTimerPool.h"
#include "TraceClasses.h"
#include "FwSingleThread.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/DbusBase.cpp.trc.h"
#endif
#endif

namespace btstackif {

DbusBase::DbusBase() :
_bts2IpcWaitingQueue(),
_bts2IpcWorkingQueue(),
_ipc2BtsCallbackQueue(),
_eventQueue(),
_terminateWorkerThread(false),
_workerThreadTerminated(true),
_singleThread(0),
_mainControl(0),
_dbusIfAvailable(false),
_timerMaster(),
_configurationExtensions(),
_connectionExtensions(),
_phonecallExtensions(),
_phonebookExtensions(),
_messagingExtensions(),
_mediaPlayerExtensions(),
_extensions(),
_component(BTS_FB_LAST),
_stackInterface(BTS_IF_LAST),
_timerHandlingEnabled(false),
_timeoutControlList(),
_controllerClient(0),
_responseTimeoutHandler(0)
{
   _bts2IpcWaitingQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _bts2IpcWorkingQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _ipc2BtsCallbackQueue.setWarningSize(NMB_MESSAGES_WARNING_THRESHOLD);
   _timerMaster.createNewContext();
   _timerMaster.requestCurrentTime();
}

DbusBase::DbusBase(const DbusBase& ref) :
_bts2IpcWaitingQueue(),
_bts2IpcWorkingQueue(),
_ipc2BtsCallbackQueue(),
_eventQueue(),
_terminateWorkerThread(false),
_workerThreadTerminated(true),
_singleThread(0),
_mainControl(0),
_dbusIfAvailable(false),
_timerMaster(),
_configurationExtensions(),
_connectionExtensions(),
_phonecallExtensions(),
_phonebookExtensions(),
_messagingExtensions(),
_mediaPlayerExtensions(),
_extensions(),
_component(BTS_FB_LAST),
_stackInterface(BTS_IF_LAST),
_timerHandlingEnabled(false),
_timeoutControlList(),
_controllerClient(0),
_responseTimeoutHandler(0)
{
   // ignore given parameter
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

DbusBase& DbusBase::operator=(const DbusBase& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   // ignore given parameter

   return *this;
}

DbusBase::~DbusBase()
{
   if(_singleThread)
   {
      delete _singleThread;
   }
   _singleThread = 0;
   _mainControl = 0;
   _controllerClient = 0;
   _responseTimeoutHandler = 0;
}

void DbusBase::setMainControl(IN IMainControl* mainControl)
{
   if(_mainControl != mainControl)
   {
      _mainControl = mainControl;
   }
}

void DbusBase::setComponent(IN const BTSFunctionBlock component)
{
   _component = component;
}

void DbusBase::setStackInterface(IN const BTSInterfaceType stackInterface)
{
   _stackInterface = stackInterface;
}

void DbusBase::setTimerHandling(IN const bool enable)
{
   if(_timerHandlingEnabled != enable)
   {
      _timerHandlingEnabled = enable;
   }
}

void DbusBase::sendBts2IpcMessage(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   if(NULL != ptrMessage)
   {
      delete ptrMessage;
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
   }
}

void DbusBase::stop(void)
{
}

void DbusBase::run(void)
{
}

void DbusBase::resetHandler(void)
{
}

BTSErrorCode DbusBase::setCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   (void)(component);
   (void)(stackInterface);
   (void)(subComponent);
   (void)(userMode);
   (void)(controller);
   (void)(dbusInterfaces);
   (void)(configuration);

   _dbusIfAvailable = true;

   // return OK in case of controller interface is not needed (e.g. in case of simulation)
   return BTS_OK;
}

void DbusBase::resetCcDbusIfControllerIf(void)
{
   _dbusIfAvailable = false;
}

void DbusBase::handleTimerTick(void)
{
   // to be implemented by child class
}

void DbusBase::printQueueStatistics(void)
{
   // _bts2IpcWaitingQueue
   _bts2IpcWaitingQueue.lockAccess();
   ETG_TRACE_USR1((" ***DbusBase::_bts2IpcWaitingQueue.getSize()=%u", _bts2IpcWaitingQueue.getSize()));
   for(MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcWaitingQueue.getBegin(); it != _bts2IpcWaitingQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***DbusBase::_bts2IpcWaitingQueue[]: Bts2Ipc 0x%04X", (*it)->getTraceOpCode()));
   }
   _bts2IpcWaitingQueue.unlockAccess();

   // _bts2IpcWorkingQueue
   _bts2IpcWorkingQueue.lockAccess();
   ETG_TRACE_USR1((" ***DbusBase::_bts2IpcWorkingQueue.getSize()=%u", _bts2IpcWorkingQueue.getSize()));
   for(MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcWorkingQueue.getBegin(); it != _bts2IpcWorkingQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***DbusBase::_bts2IpcWorkingQueue[]: Bts2Ipc 0x%04X", (*it)->getTraceOpCode()));
   }
   _bts2IpcWorkingQueue.unlockAccess();

   // _ipc2BtsCallbackQueue
   _ipc2BtsCallbackQueue.lockAccess();
   ETG_TRACE_USR1((" ***DbusBase::_ipc2BtsCallbackQueue.getSize()=%u", _ipc2BtsCallbackQueue.getSize()));
   for(MessageQueue<Ipc2Bts_BaseMessage>::It it = _ipc2BtsCallbackQueue.getBegin(); it != _ipc2BtsCallbackQueue.getEnd(); ++it)
   {
      ETG_TRACE_USR1((" ***DbusBase::_ipc2BtsCallbackQueue[]: Bts2Ipc 0x%04X", (*it)->getTraceOpCode()));
   }
   _ipc2BtsCallbackQueue.unlockAccess();
}

void DbusBase::setSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   (void)testCommand;
   (void)testData;
}

void DbusBase::setSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   (void)testCommand;
   (void)testData;
}

void DbusBase::setSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   (void)testCommand;
   (void)testData;
}

void DbusBase::startThread(void)
{
   if(0 == _singleThread)
   {
      _singleThread = new ::fw::SingleThread();
   }

   if(_singleThread)
   {
      _eventQueue.clear();
      _terminateWorkerThread = false;
      _workerThreadTerminated = false;
      const ::std::string name("BTS_DBUS_IF");
      _singleThread->start(this, name, 0);
   }
}

void DbusBase::stopThread(void)
{
   if(true != _workerThreadTerminated)
   {
      _terminateWorkerThread = true;

      if(_singleThread)
      {
         _singleThread->stop();
      }
   }
}

void DbusBase::pushIpc2BtsMessage(IN Ipc2Bts_BaseMessage* ptrMessage, IN const bool highPrio /*= false*/)
{
   FW_IF_NULL_PTR_RETURN(ptrMessage);

   if(0 != _mainControl)
   {
      // add component and stack interface, sub component is part of message opcode
      ptrMessage->setComponent(_component);
      ptrMessage->setStackInterface(_stackInterface);
      _mainControl->pushIpc2BtsMessage(ptrMessage, highPrio);
   }
   else
   {
      // should never happen else you have programmed something wrong
      // #error_indication
      FW_NORMAL_ASSERT_ALWAYS();
      delete ptrMessage;
   }
}

void DbusBase::sendIpc2BtsMessageList(IN ::std::vector<Ipc2Bts_BaseMessage*>& ipc2BtsMsgList)
{
   if(0 < ipc2BtsMsgList.size())
   {
      for(size_t i = 0; i < ipc2BtsMsgList.size(); i++)
      {
         if(NULL != ipc2BtsMsgList[i])
         {
            // message was created successfully

            // forward to main control
            pushIpc2BtsMessage(ipc2BtsMsgList[i]);
         }
      }

      ipc2BtsMsgList.clear();
   }
}

Bts2Ipc_BaseMessage* DbusBase::findAndRemoveBts2IpcWorkingMessage(IN const Ipc2Bts_BaseMessage* ptrMessage, IN const bool withLock /*= true*/)
{
   Bts2Ipc_BaseMessage* msg = NULL;

   if((NULL != ptrMessage) && (::ccdbusif::DEFAULT_ACT != ptrMessage->getDbusToken()))
   {
      if(true == withLock)
      {
         _bts2IpcWorkingQueue.lockAccess();
      }

      MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcWorkingQueue.getBegin();

      while(it != _bts2IpcWorkingQueue.getEnd())
      {
         if((*it)->getDbusToken() == ptrMessage->getDbusToken())
         {
            msg = *it;
            break;
         }

         ++it;
      }

      if(it != _bts2IpcWorkingQueue.getEnd())
      {
         _bts2IpcWorkingQueue.doErase(it);
      }

      if(true == withLock)
      {
         _bts2IpcWorkingQueue.unlockAccess();
      }
   }

   return msg;
}

void DbusBase::findAndRemoveBts2IpcWorkingMessages(OUT ::std::vector<Bts2Ipc_BaseMessage*>& bts2IpcMsgList, IN const Ipc2Bts_BaseMessage* ptrMessage, IN const bool withLock /*= true*/)
{
   if((NULL != ptrMessage) && (0 < ptrMessage->getObjectId().size()))
   {
      if(true == withLock)
      {
         _bts2IpcWorkingQueue.lockAccess();
      }

      bts2IpcMsgList.reserve(_bts2IpcWorkingQueue.getSize());

      // find and remove all messages having same object id
      MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcWorkingQueue.getBegin();

      while(it != _bts2IpcWorkingQueue.getEnd())
      {
         if((*it)->getObjectId() == ptrMessage->getObjectId())
         {
            bts2IpcMsgList.push_back(*it);
            it = _bts2IpcWorkingQueue.doEraseAndNext(it);
         }
         else
         {
            ++it;
         }
      }

      if(true == withLock)
      {
         _bts2IpcWorkingQueue.unlockAccess();
      }
   }
}

void DbusBase::removeBts2IpcWorkingMessage(IN const Bts2Ipc_BaseMessage* ptrMessage, IN const bool withLock /*= true*/)
{
   if(NULL != ptrMessage)
   {
      if(true == withLock)
      {
         _bts2IpcWorkingQueue.lockAccess();
      }

      MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcWorkingQueue.getBegin();

      while(it != _bts2IpcWorkingQueue.getEnd())
      {
         if(*it == ptrMessage)
         {
            break;
         }

         ++it;
      }

      if(it != _bts2IpcWorkingQueue.getEnd())
      {
         _bts2IpcWorkingQueue.doErase(it);
      }

      if(true == withLock)
      {
         _bts2IpcWorkingQueue.unlockAccess();
      }
   }
}

void DbusBase::addDbusIfHandler(IN IDbusIfHandler* handler)
{
   FW_IF_NULL_PTR_RETURN(handler);

   _extensions.push_back(handler);
}

void DbusBase::setResponseTimeoutHandler(IN IDbusResponseTimeoutHandler* handler)
{
   _responseTimeoutHandler = handler;

   FW_NORMAL_ASSERT(0 != _responseTimeoutHandler);
}

void DbusBase::createTimeoutControlList(IN const unsigned int count)
{
   _timeoutControlList.reserve(count);
   DbusTimeoutControl* control;
   for(size_t i = 0; i < count; i++)
   {
      control = createTimeoutControlEntry();
      if(0 != control)
      {
         _timeoutControlList.push_back(control);
      }
   }
}

DbusTimeoutControl* DbusBase::createTimeoutControlEntry(void)
{
   FW_IF_NULL_PTR_RETURN_NULL(_responseTimeoutHandler);

   DbusTimeoutControl* control = new DbusTimeoutControl(_responseTimeoutHandler); // create via new because we will use "this" pointer as reference for timeout handler
   if(0 != control)
   {
      Timer* timer = TimerPool::getInstance().getTimer();
      if(0 != timer)
      {
         control->setTimer(timer);
         timer->setContext(_timerMaster.getContext());
      }
      else
      {
         FW_NORMAL_ASSERT_ALWAYS();
         delete control;
         control = 0;
      }
   }
   else
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   return control;
}

void DbusBase::destroyTimeoutControlList(void)
{
   DbusTimeoutControl* control;
   for(size_t i = 0; i < _timeoutControlList.size(); i++)
   {
      control = _timeoutControlList[i];
      if(0 != control)
      {
         if(0 != control->getTimer())
         {
            control->getTimer()->stop();
            TimerPool::getInstance().releaseTimer(control->getTimer());
         }
         delete control;
      }
   }
   _timeoutControlList.clear();
}

void DbusBase::stopAllTimeoutControlEntries(void)
{
   DbusTimeoutControl* control;
   for(size_t i = 0; i < _timeoutControlList.size(); i++)
   {
      control = _timeoutControlList[i];
      if(0 != control)
      {
         if(0 != control->getTimer())
         {
            control->getTimer()->stop();
         }
      }
   }
}

DbusTimeoutControl* DbusBase::getTimeoutControlEntry(void)
{
   DbusTimeoutControl* entry(0);
   DbusTimeoutControl* control;
   bool timerAvailable = false;
   for(size_t i = 0; i < _timeoutControlList.size(); i++)
   {
      control = _timeoutControlList[i];
      if(0 != control)
      {
         if(false == control->getInUse())
         {
            entry = control;
            control->setInUse(true);
            timerAvailable = true;
            break;
         }
      }
   }

   if(false == timerAvailable)
   {
      control = createTimeoutControlEntry();
      if(0 != control)
      {
         entry = control;
         control->setInUse(true);
         _timeoutControlList.push_back(control);
      }
   }

   return entry;
}

void DbusBase::releaseTimeoutControlEntry(IN const Bts2Ipc_BaseMessage* ptrMessage)
{
   if(0 == ptrMessage)
   {
      return;
   }

   DbusTimeoutControl* control;
   for(size_t i = 0; i < _timeoutControlList.size(); i++)
   {
      control = _timeoutControlList[i];
      if(0 != control)
      {
         if(ptrMessage == control->getMessage())
         {
            control->setMessage(0);
            control->setInUse(false);
            if(0 != control->getTimer())
            {
               control->getTimer()->stop();
            }
            break;
         }
      }
   }
}

void DbusBase::releaseAllTimeoutControlEntries(void)
{
   DbusTimeoutControl* control;

   for(size_t i = 0; i < _timeoutControlList.size(); i++)
   {
      control = _timeoutControlList[i];
      if(0 != control)
      {
         control->setMessage(0);
         control->setInUse(false);
         if(0 != control->getTimer())
         {
            control->getTimer()->stop();
         }
      }
   }
}

void DbusBase::handleSendingSuccess(IN Bts2Ipc_BaseMessage* ptrMessage)
{
   // HINT: This function is called within context of CcDbusIf library thread.

   FW_IF_NULL_PTR_RETURN(ptrMessage);

   // clean working queue (check for existing token)
   cleanBts2IpcWorkingQueue(ptrMessage, false);

   // success case => push to working queue
   _bts2IpcWorkingQueue.push(ptrMessage, false);

   // check for timeout
   if(0 == ptrMessage->getTimeout())
   {
      return;
   }

   // add to timer list
   DbusTimeoutControl* control(getTimeoutControlEntry());

   FW_IF_NULL_PTR_RETURN(control);
   FW_IF_NULL_PTR_RETURN(control->getTimer());

   _timerMaster.requestCurrentTime();

   control->setMessage(ptrMessage);
   control->getTimer()->start(ptrMessage->getTimeout(), control);
}

BTSErrorCode DbusBase::executeSetCcDbusIfControllerIf(IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent, IN const BTSUserMode userMode,
         IN ::ccdbusif::ICcDbusIfController* controller, IN const ::std::vector< BTSDbusInterfaceItem >& dbusInterfaces, IN const BTSLocalConfigurationContainer& configuration)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         _extensions[i]->setCcDbusIfControllerIf(component, stackInterface, subComponent, userMode, controller, dbusInterfaces, configuration);
      }
   }

   return BTS_OK;
}

void DbusBase::executeResetCcDbusIfControllerIf(void)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         _extensions[i]->resetCcDbusIfControllerIf();
      }
   }
}

bool DbusBase::executeSimulationTestCommand(IN const char* testCommand, IN const unsigned int testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSimulationTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusBase::executeSimulationTestCommand(IN const char* testCommand, IN const unsigned char* testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSimulationTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusBase::executeSimulationTestCommand(IN const char* testCommand, IN const Ipc2Bts_BaseMessage& testData)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->setSimulationTestCommand(testCommand, testData))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusBase::processMessage(OUT bool& deleteMsg, OUT bool& sendErrorMsg, IN Bts2Ipc_BaseMessage* message)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->sendMessage(deleteMsg, sendErrorMsg, message))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusBase::createErrorMessage(OUT Ipc2Bts_BaseMessage** ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage, IN const BTSIpcCommonErrorCode errorCode) const
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->createErrorMessage(ipc2BtsMessage, bts2IpcMessage, errorCode))
         {
            return true;
         }
      }
   }

   return false;
}

bool DbusBase::transferMessageData(OUT Ipc2Bts_BaseMessage* ipc2BtsMessage, IN const Bts2Ipc_BaseMessage* bts2IpcMessage) const
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         if(true == _extensions[i]->transferMessageData(ipc2BtsMessage, bts2IpcMessage))
         {
            return true;
         }
      }
   }

   return false;
}

void DbusBase::destroyDbusIfHandlerList(void)
{
   for(size_t i = 0; i < _extensions.size(); i++)
   {
      if(0 != _extensions[i])
      {
         delete _extensions[i];
      }
   }

   _extensions.clear();
}

void DbusBase::cleanBts2IpcWorkingQueue(IN Bts2Ipc_BaseMessage* ptrMessage, IN const bool withLock /*= true*/)
{
   if((NULL != ptrMessage) && (::ccdbusif::DEFAULT_ACT != ptrMessage->getDbusToken()))
   {
      Bts2Ipc_BaseMessage* msg = NULL;

      if(true == withLock)
      {
         _bts2IpcWorkingQueue.lockAccess();
      }

      MessageQueue<Bts2Ipc_BaseMessage>::It it = _bts2IpcWorkingQueue.getBegin();

      while(it != _bts2IpcWorkingQueue.getEnd())
      {
         if((*it)->getDbusToken() == ptrMessage->getDbusToken())
         {
            ETG_TRACE_USR1((" ***DbusBase::Received token already exist in _bts2IpcWorkingQueue"));

            msg = *it;
            break;
         }

         ++it;
      }

      if(it != _bts2IpcWorkingQueue.getEnd())
      {
         _bts2IpcWorkingQueue.doErase(it);

         // remove timer entry
         releaseTimeoutControlEntry(msg);

         // delete message
         if(NULL != msg)
         {
            delete msg;
         }
      }

      if(true == withLock)
      {
         _bts2IpcWorkingQueue.unlockAccess();
      }
   }
}

} //btstackif
