/**
 * @file DbusServiceAvailability.h
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS service availability handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details DBUS service availability handling.
 */

#ifndef _DBUS_SERVICE_AVAILABILITY_H_
#define _DBUS_SERVICE_AVAILABILITY_H_

#include "BtStackInternalTypes.h"
#include "ITimeoutHandler.h"

namespace btstackif {

// class forward declarations
class IBasicControl;
class Timer;
class DbusServiceEntry;
class DbusServiceInfo;
class DbusServiceStatus;
class DbusServiceGroupStatus;

/**
 * Type definition for DBUS service information.
 */
class DbusServiceEntry : public BTSProtocolBaseEntry
{
public:
   /**
    * Default constructor.
    */
   DbusServiceEntry();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   DbusServiceEntry(IN const DbusServiceEntry& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   DbusServiceEntry& operator=(IN const DbusServiceEntry& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: equal
    * @return = false: not equal
    */
   bool operator==(IN const DbusServiceEntry& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: not equal
    * @return = false: equal
    */
   bool operator!=(IN const DbusServiceEntry& ref) const;

   /**
    * Less than operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: less than
    * @return = false: not less than
    */
   bool operator<(const DbusServiceEntry& ref) const;

   /**
    * Greater than operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: greater than
    * @return = false: not greater than
    */
   bool operator>(const DbusServiceEntry& ref) const;

   /**
    * Destructor.
    */
   virtual ~DbusServiceEntry();

   // member with public access
   BTSBusName busName; /**< bus name */
   BTSObjectPath objPath; /**< object path */
   ::ccdbusif::DbusBusType busType; /**< bus type */

protected:
   int entryCompare(const DbusServiceEntry& ref) const;
};

/**
 * Type definition for DBUS service information.
 */
class DbusServiceInfo : public ITimeoutHandler
{
public:
   /**
    * Default constructor.
    */
   DbusServiceInfo();

   /**
    * Copy constructor.
    *
    * @param[in] ref: reference of message to be copied
    */
   DbusServiceInfo(IN const DbusServiceInfo& ref);

   /**
    * Assignment operator.
    *
    * @param[in] ref: reference of message to be assigned
    */
   DbusServiceInfo& operator=(IN const DbusServiceInfo& ref);

   /**
    * Equality operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: equal
    * @return = false: not equal
    */
   bool operator==(IN const DbusServiceInfo& ref) const;

   /**
    * Inequality operator.
    *
    * @param[in] ref: reference of message to be compared
    *
    * @return = true: not equal
    * @return = false: equal
    */
   bool operator!=(IN const DbusServiceInfo& ref) const;

   /**
    * Destructor.
    */
   virtual ~DbusServiceInfo();

   /**
    * Function to handle timeout.
    *
    * @param[in] timerId: timer id
    */
   virtual void handleTimeout(IN const BTSTimerId timerId);

   /**
    * Setter for interface for sending Ipc2Bts message.
    *
    * @param[in] sendIf: interface for sending Ipc2Bts message
    */
   inline void setIpc2BtsSendIf(IN IBasicControl* sendIf) { _ipc2BtsSendIf = sendIf; }

   /**
    * Set interface/component/service.
    *
    * @param[in] interface: interface for sending Ipc2Bts message
    */
   inline void setInterface(IN const BTSCommonEnumClass interface) { _interface = interface; }

   /**
    * Get interface/component/service.
    *
    * @return = interface/component/service
    */
   inline BTSCommonEnumClass getInterface(void) const { return _interface; }

   /**
    * Start timer.
    */
   void startTimer(void);

   /**
    * Stop timer.
    */
   void stopTimer(void);

   /**
    * Release timer.
    */
   void releaseTimer(void);

   /**
    * Set timeout.
    *
    * @param[in] timeout: timeout
    */
   static void setTimeout(IN const BTSTimeValue timeout);

   // member with public access
   BTSBusName busName; /**< bus name */
   BTSObjectPath objPath; /**< object path TODO: check if really needed here */
   ::ccdbusif::DbusBusType busType; /**< bus type */
   BTSDbusServiceAvailability serviceAvailability; /**< service availability */

private:
   IBasicControl* _ipc2BtsSendIf; /**< interface for sending Ipc2Bts message */
   BTSCommonEnumClass _interface; /**< related interface/component/service */
   Timer* _timer; /**< timer */
   static BTSTimeValue _dbusServiceTimeout; /**< DBUS service timeout */

   /**
    * Setter for timer.
    *
    * @param[in] sendIf: interface for sending Ipc2Bts message
    */
   inline void setTimer(IN Timer* timer) { _timer = timer; }
};

/**
 *
 */
class DbusServiceStatus
{
public:
   /**
    * Default constructor.
    */
   DbusServiceStatus();

   /**
    * Constructor.
    *
    * @param[in] name: DBUS service name
    */
   DbusServiceStatus(IN const ::std::string& name);

   /**
    * Destructor.
    */
   virtual ~DbusServiceStatus();

   /**
    * Set enabled flag.
    *
    * @param[in] enabled: enabled flag
    */
   inline void setEnabled(IN const bool enabled) { _enabled = enabled; }

   /**
    * Get enabled flag.
    *
    * @return = enabled flag
    */
   inline bool getEnabled(void) const { return _enabled; }

   /**
    * Set name.
    *
    * @param[in] name: name
    */
   inline void setName(IN const ::std::string& name) { _name = name; }

   /**
    * Get name.
    *
    * @return = name
    */
   inline const ::std::string& getName(void) const { return _name; }

   /**
    * Disable timer.
    */
   inline void disableTimer(void) { _timerEnabled = false; }

   /**
    * Setter for interface for sending Ipc2Bts message.
    *
    * @param[in] sendIf: interface for sending Ipc2Bts message
    */
   inline void setIpc2BtsSendIf(IN IBasicControl* sendIf) { _ipc2BtsSendIf = sendIf; }

   /**
    * Set interface/component/service.
    *
    * @param[in] interface: interface for sending Ipc2Bts message
    */
   inline void setInterface(IN const BTSCommonEnumClass interface) { _interface = interface; }

   /**
    * Get interface/component/service.
    *
    * @return = interface/component/service
    */
   inline BTSCommonEnumClass getInterface(void) const { return _interface; }

   /**
    * Get service/proxy availability of all interfaces/components/services together.
    *
    * @return = service/proxy availability
    */
   BTSDbusServiceAvailability getGroupAvailability(void) const;

   /**
    * Remove all entries from internal list.
    */
   virtual void removeAllServiceInfos(void);

   //###########################################################################

   // --- for all interfaces with fixed object path ---
   /**
    * Add entry to internal list.
    *
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    */
   virtual void removeServiceInfo(void);

   /**
    * Set service/proxy availability.
    *
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(void) const;

   //###########################################################################

   // --- for all interfaces based on bus type, bus name and object path ---
   /**
    * Add entry to internal list.
    *
    * @param[in] busType: bus type
    * @param[in] busName: bus name (DBUS service)
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    *
    * @param[in] busType: bus type
    * @param[in] busName: bus name (DBUS service)
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void removeServiceInfo(IN const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath);

   /**
    * Set service/proxy availability.
    *
    * @param[in] busType: bus type
    * @param[in] busName: bus name (DBUS service)
    * @param[in] objPath: object path of related DBUS service/proxy
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath, IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @param[in] busType: bus type
    * @param[in] busName: bus name (DBUS service)
    * @param[in] objPath: object path of related DBUS service/proxy
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(IN const ::ccdbusif::DbusBusType busType, IN const BTSBusName& busName, IN const BTSObjectPath& objPath) const;

   //###########################################################################

   // --- for all interfaces with variable object path based on device address ---
   /**
    * Add entry to internal list.
    *
    * @param[in] address: device address
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const BTSBDAddress& address, IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    *
    * @param[in] address: device address
    */
   virtual void removeServiceInfo(IN const BTSBDAddress& address);

   /**
    * Set service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const BTSBDAddress& address, IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @param[in] address: device address
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(IN const BTSBDAddress& address) const;

   /**
    * Get availability of entry itself.
    *
    * @param[in] address: device address
    *
    * @return = entry availability
    */
   virtual bool isEntryAvailable(IN const BTSBDAddress& address) const;

   //###########################################################################

   // --- for all interfaces with variable object path based on device address + protocol id ---
   /**
    * Add entry to internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    */
   virtual void removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId);

   /**
    * Set service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId) const;

   //###########################################################################

   // --- for all interfaces with variable object path based on device address + protocol id + SPP uuid ---
   /**
    * Add entry to internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    */
   virtual void removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid);

   /**
    * Set service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid) const;

   //###########################################################################

   // --- for all interfaces with variable object path based on device address + protocol id + MAS instance id ---
   /**
    * Add entry to internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] instanceId: MAS instance id
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId, IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] instanceId: MAS instance id
    */
   virtual void removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId);

   /**
    * Set service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] instanceId: MAS instance id
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId, IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] instanceId: MAS instance id
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSMasInstanceId instanceId) const;

protected:
   bool _enabled; /**< service enabled flag */
   ::std::string _name; /**< service name */
   ::std::map<DbusServiceEntry, DbusServiceInfo> _info; /**< service information */
   bool _timerEnabled; /**< timer enabled flag */
   IBasicControl* _ipc2BtsSendIf; /**< interface for sending Ipc2Bts message */
   BTSCommonEnumClass _interface; /**< related interface/component/service */

private:
   /**
    * Add entry to internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] instanceId: MAS instance id
    * @param[in] objPath: object path of related DBUS service/proxy
    */
   virtual void addServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId, IN const BTSObjectPath& objPath);

   /**
    * Remove entry from internal list.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] instanceId: MAS instance id
    */
   virtual void removeServiceInfo(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId);

   /**
    * Set service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] instanceId: MAS instance id
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId, IN const BTSDbusServiceAvailability availability);

   /**
    * Set service/proxy availability.
    *
    * @param[in] info: service info
    * @param[in] availability: service/proxy availability
    */
   virtual void setAvailability(IN DbusServiceInfo& info, IN const BTSDbusServiceAvailability availability);

   /**
    * Get service/proxy availability.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] instanceId: MAS instance id
    *
    * @return = service/proxy availability
    */
   virtual BTSDbusServiceAvailability getAvailability(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId) const;

   /**
    * Get availability of entry itself.
    *
    * @param[in] address: device address
    * @param[in] protocolId: protocol id
    * @param[in] uuid: SPP uuid
    * @param[in] instanceId: MAS instance id
    *
    * @return = entry availability
    */
   virtual bool isEntryAvailable(IN const BTSBDAddress& address, IN const BTSProtocolId protocolId, IN const BTSUuid& uuid, IN const BTSMasInstanceId instanceId) const;
};

/**
 *
 */
class DbusServiceGroupStatus : public DbusServiceStatus
{
public:
   /**
    * Default constructor.
    */
   DbusServiceGroupStatus();

   /**
    * Constructor.
    *
    * @param[in] name: DBUS service name
    */
   DbusServiceGroupStatus(IN const ::std::string& name);

   /**
    * Destructor.
    */
   virtual ~DbusServiceGroupStatus();

   /**
    * Add entry to internal list. To be used only for services/proxies with fixed path.
    *
    * @param[in] status: service status entry
    */
   void addServiceStatus(IN DbusServiceStatus* status);

   /**
    * Check availability.
    */
   void checkAvailability(void);

protected:
   ::std::vector<DbusServiceStatus*> _groupInfo; /**< service group information */
};

} //btstackif

#endif //_DBUS_SERVICE_AVAILABILITY_H_
