/**
 * @file DbusCallbackIf.h
 *
 * @par SW-Component
 * IPC
 *
 * @brief DBUS basic callback handling.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for DBUS callback handling.
 */

#ifndef _DBUS_CALLBACK_IF_H_
#define _DBUS_CALLBACK_IF_H_

#include "BtStackInternalTypes.h"
#include "asf/core/Types.h"
#include <boost/shared_ptr.hpp>

// class forward declarations
namespace asf {
namespace dbus {
namespace DBusTypes {
class DBusError;
class DefaultMessage;
} //DBusTypes
} //dbus
} //asf

namespace btstackif {

// class forward declarations
class IDbusRecHandler;
class Ipc2Bts_BaseMessage;

/**
 *
 */
class DbusCallbackIf
{
public:
   /**
    * Constructor.
    *
    * @param[in] recHandler: DBUS message receive handler
    */
   DbusCallbackIf(IDbusRecHandler* recHandler);

   /**
    * Destructor.
    */
   virtual ~DbusCallbackIf();

protected:
   /**
    * Default constructor.
    */
   DbusCallbackIf();

   /**
    * Response handler for messages of type DefaultMessage.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] response: received message
    */
   void onResponse(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DefaultMessage >& response);

   /**
    * Response handler for messages of different type than DefaultMessage.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] token: token of received message
    */
   void onResponse(Ipc2Bts_BaseMessage* ptrMsg, act_t token);

   /**
    * Signal handler for messages of type DefaultMessage.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] signal: received message
    */
   void onSignal(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DefaultMessage >& signal);

   /**
    * Signal handler for messages of different type than DefaultMessage.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] token: token of received message
    * @param[in] highPrio: priority flag
    */
   void onSignal(Ipc2Bts_BaseMessage* ptrMsg, act_t token, const bool highPrio = false);

   /**
    * Error handler for messages of type DBusError.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] error: received message
    */
   void onError(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError >& error);

   /**
    * Error handler for messages of type DBusError.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] error: received message
    * @param[in] errorCode: error code
    */
   void onError(Ipc2Bts_BaseMessage* ptrMsg, const ::boost::shared_ptr< ::asf::dbus::DBusTypes::DBusError >& error, const BTSIpcCommonErrorCode errorCode);

   /**
    * Update handler for messages of different type than DefaultMessage.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] token: token of received message
    */
   void onUpdate(Ipc2Bts_BaseMessage* ptrMsg, act_t token);

   /**
    * Request handler for messages of different type than DefaultMessage.
    *
    * @param[in] ptrMsg: internal message
    * @param[in] token: token of received message
    */
   void onRequest(Ipc2Bts_BaseMessage* ptrMsg, act_t token);

   /**
    * set receive handler.
    *
    * @param[in] recHandler: DBUS message receive handler
    */
   inline void setHandler(IDbusRecHandler* recHandler) { if(_recHandler != recHandler) { _recHandler = recHandler; } }

private:
   IDbusRecHandler* _recHandler; /**< receive handler used for forwarding of internal message */
};

} //btstackif

#endif //_DBUS_CALLBACK_IF_H_
