/**
 * @file TimerTickSource.cpp
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer tick source.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of timer tick source.
 */

#include "TimerTickSource.h"
#include "ITimerTickHandler.h"

#include <cstring>
#include <cerrno>
#include <sys/types.h>
#include <sys/syscall.h>
#include <sys/prctl.h>
#include <unistd.h>

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TimerTickSource.cpp.trc.h"
#endif
#endif

namespace btstackif {

#ifdef ENABLE_USAGE_NANOSLEEP

TimerTickSource::TimerTickSource()
: _timeoutInMs(100) // max allowed timeout value is 999ms
{
   _threadHandle = 0;
   _threadCreated = false;
   _terminateThread = false;
   _tickHandler = NULL;
}

TimerTickSource::~TimerTickSource()
{
   stop();

   _tickHandler = NULL;
}

void TimerTickSource::start(IN ITimerTickHandler* tickHandler)
{
   if(true == _threadCreated)
   {
      // already done
      return;
   }

   int result = -1;

   _terminateThread = false;

   _tickHandler = tickHandler;

   // create thread
   for(int i = 0; i < MAX_TRIES_FOR_THREAD_CREATION; i++)
   {
      result = pthread_create(&_threadHandle, NULL, &timerThread, (void*)this);
      if(0 != result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_create.html:
            The pthread_create() function shall fail if:

            [EAGAIN]
              The system lacked the necessary resources to create another thread, or the system-imposed limit on the total number of threads in a process {PTHREAD_THREADS_MAX} would be exceeded.
            [EPERM]
              The caller does not have appropriate permission to set the required scheduling parameters or scheduling policy.
              => should never happen

            The pthread_create() function may fail if:

            [EINVAL]
              The attributes specified by attr are invalid.
              => should never happen because no attribute setting is given

            The pthread_create() function shall not return an error code of [EINTR].
          *
          */

         if(EAGAIN == result)
         {
            // try again
         }
         else
         {
            ETG_TRACE_ERR((" start(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
            FW_NORMAL_ASSERT_ALWAYS();
            return;
         }
      }
      else
      {
         // success
         break;
      }
   }
   if(0 != result)
   {
      ETG_TRACE_ERR((" start(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   _threadCreated = true;

   ETG_TRACE_USR1((" start(): timer thread started"));
}

void TimerTickSource::stop(void)
{
   if(true != _threadCreated)
   {
      // already done
      return;
   }

   // TODO: check for improvement
   _terminateThread = true;
   usleep((_timeoutInMs + 25) * 1000);

   _threadCreated = false;

   ETG_TRACE_USR1((" stop(): timer thread canceled"));
}

void* TimerTickSource::timerThread(void* context)
{
   ETG_TRACE_USR1((" timerThread(): enter"));

   // set thread name
   (void)prctl(PR_SET_NAME, "BTSTACKIF_TIMER", 0, 0, 0);

   TimerTickSource* timer = (TimerTickSource*)context;
   if(NULL != timer)
   {
      int errNumber;
      struct timespec timePeriod;
      struct timespec remainingTime;

      timePeriod.tv_sec = 0;

      while(false == timer->_terminateThread)
      {
         timePeriod.tv_nsec = (timer->_timeoutInMs % 1000) * 1000000L;

         for(;;)
         {
            if(-1 == nanosleep(&timePeriod, &remainingTime))
            {
               /*
                * from http://pubs.opengroup.org/onlinepubs/009695399/functions/nanosleep.html:
                  The nanosleep() function shall fail if:

                  [EINTR]
                    The nanosleep() function was interrupted by a signal.
                    => will be handled
                  [EINVAL]
                    The rqtp argument specified a nanosecond value less than zero or greater than or equal to 1000 million.
                    => should never happen because valid value is provided
                *
                */

               errNumber = errno;

               if(EINTR == errNumber)
               {
                  // try again
                  timePeriod.tv_nsec = remainingTime.tv_nsec;
               }
               else
               {
                  ETG_TRACE_ERR((" timerThread(): nanosleep failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
                  FW_NORMAL_ASSERT_ALWAYS();
                  break;
               }
            }
            else
            {
               // success
               break;
            }
         }

         // call timer handler
         if(NULL != timer->_tickHandler)
         {
            timer->_tickHandler->setTimerTick();
         }
      }

      timer->_tickHandler = NULL;
   }

   ETG_TRACE_USR1((" timerThread(): exit"));

   return NULL;
}

#else

#ifdef ENABLE_SIG_ACTION_THREAD
bool TimerTickSource::_threadStarted = false;
#endif
TimerTickSource* TimerTickSource::_myExemplar = NULL;

TimerTickSource::TimerTickSource()
: _timeoutInMs(100)
{
   _threadCreated = false;
#ifdef ENABLE_SIG_ACTION_THREAD
   memset(&_threadAttr, 0, sizeof(_threadAttr));
   _threadHandle = 0;
   _threadID = 0;
   _threadStarted = false;
#endif
   memset(&_sigAction, 0, sizeof(_sigAction));
   _timerId = 0;
   _timerCreated = false;
   _timerStarted = false;
   _myExemplar = this;
}

TimerTickSource::~TimerTickSource()
{
   stop();
   _myExemplar = NULL;
}

void TimerTickSource::start(void)
{
   if(true == _threadCreated)
   {
      // already done
      return;
   }

#ifdef ENABLE_SIG_ACTION_THREAD
   _threadStarted = false;

   // initialize the thread attributes object
   int result = pthread_attr_init(&_threadAttr);
   if(0 != result)
   {
      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_attr_init.html:
         The pthread_attr_init() function shall fail if:

         [ENOMEM]
           Insufficient memory exists to initialize the thread attributes object.
           => should never happen

         The pthread_attr_destroy() function may fail if:

         [EINVAL]
           The value specified by attr does not refer to an initialized thread attribute object.

         The pthread_attr_init() function may fail if:

         [EBUSY]
           The implementation has detected an attempt to reinitialize the thread attribute referenced by attr, a previously initialized, but not yet destroyed, thread attribute.
           => should never happen because init will be done only once

         These functions shall not return an error code of [EINTR].
       *
       */

      ETG_TRACE_ERR((" start(): pthread_attr_init failed: ERROR=%d (%s)", result, strerror(result)));
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // set the stack size attribute because default stack size is too small (based on information in Framework from Media Player)
   result = pthread_attr_setstacksize(&_threadAttr, DEFAULT_STACKSIZE);
   if(0 != result)
   {
      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_attr_setstacksize.html:
         The pthread_attr_setstacksize() function shall fail if:

         [EINVAL]
           The value of stacksize is less than {PTHREAD_STACK_MIN} or exceeds a system-imposed limit.
           => should never happen because stack size is greater than PTHREAD_STACK_MIN

         These functions may fail if:

         [EINVAL]
           The value specified by attr does not refer to an initialized thread attribute object.
           => should never happen because init was done before

         These functions shall not return an error code of [EINTR].
       *
       */

      ETG_TRACE_ERR((" start(): pthread_attr_setstacksize failed: ERROR=%d (%s)", result, strerror(result)));
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // create thread
   for(int i = 0; i < MAX_TRIES_FOR_THREAD_CREATION; i++)
   {
      result = pthread_create(&_threadHandle, &_threadAttr, &sigactionThread, (void*)this);
      if(0 != result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_create.html:
            The pthread_create() function shall fail if:

            [EAGAIN]
              The system lacked the necessary resources to create another thread, or the system-imposed limit on the total number of threads in a process {PTHREAD_THREADS_MAX} would be exceeded.
            [EPERM]
              The caller does not have appropriate permission to set the required scheduling parameters or scheduling policy.
              => should never happen

            The pthread_create() function may fail if:

            [EINVAL]
              The attributes specified by attr are invalid.
              => should never happen because valid attribute setting is given

            The pthread_create() function shall not return an error code of [EINTR].
          *
          */

         if(EAGAIN == result)
         {
            // try again
         }
         else
         {
            ETG_TRACE_ERR((" start(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
            FW_NORMAL_ASSERT_ALWAYS();
            return;
         }
      }
      else
      {
         // success
         break;
      }
   }
   if(0 != result)
   {
      ETG_TRACE_ERR((" start(): pthread_create failed: ERROR=%d (%s)", result, strerror(result)));
      FW_NORMAL_ASSERT_ALWAYS();
      return;
   }

   // wait for tid set
   while(false == _threadStarted)
   {
      usleep(2000);
   }
#else
   // initialize and empty a signal set
   if(-1 == sigemptyset(&_sigAction.sa_mask))
   {
      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sigemptyset.html:
         No errors are defined.
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" start(): sigemptyset failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }

   // set callback function
   _sigAction.sa_sigaction = &defaultCallback;
   _sigAction.sa_flags = SA_SIGINFO;

   // set signal action
   if(-1 == sigaction(SIGUSR1, &_sigAction, NULL))
   {
      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sigaction.html:
         The sigaction() function shall fail if:

         [EINVAL]
           The sig argument is not a valid signal number or an attempt is made to catch a signal that cannot be caught or ignore a signal that cannot be ignored.
           => should never happen because valid signal number is given
         [ENOTSUP]
           The SA_SIGINFO bit flag is set in the sa_flags field of the sigaction structure, and the implementation does not support either the Realtime Signals Extension option, or the XSI Extension option.
           => should never happen because this must be supported

         The sigaction() function may fail if:

         [EINVAL]
           An attempt was made to set the action to SIG_DFL for a signal that cannot be caught or ignored (or both).
           => should never happen because this is not done
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" start(): sigaction failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }
#endif

   _threadCreated = true;

   createAndStartPeriodicTimer();

   ETG_TRACE_USR1((" start(): timer thread started"));
}

void TimerTickSource::stop(void)
{
   stopAndDeletePeriodicTimer();

   if(true == _threadCreated)
   {
#ifdef ENABLE_SIG_ACTION_THREAD
      // cancel execution of a thread
      int result = pthread_cancel(_threadHandle);
      if(0 != result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_cancel.html:
            The pthread_cancel() function may fail if:

            [ESRCH]
              No thread could be found corresponding to that specified by the given thread ID.
              => should never happen else the thread was terminated before

            The pthread_cancel() function shall not return an error code of [EINTR].
          *
          */

         ETG_TRACE_ERR((" stop(): pthread_cancel failed: ERROR=%d (%s)", result, strerror(result)));
         FW_NORMAL_ASSERT_ALWAYS();
      }
#endif
      _threadCreated = false;

      ETG_TRACE_USR1((" stop(): timer thread canceled"));
   }
}

void TimerTickSource::createAndStartPeriodicTimer(void)
{
   if(false == _timerCreated)
   {
      // create timer
      struct sigevent sev;
      int result = -1;
      int errNumber = EAGAIN;

      memset(&sev, 0, sizeof(sev));
#ifdef ENABLE_SIG_ACTION_THREAD
      sev.sigev_notify = SIGEV_THREAD_ID;
      sev._sigev_un._tid = _threadID;
#else
      sev.sigev_notify = SIGEV_SIGNAL;
#endif
      sev.sigev_signo = SIGUSR1;
      sev.sigev_value.sival_ptr = (void*)this;

      for(int i = 0; i < MAX_TRIES_FOR_TIMER_CREATION; i++)
      {
         result = timer_create(CLOCK_MONOTONIC, &sev, &_timerId);
         if(-1 == result)
         {
            /*
             * from http://pubs.opengroup.org/onlinepubs/009695399/functions/timer_create.html:
               The timer_create() function shall fail if:

               [EAGAIN]
                 The system lacks sufficient signal queuing resources to honor the request.
               [EAGAIN]
                 The calling process has already created all of the timers it is allowed by this implementation.
                 => should never happen because only 1 timer is created
               [EINVAL]
                 The specified clock ID is not defined.
                 => should never happen because valid clock id is provided
               [ENOTSUP]
                 [CPT|TCT] [Option Start] The implementation does not support the creation of a timer attached to the CPU-time clock that is specified by clock_id and associated with a process or thread different from the process or thread invoking timer_create(). [Option End]
                 => should never happen because it must be supported
             *
             */

            errNumber = errno;

            if(EAGAIN == errNumber)
            {
               // try again
            }
            else
            {
               ETG_TRACE_ERR((" createAndStartPeriodicTimer(): timer_create failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            }
         }
         else
         {
            // success
            break;
         }
      }
      if((-1 == result) && (EAGAIN == errNumber))
      {
         ETG_TRACE_ERR((" createAndStartPeriodicTimer(): timer_create failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }
      if(-1 == result)
      {
         return;
      }

      _timerCreated = true;
   }

   if(false == _timerStarted)
   {
      // start timer
      struct itimerspec its;

      memset(&its, 0, sizeof(its));
      its.it_value.tv_sec = _timeoutInMs / 1000;
      its.it_value.tv_nsec = (_timeoutInMs % 1000) * 1000000L;
      its.it_interval.tv_sec = _timeoutInMs / 1000;
      its.it_interval.tv_nsec = (_timeoutInMs % 1000) * 1000000L;

      if(-1 == timer_settime(_timerId, 0, &its, NULL))
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/timer_getoverrun.html:
            The timer_settime() function shall fail if:

            [EINVAL]
              A value structure specified a nanosecond value less than zero or greater than or equal to 1000 million, and the it_value member of that structure did not specify zero seconds and nanoseconds.
              => should never happen because valid values provided

            These functions may fail if:

            [EINVAL]
              The timerid argument does not correspond to an ID returned by timer_create() but not yet deleted by timer_delete().
              => should never happen because valid timer id provided

            The timer_settime() function may fail if:

            [EINVAL]
              The it_interval member of value is not zero and the timer was created with notification by creation of a new thread ( sigev_sigev_notify was SIGEV_THREAD) and a fixed stack address has been set in the thread attribute pointed to by sigev_notify_attributes.
              => should never happen
          *
          */

         int errNumber = errno;

         ETG_TRACE_ERR((" createAndStartPeriodicTimer(): timer_settime failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }
      else
      {
         _timerStarted = true;
      }
   }
}

void TimerTickSource::stopAndDeletePeriodicTimer(void)
{
   if(true == _timerStarted)
   {
      struct itimerspec its;

      // stop the timer
      memset(&its, 0, sizeof(its));
      if(-1 == timer_settime(_timerId, 0, &its, NULL))
      {
         int errNumber = errno;

         ETG_TRACE_ERR((" stopAndDeletePeriodicTimer(): timer_settime failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      _timerStarted = false;
   }

   if(true == _timerCreated)
   {
      // delete the timer
      if(-1 == timer_delete(_timerId))
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/timer_delete.html:
            The timer_delete() function may fail if:

            [EINVAL]
              The timer ID specified by timerid is not a valid timer ID.
              => should not happen because valid timer id is provided
          *
          */

         int errNumber = errno;

         ETG_TRACE_ERR((" stopAndDeletePeriodicTimer(): timer_delete failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      _timerCreated = false;
   }
}

#ifdef ENABLE_SIG_ACTION_THREAD
void* TimerTickSource::sigactionThread(void* context)
{
   TimerTickSource* timer = (TimerTickSource*)context;
   if(NULL != timer)
   {
      // set cancelability state
      int result = pthread_setcancelstate(PTHREAD_CANCEL_ENABLE, NULL);
      if(0 != result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_setcancelstate.html:
            The pthread_setcancelstate() function may fail if:

            [EINVAL]
              The specified state is not PTHREAD_CANCEL_ENABLE or PTHREAD_CANCEL_DISABLE.
              => should never happen because PTHREAD_CANCEL_ENABLE is used

            These functions shall not return an error code of [EINTR].
          *
          */

         ETG_TRACE_ERR((" sigactionThread(): pthread_setcancelstate failed: ERROR=%d (%s)", result, strerror(result)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      result = pthread_setcanceltype(PTHREAD_CANCEL_ASYNCHRONOUS, NULL);
      if(0 != result)
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/pthread_setcancelstate.html:
            The pthread_setcanceltype() function may fail if:

            [EINVAL]
              The specified type is not PTHREAD_CANCEL_DEFERRED or PTHREAD_CANCEL_ASYNCHRONOUS.
              => should never happen because PTHREAD_CANCEL_ASYNCHRONOUS is used

            These functions shall not return an error code of [EINTR].
          *
          */

         ETG_TRACE_ERR((" sigactionThread(): pthread_setcanceltype failed: ERROR=%d (%s)", result, strerror(result)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      // initialize and empty a signal set
      if(-1 == sigemptyset(&timer->_sigAction.sa_mask))
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sigemptyset.html:
            No errors are defined.
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" sigactionThread(): sigemptyset failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      // set callback function
      timer->_sigAction.sa_sigaction = &defaultCallback;
      timer->_sigAction.sa_flags = SA_SIGINFO;

      // set signal action
      if(-1 == sigaction(SIGUSR1, &timer->_sigAction, NULL))
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sigaction.html:
            The sigaction() function shall fail if:

            [EINVAL]
              The sig argument is not a valid signal number or an attempt is made to catch a signal that cannot be caught or ignore a signal that cannot be ignored.
              => should never happen because valid signal number is given
            [ENOTSUP]
              The SA_SIGINFO bit flag is set in the sa_flags field of the sigaction structure, and the implementation does not support either the Realtime Signals Extension option, or the XSI Extension option.
              => should never happen because this must be supported

            The sigaction() function may fail if:

            [EINVAL]
              An attempt was made to set the action to SIG_DFL for a signal that cannot be caught or ignored (or both).
              => should never happen because this is not done
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" sigactionThread(): sigaction failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      timer->_threadID = syscall(SYS_gettid);
      _threadStarted = true;

      ETG_TRACE_USR1((" sigactionThread(): TID=%u", timer->_threadID));

      // endless: processing of incoming signals
      for(;;)
      {
         sleep(1000);
      }
   }
   else
   {
      _threadStarted = true;
   }

   ETG_TRACE_USR1((" sigactionThread(): exit"));

   return NULL;
}
#endif

void TimerTickSource::defaultCallback(int signo, siginfo_t* info, void* context)
{
   (void)context;

   if(NULL != info)
   {
      if((info->si_code == SI_TIMER) && (signo == SIGUSR1))
      {
         // TODO: implement
         ETG_TRACE_USR1((" defaultCallback(): expired"));

         if(info->si_value.sival_ptr == (void*)_myExemplar)
         {
            // OK
         }
         else
         {
            // NOK
            ETG_TRACE_ERR((" defaultCallback(): wrong signal value"));
         }
      }
   }
}

#endif

} //btstackif
