/**
 * @file TimeoutSemaphore.cpp
 *
 * @par SW-Component
 * IPC
 *
 * @brief Semaphore with timeout.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of semaphore with timeout handling.
 */

#include "TimeoutSemaphore.h"

#include <cstring>
#include <cerrno>
#include <time.h>
#include <stdint.h> // using <cstdint> compiler complains that std=c++0x or -std=gnu++0x compiler option must be enabled

#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_COMMON
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TimeoutSemaphore.cpp.trc.h"
#endif
#endif

//#define ENABLE_PRINT_SEMAPHORE
#undef ENABLE_PRINT_SEMAPHORE

namespace btstackif {

TimeoutSemaphore::TimeoutSemaphore()
{
   memset(&_semaphore, 0, sizeof(_semaphore));
   if(-1 == sem_init(&_semaphore, 0, 0))
   {
      // check result for debugging purpose

      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_init.html:
         The sem_init() function shall fail if:

         [EINVAL]
            The value argument exceeds {SEM_VALUE_MAX}.
            => value is 0 and cannot exceed SEM_VALUE_MAX
         [ENOSPC]
            A resource required to initialize the semaphore has been exhausted, or the limit on semaphores ( {SEM_NSEMS_MAX}) has been reached.
            => should never happen
         [EPERM]
            The process lacks the appropriate privileges to initialize the semaphore.
            => should never happen else there is a general problem
         [ENOSYS]
            pshared is nonzero, but the system does not support process-shared semaphores
            => pshared is 0
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" TimeoutSemaphore(): sem_init failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }

#ifdef ENABLE_PRINT_SEMAPHORE
   printSemaphoreCount();
#endif

   _resetDone = false;
}

TimeoutSemaphore::~TimeoutSemaphore()
{
   (void)sem_destroy(&_semaphore); // ignore return value
}

void TimeoutSemaphore::post(void)
{
   if(-1 == sem_post(&_semaphore))
   {
      // check result for debugging purpose

      /*
       * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_post.html:
         The sem_post() function may fail if:

         [EINVAL]
            The sem argument does not refer to a valid semaphore.
            => a valid semaphore is provided
         [EOVERFLOW]
            The maximum allowable value for a semaphore would be exceeded.
            => should never happen
       *
       */

      int errNumber = errno;
      ETG_TRACE_ERR((" post(): sem_post failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }

#ifdef ENABLE_PRINT_SEMAPHORE
   printSemaphoreCount();
#endif
}

BTSErrorCode TimeoutSemaphore::wait(IN const BTSTimeValue timeoutInMs /*= 0*/)
{
   BTSErrorCode errCode = BTS_ERROR;

   if(0 == timeoutInMs)
   {
      for(;;)
      {
         if(-1 == sem_wait(&_semaphore))
         {
            /*
             * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_wait.html:
               The sem_trywait() and sem_wait() functions shall fail if:

               [EAGAIN]
                  The semaphore was already locked, so it cannot be immediately locked by the sem_trywait() operation ( sem_trywait() only).
                  => valid for sem_trywait() only, no need to handle

               The sem_trywait() and sem_wait() functions may fail if:

               [EDEADLK]
                  A deadlock condition was detected.
                  => handle as error
               [EINTR]
                  A signal interrupted this function.
                  => handle as normal behavior, do a while loop
               [EINVAL]
                  The sem argument does not refer to a valid semaphore.
                  => a valid semaphore is provided
               [ETIMEDOUT]
                  The call timed out before the semaphore could be locked.
                  => valid for sem_timedwait() only, no need to handle
             *
             */

            int errNumber = errno;

            if(EINTR != errNumber)
            {
               // error
               ETG_TRACE_ERR((" wait(): sem_wait failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            }
         }
         else
         {
            // success
            errCode = BTS_OK;
            break;
         }
      }
   }
   else
   {
      struct timespec deadLine;

      // get current time
      if(-1 == clock_gettime(CLOCK_REALTIME, &deadLine))
      {
         // check result for debugging purpose

         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/clock_getres.html:
            The clock_getres(), clock_gettime(), and clock_settime() functions shall fail if:

            [EINVAL]
               The clock_id argument does not specify a known clock.
               => valid clock id is given
          *
          */

         int errNumber = errno;
         ETG_TRACE_ERR((" wait(): clock_gettime failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
         FW_NORMAL_ASSERT_ALWAYS();
      }

      // update deadLine with wait duration
      deadLine.tv_sec += timeoutInMs / 1000;

      if(1000000000L <= deadLine.tv_nsec)
      {
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();
         deadLine.tv_sec += 1;
         deadLine.tv_nsec = 0;
      }
      else if(0 > deadLine.tv_nsec)
      {
         // should never happen
         FW_NORMAL_ASSERT_ALWAYS();
         deadLine.tv_nsec = 0;
      }
      else
      {
         long int diff = 1000000000L - deadLine.tv_nsec; // max 1000000000L
         long int nsec = (timeoutInMs % 1000) * 1000000L; // max 999000000L

         if(nsec >= diff)
         {
            // overflow
            deadLine.tv_sec += 1;
            deadLine.tv_nsec = (nsec - diff);
         }
         else
         {
            // no overflow
            deadLine.tv_nsec += nsec;
         }
      }

      for(;;)
      {
         if(-1 == sem_timedwait(&_semaphore, &deadLine))
         {
            /*
             * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_timedwait.html:
               The sem_timedwait() function shall fail if:

               [EINVAL]
                 The process or thread would have blocked, and the abs_timeout parameter specified a nanoseconds field value less than zero or greater than or equal to 1000 million.
                 => nanoseconds field is in valid range
               [ETIMEDOUT]
                 The semaphore could not be locked before the specified timeout expired.
                 => timeout occurred => handle as expected error

               The sem_timedwait() function may fail if:

               [EDEADLK]
                 A deadlock condition was detected.
                 => handle as error
               [EINTR]
                 A signal interrupted this function.
                 => handle as normal behavior, do a while loop
               [EINVAL]
                 The sem argument does not refer to a valid semaphore.
                 => a valid semaphore is provided
             *
             */

            int errNumber = errno;

            if(EINTR == errNumber)
            {
               // continue and try again
            }
            else if(ETIMEDOUT == errNumber)
            {
               // timeout happened
               ETG_TRACE_ERR((" wait(): sem_timedwait failed: ETIMEDOUT"));
               errCode = BTS_TIMEOUT_ERROR;
               break;
            }
            else
            {
               // error
               ETG_TRACE_ERR((" wait(): sem_timedwait failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
               FW_NORMAL_ASSERT_ALWAYS();
               break;
            }
         }
         else
         {
            // success
            errCode = BTS_OK;
            break;
         }
      }
   }

#ifdef ENABLE_PRINT_SEMAPHORE
   printSemaphoreCount();
#endif

   return errCode;
}

void TimeoutSemaphore::tryWait(void)
{
   if(true == _resetDone)
   {
      return;
   }

   // semaphore could be in post state => try to wait
   for(;;)
   {
      if(-1 == sem_trywait(&_semaphore))
      {
         /*
          * from http://pubs.opengroup.org/onlinepubs/009695399/functions/sem_trywait.html:
            The sem_trywait() and sem_wait() functions shall fail if:

            [EAGAIN]
              The semaphore was already locked, so it cannot be immediately locked by the sem_trywait() operation ( sem_trywait() only).
              => handle as expected error

            The sem_trywait() and sem_wait() functions may fail if:

            [EDEADLK]
              A deadlock condition was detected.
               => handle as error
            [EINTR]
              A signal interrupted this function.
               => handle as normal behavior, do a while loop
            [EINVAL]
              The sem argument does not refer to a valid semaphore.
               => a valid semaphore is provided
          *
          */

         int errNumber = errno;

         if(EINTR == errNumber)
         {
            // continue and try again
         }
         else if(EAGAIN == errNumber)
         {
            // semaphore already locked
            ETG_TRACE_USR3((" tryWait(): sem_trywait failed: EAGAIN"));
            break;
         }
         else
         {
            // error
            ETG_TRACE_ERR((" tryWait(): sem_trywait failed: ERROR=%d (%s)", errNumber, strerror(errNumber)));
            break;
         }
      }
      else
      {
         // success
         break;
      }
   }

#ifdef ENABLE_PRINT_SEMAPHORE
   printSemaphoreCount();
#endif

   _resetDone = true;
}

void TimeoutSemaphore::reset(void)
{
   _resetDone = false;
}

void TimeoutSemaphore::printSemaphoreCount(void)
{
   int semCount = 0;
   if(-1 == sem_getvalue(&_semaphore, &semCount))
   {
      int errNumber = errno;
      ETG_TRACE_ERR((" printSemaphoreCount(): [this=0x%08X]: sem_getvalue failed: ERROR=%d (%s)", (uintptr_t)this, errNumber, strerror(errNumber)));
      FW_NORMAL_ASSERT_ALWAYS();
   }
   else
   {
      ETG_TRACE_USR1((" printSemaphoreCount(): [this=0x%08X]: semCount=%d", (uintptr_t)this, semCount));
   }
}

} //btstackif
