/**
 * @file TTYDeviceMonitor.cpp
 *
 * @par SW-Component
 * Main
 *
 * @brief TTY device monitor.
 *
 * @copyright (C) 2018 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of TTY device monitor.
 */

#include "TTYDeviceMonitor.h"
#include "FwSingleThread.h"
#include "TraceBase.h"
#include "ITTYDeviceMonitorHandler.h"
#include "TraceClasses.h"
#include "FwAssert.h"
#include "FwTrace.h"

#include <cstring>
#include <sys/select.h>
#include <errno.h>
#include <unistd.h>
#include <libudev.h>

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_CONTROL
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TTYDeviceMonitor.cpp.trc.h"
#endif
#endif

namespace btstackif {

TTYDeviceMonitor::TTYDeviceMonitor() :
_singleThread(0),
_terminateWorkerThread(false),
_workerThreadTerminated(true),
_timeoutInMs(100), // max allowed timeout value is 999ms
_handlerList(),
_lock()
{
}

TTYDeviceMonitor::TTYDeviceMonitor(const TTYDeviceMonitor& ref) :
_singleThread(0),
_terminateWorkerThread(false),
_workerThreadTerminated(true),
_timeoutInMs(100), // max allowed timeout value is 999ms
_handlerList(),
_lock()
{
   // ignore given parameter
   (void)(ref);

   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();
}

TTYDeviceMonitor& TTYDeviceMonitor::operator=(const TTYDeviceMonitor& ref)
{
   // DO NOT USE!!!
   FW_NORMAL_ASSERT_ALWAYS();

   if(this == &ref)
   {
      return *this;
   }

   // ignore given parameter

   return *this;
}

TTYDeviceMonitor::~TTYDeviceMonitor()
{
   if(_singleThread)
   {
      delete _singleThread;
   }
   _singleThread = 0;
}

void TTYDeviceMonitor::threadFunction(void* arguments)
{
   (void)(arguments);

   ThreadInfo threadInfo(this);
   ETG_TRACE_USR1((" Do(): %s --- thread start", threadInfo.getInfo()));

   while(false == _terminateWorkerThread)
   {
      monitorUdev();
   }

   ETG_TRACE_USR1((" Do(): %s --- thread end", threadInfo.getInfo()));

   _workerThreadTerminated = true;
}

void TTYDeviceMonitor::setTerminate(void* arguments)
{
   (void)(arguments);

   _terminateWorkerThread = true;
}

void TTYDeviceMonitor::start(void)
{
   if(0 == _singleThread)
   {
      _singleThread = new ::fw::SingleThread();
   }
   FW_NORMAL_ASSERT(0 != _singleThread);

   if(_singleThread)
   {
      _terminateWorkerThread = false;
      _workerThreadTerminated = false;
      const ::std::string name("BTS_TTY_MONITOR");
      _singleThread->start(this, name, 0);
   }
}

void TTYDeviceMonitor::stop(void)
{
   if(true != _workerThreadTerminated)
   {
      _terminateWorkerThread = true;

      if(_singleThread)
      {
         _singleThread->stop();
      }
   }
}

void TTYDeviceMonitor::addHandler(IN ITTYDeviceMonitorHandler* handler, IN const ::std::string& filter)
{
   FW_IF_NULL_PTR_RETURN(handler);
   FW_IF_NULL_PTR_RETURN(filter.size());

   (void)_lock.lock();

   _handlerList[handler] = filter;

   _lock.unlock();
}

void TTYDeviceMonitor::monitorUdev(void)
{
   // create the udev object
   struct udev* udevObj(udev_new());

   if(0 == udevObj)
   {
      // failed
      ETG_TRACE_ERR((" monitorUdev: udev_new failed"));
   }
   else
   {
      // create the udev monitor
      struct udev_monitor* udevMon(udev_monitor_new_from_netlink(udevObj, "udev"));

      if(0 == udevMon)
      {
         ETG_TRACE_ERR((" monitorUdev: udev_monitor_new_from_netlink failed"));
      }
      else
      {
         if(0 > udev_monitor_filter_add_match_subsystem_devtype(udevMon, "tty", 0))
         {
            ETG_TRACE_ERR((" monitorUdev: udev_monitor_filter_add_match_subsystem_devtype failed"));
         }
         else
         {
            if(0 > udev_monitor_enable_receiving(udevMon))
            {
               ETG_TRACE_ERR((" monitorUdev: udev_monitor_enable_receiving failed"));
            }
            else
            {
               // get the file descriptor (fd) for the monitor; this fd will get passed to select()
               int fd = udev_monitor_get_fd(udevMon);

               if(0 > fd)
               {
                  ETG_TRACE_ERR((" monitorUdev: udev_monitor_get_fd failed"));
               }
               else
               {
                  while(false == _terminateWorkerThread)
                  {
                     fd_set fds;
                     struct timeval tv;

                     FD_ZERO(&fds);
                     FD_SET(fd, &fds);
                     tv.tv_sec = 0;
                     tv.tv_usec = _timeoutInMs * 1000L;

                     const int ret = select(fd+1, &fds, 0, 0, &tv);

                     // check if our file descriptor has received data
                     if(0 == ret)
                     {
                        // timeout
                     }
                     else if(0 > ret)
                     {
                        const int errorCode(errno);

                        if(EINTR == errorCode)
                        {
                           ETG_TRACE_USR2((" monitorUdev: select failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
                        }
                        else
                        {
                           ETG_TRACE_ERR((" monitorUdev: select failed: ERROR=%d (%s)", errorCode, strerror(errorCode)));
                        }
                     }
                     else if(FD_ISSET(fd, &fds))
                     {
                        // make the call to receive the device; select() ensured that this will not block
                        struct udev_device* udevDev(udev_monitor_receive_device(udevMon));

                        if(0 == udevDev)
                        {
                           ETG_TRACE_ERR((" monitorUdev: udev_monitor_receive_device failed"));
                        }
                        else
                        {
                           const char* devnode(udev_device_get_devnode(udevDev));
                           const char* subsystem(udev_device_get_subsystem(udevDev));
                           const char* action(udev_device_get_action(udevDev));

                           ETG_TRACE_USR4((" monitorUdev: got device"));
                           ETG_TRACE_USR4((" monitorUdev:    Node: %s", (devnode ? devnode : "<ERROR>")));
                           ETG_TRACE_USR4((" monitorUdev:    Subsystem: %s", (subsystem ? subsystem : "<ERROR>")));
                           ETG_TRACE_USR4((" monitorUdev:    Action: %s", (action ? action : "<ERROR>")));

                           checkReportedDevice(devnode, action);

                           udev_device_unref(udevDev);
                        }
                     }
                  }
               }
            }
         }

         // destroy and free udev monitor
         udev_monitor_unref(udevMon);
      }

      // destroy and free udev object
      udev_unref(udevObj);
   }
}

void TTYDeviceMonitor::checkReportedDevice(const char* device, const char* action)
{
   FW_IF_NULL_PTR_RETURN(device);
   FW_IF_NULL_PTR_RETURN(action);

   const ::std::string deviceName(device);
   const ::std::string actionName(action);

   bool added;

   if(0 == actionName.compare("add"))
   {
      added = true;
   }
   else if(0 == actionName.compare("remove"))
   {
      added = false;
   }
   else
   {
      return;
   }

   // ETG_TRACE_USR2((" checkReportedDevice: device=%50s action=%s", deviceName.c_str(), actionName.c_str()));

   (void)_lock.lock();

   for(::std::map< ITTYDeviceMonitorHandler*, ::std::string >::const_iterator it = _handlerList.begin(); it != _handlerList.end(); ++it)
   {
      if(0 != it->first)
      {
         if(::std::string::npos != deviceName.find(it->second))
         {
            (it->first)->handleAddedRemovedCharacterDevice(deviceName, added);
         }
      }
   }

   _lock.unlock();
}

} //btstackif
