/**
 * @file BtsTimerPool.cpp
 *
 * @par SW-Component
 * Timer
 *
 * @brief TimerPool.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Timer pool handling.
 */

#include "BtsTimerPool.h"
#include "BtsTimer.h"
#include "ExtendedTimer.h"

namespace btstackif {

TimerPool::TimerPool() :
_count(25),
_extendedCount(100),
_created(false),
_list(),
_extendedList(),
_lock()
{
}

TimerPool::~TimerPool()
{
   destroy();
}

ITimerPool& TimerPool::getInstance(void)
{
   static TimerPool instance;
   return instance;
}

void TimerPool::create(void)
{
   _lock.lock();

   if(false == _created)
   {
      _list.reserve(_count);
      _list.resize(_count);

      for(unsigned int i = 0; i < _list.size(); i++)
      {
         _list[i].timer = new Timer();
         _list[i].inUse = false;
      }

      _extendedList.reserve(_extendedCount);
      _extendedList.resize(_extendedCount);

      for(unsigned int i = 0; i < _extendedList.size(); i++)
      {
         _extendedList[i].setTimerPool(this);
         _extendedList[i].setTimer(new ExtendedTimer());
         _extendedList[i].setInUse(false);
      }

      _created = true;
   }

   _lock.unlock();
}

void TimerPool::destroy(void)
{
   _lock.lock();

   if(true == _created)
   {
      for(unsigned int i = 0; i < _list.size(); i++)
      {
         if(0 != _list[i].timer)
         {
            _list[i].timer->stop();
            delete _list[i].timer;
         }
      }

      _list.clear();

      for(unsigned int i = 0; i < _extendedList.size(); i++)
      {
         if(0 != _extendedList[i].getTimer())
         {
            _extendedList[i].stop();
            delete _extendedList[i].getTimer();
         }
      }

      _extendedList.clear();

      _created = false;
   }

   _lock.unlock();
}

Timer* TimerPool::getTimer(void)
{
   Timer* returnTimer = 0;
   bool timerAvailable = false;

   _lock.lock();

   for(unsigned int i = 0; i < _list.size(); i++)
   {
      if(false == _list[i].inUse)
      {
         returnTimer = _list[i].timer;
         _list[i].inUse = true;
         timerAvailable = true;
         break;
      }
   }

   if(false == timerAvailable)
   {
      TimerEntry entry;
      entry.timer = new Timer();
      entry.inUse = true;
      _list.push_back(entry);
      returnTimer = entry.timer;
   }

   _lock.unlock();

   return returnTimer;
}

void TimerPool::releaseTimer(IN Timer* timer)
{
   if(0 == timer)
   {
      return;
   }

   _lock.lock();

   for(unsigned int i = 0; i < _list.size(); i++)
   {
      if(timer == _list[i].timer)
      {
         _list[i].timer->stop();
         _list[i].inUse = false;
         break;
      }
   }

   _lock.unlock();
}

IExtendedTimer* TimerPool::getExtendedTimer(void)
{
   IExtendedTimer* returnTimer = 0;
   bool timerAvailable = false;

   _lock.lock();

   for(unsigned int i = 0; i < _extendedList.size(); i++)
   {
      if(false == _extendedList[i].getInUse())
      {
         returnTimer = _extendedList[i].getTimer();
         _extendedList[i].setInUse(true);
         timerAvailable = true;
         break;
      }
   }

   if(false == timerAvailable)
   {
      ExtendedTimerEntry entry;
      entry.setTimer(new ExtendedTimer());
      entry.setInUse(true);
      _extendedList.push_back(entry);
      returnTimer = entry.getTimer();
   }

   _lock.unlock();

   return returnTimer;
}

void TimerPool::releaseExtendedTimer(IN IExtendedTimer* timer)
{
   if(0 == timer)
   {
      return;
   }

   _lock.lock();

   for(unsigned int i = 0; i < _extendedList.size(); i++)
   {
      if(timer == _extendedList[i].getTimer())
      {
         _extendedList[i].stop();
         _extendedList[i].setInUse(false);
         break;
      }
   }

   _lock.unlock();
}

void TimerPool::getTimerCount(unsigned int& nmbTimer, unsigned int& nmbUsedTimer, unsigned int& nmbExtendedTimer, unsigned int& nmbUsedExtendedTimer)
{
   _lock.lock();

   nmbTimer = (unsigned int)_list.size();

   size_t count(0);
   for(size_t i = 0; i < _list.size(); i++)
   {
      if(true == _list[i].inUse)
      {
         ++count;
      }
   }

   nmbUsedTimer = (unsigned int)count;

   nmbExtendedTimer = (unsigned int)_extendedList.size();

   count = 0;
   for(size_t i = 0; i < _extendedList.size(); i++)
   {
      if(true == _extendedList[i].getInUse())
      {
         ++count;
      }
   }

   nmbUsedExtendedTimer = (unsigned int)count;

   _lock.unlock();
}

} //btstackif
