/**
 * @file BtsTimer.cpp
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Timer handling. This timer handling is not thread safe.
 */

#include "BtsTimer.h"
#include "FwAssert.h"

namespace btstackif {

Timer::Timer()
{
   _timeout = 0;
   _callback = NULL;
   _timeoutHandler = NULL;
   _id = 0;
   _cyclic = false;
}

Timer::~Timer()
{
   stop();
   _callback = NULL;
   _timeoutHandler = NULL;
}

void Timer::start(void)
{
   // timeout and callback must be set before
   if(0 == _timeout)
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }
   if((NULL == _callback) && (NULL == _timeoutHandler))
   {
      FW_NORMAL_ASSERT_ALWAYS();
   }

   bool currentActive = _active;
   _active = true;

   // if timer is already active we will restart
   _id = addTimer(_context, _id, _timeout, _callback, _timeoutHandler, _cyclic, currentActive);
}

void Timer::start(IN const bool cyclic)
{
   setCyclicFlag(cyclic);
   start();
}

void Timer::start(IN const BTSTimeValue timeout)
{
   setTimeout(timeout);
   start();
}

void Timer::start(IN const BTSTimeValue timeout, IN const bool cyclic)
{
   setTimeout(timeout);
   setCyclicFlag(cyclic);
   start();
}

void Timer::start(IN BTSTimerCallBack callback)
{
   setCallback(callback);
   start();
}

void Timer::start(IN ITimeoutHandler* handler)
{
   setTimeoutHandler(handler);
   start();
}

void Timer::start(IN BTSTimerCallBack callback, IN const bool cyclic)
{
   setCallback(callback);
   setCyclicFlag(cyclic);
   start();
}

void Timer::start(IN ITimeoutHandler* handler, IN const bool cyclic)
{
   setTimeoutHandler(handler);
   setCyclicFlag(cyclic);
   start();
}

void Timer::start(IN const BTSTimeValue timeout, IN BTSTimerCallBack callback)
{
   setTimeout(timeout);
   setCallback(callback);
   start();
}

void Timer::start(IN const BTSTimeValue timeout, IN ITimeoutHandler* handler)
{
   setTimeout(timeout);
   setTimeoutHandler(handler);
   start();
}

void Timer::start(IN const BTSTimeValue timeout, IN BTSTimerCallBack callback, IN const bool cyclic)
{
   setTimeout(timeout);
   setCallback(callback);
   setCyclicFlag(cyclic);
   start();
}

void Timer::start(IN const BTSTimeValue timeout, IN ITimeoutHandler* handler, IN const bool cyclic)
{
   setTimeout(timeout);
   setTimeoutHandler(handler);
   setCyclicFlag(cyclic);
   start();
}

void Timer::stop(void)
{
   if(true == _active)
   {
      // use internal timer id to avoid any problem (I hope that no one will reset callback function while timer is active)
      removeTimer(_context, _id);

      // do not reset timer id because it will be reused

      _active = false;
   }
}

} //btstackif
