/**
 * @file BtStackInternalTypes.cpp
 *
 * @par SW-Component
 * Types
 *
 * @brief Basic types.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of all internal type classes.
 */

#include "BtStackInternalTypes.h"
#include "BtsUtils.h"
#include "FwAssert.h"

namespace btstackif {

BTSApp2BtsMessageCompareItem::BTSApp2BtsMessageCompareItem()
{
   opCode = App2BtsOC_Last;
   // deviceAddress
   protocolId = BTS_PROTO_LAST;
   // sppUuid
   // masInstance
   searchType = BTS_SEARCH_LAST;
}

BTSApp2BtsMessageCompareItem::BTSApp2BtsMessageCompareItem(IN const BTSApp2BtsMessageCompareItem& ref)
{
   opCode = ref.opCode;
   deviceAddress = ref.deviceAddress;
   protocolId = ref.protocolId;
   sppUuid = ref.sppUuid;
   masInstance = ref.masInstance;
   searchType = ref.searchType;
}

BTSApp2BtsMessageCompareItem& BTSApp2BtsMessageCompareItem::operator=(IN const BTSApp2BtsMessageCompareItem& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   opCode = ref.opCode;
   deviceAddress = ref.deviceAddress;
   protocolId = ref.protocolId;
   sppUuid = ref.sppUuid;
   masInstance = ref.masInstance;
   searchType = ref.searchType;

   return *this;
}

bool BTSApp2BtsMessageCompareItem::operator==(IN const BTSApp2BtsMessageCompareItem& ref) const
{
   bool result = true;

   result = (true == result) && (opCode == ref.opCode);
   result = (true == result) && (deviceAddress == ref.deviceAddress);
   result = (true == result) && (protocolId == ref.protocolId);
   result = (true == result) && (sppUuid == ref.sppUuid);
   result = (true == result) && (masInstance == ref.masInstance);
   result = (true == result) && (searchType == ref.searchType);

   return result;
}

bool BTSApp2BtsMessageCompareItem::operator!=(IN const BTSApp2BtsMessageCompareItem& ref) const
{
   return !(operator==(ref));
}

BTSApp2BtsMessageCompareItem::~BTSApp2BtsMessageCompareItem()
{
}

void BTSApp2BtsMessageCompareItem::reset(void)
{
   opCode = App2BtsOC_Last;
   deviceAddress.clear();
   protocolId = BTS_PROTO_LAST;
   sppUuid.clear();
   masInstance.clear();
   searchType = BTS_SEARCH_LAST;
}

//------------------------------------------------------------------------------

BTSApp2BtsMessageMasterCompareItem::BTSApp2BtsMessageMasterCompareItem()
{
   compareOpCode = true;
   compareDeviceAddress = false;
   compareProtocolId = false;
   compareSppUuid = false;
   compareMasInstance = false;
   compareSearchType = false;
}

BTSApp2BtsMessageMasterCompareItem::BTSApp2BtsMessageMasterCompareItem(IN const BTSApp2BtsMessageMasterCompareItem& ref)
: BTSApp2BtsMessageCompareItem(ref)
{
   compareOpCode = ref.compareOpCode;
   compareDeviceAddress = ref.compareDeviceAddress;
   compareProtocolId = ref.compareProtocolId;
   compareSppUuid = ref.compareSppUuid;
   compareMasInstance = ref.compareMasInstance;
   compareSearchType = ref.compareSearchType;
}

BTSApp2BtsMessageMasterCompareItem& BTSApp2BtsMessageMasterCompareItem::operator=(IN const BTSApp2BtsMessageMasterCompareItem& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   BTSApp2BtsMessageCompareItem::operator=(ref);

   compareOpCode = ref.compareOpCode;
   compareDeviceAddress = ref.compareDeviceAddress;
   compareProtocolId = ref.compareProtocolId;
   compareSppUuid = ref.compareSppUuid;
   compareMasInstance = ref.compareMasInstance;
   compareSearchType = ref.compareSearchType;

   return *this;
}

bool BTSApp2BtsMessageMasterCompareItem::operator==(IN const BTSApp2BtsMessageMasterCompareItem& ref) const
{
   bool result = BTSApp2BtsMessageCompareItem::operator==(ref);

   result = (true == result) && (compareOpCode == ref.compareOpCode);
   result = (true == result) && (compareDeviceAddress == ref.compareDeviceAddress);
   result = (true == result) && (compareProtocolId == ref.compareProtocolId);
   result = (true == result) && (compareSppUuid == ref.compareSppUuid);
   result = (true == result) && (compareMasInstance == ref.compareMasInstance);
   result = (true == result) && (compareSearchType == ref.compareSearchType);

   return result;
}

bool BTSApp2BtsMessageMasterCompareItem::operator!=(IN const BTSApp2BtsMessageMasterCompareItem& ref) const
{
   return !(operator==(ref));
}

BTSApp2BtsMessageMasterCompareItem::~BTSApp2BtsMessageMasterCompareItem()
{
}

bool BTSApp2BtsMessageMasterCompareItem::isMatch(IN const BTSApp2BtsMessageCompareItem& ref) const
{
   bool result = true;

   if(true == compareOpCode)
   {
      result = (true == result) && (opCode == ref.opCode);
   }
   if(true == compareDeviceAddress)
   {
      result = (true == result) && (deviceAddress == ref.deviceAddress);
   }
   if(true == compareProtocolId)
   {
      result = (true == result) && (protocolId == ref.protocolId);
   }
   if(true == compareSppUuid)
   {
      result = (true == result) && (sppUuid == ref.sppUuid);
   }
   if(true == compareMasInstance)
   {
      result = (true == result) && (masInstance == ref.masInstance);
   }
   if(true == compareSearchType)
   {
      result = (true == result) && (searchType == ref.searchType);
   }

   return result;
}

//------------------------------------------------------------------------------

BTSApp2BtsMessageRequestItem::BTSApp2BtsMessageRequestItem() :
item(),
user(0),
handle(0)
{
}

BTSApp2BtsMessageRequestItem::BTSApp2BtsMessageRequestItem(IN const BTSApp2BtsMessageRequestItem& ref)
{
   item = ref.item;
   user = 0;
   setUser(ref.user);
   handle = ref.handle;
}

BTSApp2BtsMessageRequestItem& BTSApp2BtsMessageRequestItem::operator=(IN const BTSApp2BtsMessageRequestItem& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   item = ref.item;
   user = 0;
   setUser(ref.user);
   handle = ref.handle;

   return *this;
}

bool BTSApp2BtsMessageRequestItem::operator==(IN const BTSApp2BtsMessageRequestItem& ref) const
{
   bool result = true;

   result = (true == result) && (item == ref.item);
   result = (true == result) && (user == ref.user);
   result = (true == result) && (handle == ref.handle);

   return result;
}

bool BTSApp2BtsMessageRequestItem::operator!=(IN const BTSApp2BtsMessageRequestItem& ref) const
{
   return !(operator==(ref));
}

BTSApp2BtsMessageRequestItem::~BTSApp2BtsMessageRequestItem()
{
   user = 0;
}

void BTSApp2BtsMessageRequestItem::reset(void)
{
   item.reset();
   user = 0;
   handle = 0;
}

//------------------------------------------------------------------------------

BTSHandleIpc2BtsMessageItem::BTSHandleIpc2BtsMessageItem() :
item(),
deleteMessage(false),
message(0),
checkIf(0)
{
}

BTSHandleIpc2BtsMessageItem::BTSHandleIpc2BtsMessageItem(IN const BTSHandleIpc2BtsMessageItem& ref)
{
   item = ref.item;
   deleteMessage = ref.deleteMessage;
   message = 0;
   setMessage(ref.message);
   checkIf = 0;
   setCheckIf(ref.checkIf);
}

BTSHandleIpc2BtsMessageItem& BTSHandleIpc2BtsMessageItem::operator=(IN const BTSHandleIpc2BtsMessageItem& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   item = ref.item;
   deleteMessage = ref.deleteMessage;
   message = 0;
   setMessage(ref.message);
   checkIf = 0;
   setCheckIf(ref.checkIf);

   return *this;
}

bool BTSHandleIpc2BtsMessageItem::operator==(IN const BTSHandleIpc2BtsMessageItem& ref) const
{
   bool result = true;

   result = (true == result) && (item == ref.item);
   result = (true == result) && (deleteMessage == ref.deleteMessage);
   result = (true == result) && (message == ref.message);
   result = (true == result) && (checkIf == ref.checkIf);

   return result;
}

bool BTSHandleIpc2BtsMessageItem::operator!=(IN const BTSHandleIpc2BtsMessageItem& ref) const
{
   return !(operator==(ref));
}

BTSHandleIpc2BtsMessageItem::~BTSHandleIpc2BtsMessageItem()
{
   message = 0;
   checkIf = 0;
}

void BTSHandleIpc2BtsMessageItem::reset(void)
{
   item.reset();
   deleteMessage = false;
   message = 0;
   checkIf = 0;
}

//------------------------------------------------------------------------------

BTSProtocolBaseEntry::BTSProtocolBaseEntry()
{
   // deviceAddress
   protocolId = BTS_PROTO_LAST;
   // sppUuid
   masInstanceId = 0;
}

BTSProtocolBaseEntry::BTSProtocolBaseEntry(IN const BTSProtocolBaseEntry& ref)
{
   deviceAddress = ref.deviceAddress;
   protocolId = ref.protocolId;
   sppUuid = ref.sppUuid;
   masInstanceId = ref.masInstanceId;
}

BTSProtocolBaseEntry& BTSProtocolBaseEntry::operator=(IN const BTSProtocolBaseEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   deviceAddress = ref.deviceAddress;
   protocolId = ref.protocolId;
   sppUuid = ref.sppUuid;
   masInstanceId = ref.masInstanceId;

   return *this;
}

bool BTSProtocolBaseEntry::operator==(IN const BTSProtocolBaseEntry& ref) const
{
   return (0 == compare(ref));
}

bool BTSProtocolBaseEntry::operator!=(IN const BTSProtocolBaseEntry& ref) const
{
   return (0 != compare(ref));
}

bool BTSProtocolBaseEntry::operator<(const BTSProtocolBaseEntry& ref) const
{
   return (0 > compare(ref));
}

bool BTSProtocolBaseEntry::operator>(const BTSProtocolBaseEntry& ref) const
{
   return (0 < compare(ref));
}

BTSProtocolBaseEntry::~BTSProtocolBaseEntry()
{
}

int BTSProtocolBaseEntry::compare(const BTSProtocolBaseEntry& ref) const
{
   int result;

   result = deviceAddress.compare(ref.deviceAddress);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // device address is equal

   if(protocolId < ref.protocolId)
   {
      return -1;
   }
   else if(protocolId > ref.protocolId)
   {
      return 1;
   }

   // protocol id is equal

   result = sppUuid.compare(ref.sppUuid);
   if(result < 0)
   {
      return -1;
   }
   else if(result > 0)
   {
      return 1;
   }

   // SPP uuid is equal

   if(masInstanceId < ref.masInstanceId)
   {
      return -1;
   }
   else if(masInstanceId > ref.masInstanceId)
   {
      return 1;
   }

   // MAS instance id is equal

   return 0;
}

//------------------------------------------------------------------------------

BTSDbusInterfaceItem::BTSDbusInterfaceItem()
{
   dbusInterface = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   busType = BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE;
   // busName
   // objPath
}

BTSDbusInterfaceItem::BTSDbusInterfaceItem(IN const BTSDbusInterfaceItem& ref)
{
   dbusInterface = ref.dbusInterface;
   busType = ref.busType;
   busName = ref.busName;
   objPath = ref.objPath;
}

BTSDbusInterfaceItem& BTSDbusInterfaceItem::operator=(IN const BTSDbusInterfaceItem& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   dbusInterface = ref.dbusInterface;
   busType = ref.busType;
   busName = ref.busName;
   objPath = ref.objPath;

   return *this;
}

bool BTSDbusInterfaceItem::operator==(IN const BTSDbusInterfaceItem& ref) const
{
   bool result = true;

   result = (true == result) && (dbusInterface == ref.dbusInterface);
   result = (true == result) && (busType == ref.busType);
   result = (true == result) && (busName == ref.busName);
   result = (true == result) && (objPath == ref.objPath);

   return result;
}

bool BTSDbusInterfaceItem::operator!=(IN const BTSDbusInterfaceItem& ref) const
{
   return !(operator==(ref));
}

BTSDbusInterfaceItem::~BTSDbusInterfaceItem()
{
}

//------------------------------------------------------------------------------

BTSBDAddressTrace::BTSBDAddressTrace() :
_stringAddress(),
_binaryAddress(),
_allowEmpty(false)
{
   convertBdAddressString2Binary(_binaryAddress, _stringAddress, _allowEmpty);
}

BTSBDAddressTrace::BTSBDAddressTrace(IN const BTSBDAddress& address, IN const bool allowEmpty /*= false*/) :
_stringAddress(address),
_binaryAddress(),
_allowEmpty(allowEmpty)
{
   convertBdAddressString2Binary(_binaryAddress, _stringAddress, _allowEmpty);
}

BTSBDAddressTrace::BTSBDAddressTrace(IN const BTSBDAddressTrace& ref) :
_stringAddress(ref._stringAddress),
_binaryAddress(ref._binaryAddress),
_allowEmpty(ref._allowEmpty)
{
}

BTSBDAddressTrace& BTSBDAddressTrace::operator=(IN const BTSBDAddressTrace& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   _stringAddress = ref._stringAddress;
   _binaryAddress = ref._binaryAddress;
   _allowEmpty = ref._allowEmpty;

   return *this;
}

bool BTSBDAddressTrace::operator==(IN const BTSBDAddressTrace& ref) const
{
   bool result = true;

   result = (true == result) && (_stringAddress == ref._stringAddress);
   result = (true == result) && (_binaryAddress == ref._binaryAddress);
   result = (true == result) && (_allowEmpty == ref._allowEmpty);

   return result;
}

bool BTSBDAddressTrace::operator!=(IN const BTSBDAddressTrace& ref) const
{
   return !(operator==(ref));
}

BTSBDAddressTrace::~BTSBDAddressTrace()
{
}

uint32_t BTSBDAddressTrace::getUpper(void) const
{
   if(6 == _binaryAddress.size())
   {
      return (((uint32_t)_binaryAddress[0]) << 24) | (((uint32_t)_binaryAddress[1]) << 16) | (((uint32_t)_binaryAddress[2]) << 8) | (((uint32_t)_binaryAddress[3]));
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();
      return 0;
   }
}

uint16_t BTSBDAddressTrace::getLower(void) const
{
   if(6 == _binaryAddress.size())
   {
      return (uint16_t)((((uint32_t)_binaryAddress[4]) << 8) | ((uint32_t)_binaryAddress[5]));
   }
   else
   {
      // wrong size
      FW_NORMAL_ASSERT_ALWAYS();
      return 0;
   }
}

//------------------------------------------------------------------------------

BTSUserCallbackEntry::BTSUserCallbackEntry() :
callback(0),
stackInterface(BTS_IF_UNDEFINED),
subComponent(BTS_FB_NONE)
{
}

BTSUserCallbackEntry::BTSUserCallbackEntry(IN BtStackIfCallback* cb, IN const BTSInterfaceType interface, IN const BTSFunctionBlock component) :
callback(cb),
stackInterface(interface),
subComponent(component)
{
}

BTSUserCallbackEntry::BTSUserCallbackEntry(IN const BTSUserCallbackEntry& ref) :
callback(ref.callback),
stackInterface(ref.stackInterface),
subComponent(ref.subComponent)
{
}

BTSUserCallbackEntry& BTSUserCallbackEntry::operator=(IN const BTSUserCallbackEntry& ref)
{
   if(this == &ref)
   {
      return *this;
   }

   callback = ref.callback;
   stackInterface = ref.stackInterface;
   subComponent = ref.subComponent;

   return *this;
}

bool BTSUserCallbackEntry::operator==(IN const BTSUserCallbackEntry& ref) const
{
   bool result = true;

   result = (true == result) && (callback == ref.callback);
   result = (true == result) && (stackInterface == ref.stackInterface);
   result = (true == result) && (subComponent == ref.subComponent);

   return result;
}

bool BTSUserCallbackEntry::operator!=(IN const BTSUserCallbackEntry& ref) const
{
   return !(operator==(ref));
}

BTSUserCallbackEntry::~BTSUserCallbackEntry()
{
}

//------------------------------------------------------------------------------

const BTSComponentEntry& getDbusComponentKey(IN BTSComponentEntry& entry, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   (void)(component);

   /*
    * we have to check sub component:
    * - current design is to have 1 DBUS interface for same stack interface => independent of sub component
    * - for WBL and ECNR separate DBUS interfaces are planned
    */
   switch(subComponent)
   {
      case BTS_FB_CONFIG:
      case BTS_FB_CONNECTION:
      case BTS_FB_TELEPHONY:
      case BTS_FB_PHONEBOOK:
      case BTS_FB_MESSAGING:
      case BTS_FB_MEDIAPLAYER:
      {
         entry.component = BTS_FB_NONE;
         entry.stackInterface = stackInterface;
         entry.subComponent = BTS_FB_NONE;
         break;
      }
      case BTS_FB_WBL:
      case BTS_FB_ECNR:
      {
         entry.component = BTS_FB_NONE;
         entry.stackInterface = BTS_IF_UNDEFINED;
         entry.subComponent = subComponent;
         break;
      }
      case BTS_FB_NONE:
      case BTS_FB_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return entry;
}

const BTSComponentEntry& getControlComponentKey(IN BTSComponentEntry& entry, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent)
{
   (void)(component);

   /*
    * note:
    * - current design is to have separate control handlers for each stack interface + sub component combination
    * - WBL and ECNR interfaces are independent of stack interface
    */
   switch(subComponent)
   {
      case BTS_FB_CONFIG:
      case BTS_FB_CONNECTION:
      case BTS_FB_TELEPHONY:
      case BTS_FB_PHONEBOOK:
      case BTS_FB_MESSAGING:
      case BTS_FB_MEDIAPLAYER:
      {
         entry.component = BTS_FB_NONE;
         entry.stackInterface = stackInterface;
         entry.subComponent = subComponent;
         break;
      }
      case BTS_FB_WBL:
      case BTS_FB_ECNR:
      {
         entry.component = BTS_FB_NONE;
         entry.stackInterface = BTS_IF_UNDEFINED;
         entry.subComponent = subComponent;
         break;
      }
      case BTS_FB_NONE:
      case BTS_FB_LAST:
      default:
         FW_NORMAL_ASSERT_ALWAYS();
         break;
   }

   return entry;
}

} //btstackif
