/**
 * @file TimerTickSource.h
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer tick source.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of timer tick source.
 */

#ifndef _TIMER_TICK_SOURCE_H_
#define _TIMER_TICK_SOURCE_H_

#include "BtStackInternalTypes.h"

// use or do not use nanosleep as "timer"
#define ENABLE_USAGE_NANOSLEEP
//#undef ENABLE_USAGE_NANOSLEEP

#ifdef ENABLE_USAGE_NANOSLEEP

#include <pthread.h>
#include <time.h>

#else

// use or do not use a separate thread to receive the sig action usr1 signal
//#define ENABLE_SIG_ACTION_THREAD
#undef ENABLE_SIG_ACTION_THREAD

#include <pthread.h>
#include <signal.h>
#include <time.h>
#include <limits.h>

#endif

namespace btstackif {

// class forward declarations
class ITimerTickHandler;

#ifdef ENABLE_USAGE_NANOSLEEP

/**
 *
 */
class TimerTickSource
{
public:
   TimerTickSource();
   virtual ~TimerTickSource();
   void start(IN ITimerTickHandler* tickHandler);
   void stop(void);

private:
   enum
   {
      MAX_TRIES_FOR_THREAD_CREATION = 3
   };

   const unsigned int _timeoutInMs;
   pthread_t _threadHandle;
   bool _threadCreated;
   bool _terminateThread;
   ITimerTickHandler* _tickHandler;

   static void* timerThread(void* context);
};

#else

/**
 *
 */
class TimerTickSource
{
public:
   TimerTickSource();
   virtual ~TimerTickSource();
   void start(void);
   void stop(void);

private:
   enum
   {
#ifdef ENABLE_SIG_ACTION_THREAD
      DEFAULT_STACKSIZE = (PTHREAD_STACK_MIN << 2),
      MAX_TRIES_FOR_THREAD_CREATION = 3,
#endif
      MAX_TRIES_FOR_TIMER_CREATION = 3
   };

   const unsigned int _timeoutInMs;
   bool _threadCreated;
#ifdef ENABLE_SIG_ACTION_THREAD
   pthread_attr_t _threadAttr;
   pthread_t _threadHandle;
   pid_t _threadID;
   static bool _threadStarted;
#endif
   struct sigaction _sigAction;
   timer_t _timerId;
   bool _timerCreated;
   bool _timerStarted;
   static TimerTickSource* _myExemplar;

   void createAndStartPeriodicTimer(void);
   void stopAndDeletePeriodicTimer(void);
#ifdef ENABLE_SIG_ACTION_THREAD
   static void* sigactionThread(void* context);
#endif
   static void defaultCallback(int signo, siginfo_t* info, void* context);
};

#endif

} //btstackif

#endif //_TIMER_TICK_SOURCE_H_
