/**
 * @file SppTestApp.h
 *
 * @par SW-Component
 * Test
 *
 * @brief SPP test application.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details SPP test application.
 */

#ifndef _SPP_TEST_APP_H_
#define _SPP_TEST_APP_H_

#include <string>

#include "FwLock.h"

namespace btstackif {

/**
 *
 */
class SppTestApp
{
public:
   /**
    * Get instance of SPP test application (reference).
    *
    * @return = reference to SPP test application object
    */
   static SppTestApp& getInstance(void);

   /**
    * Open device.
    *
    * @param[in] device: device name
    * @param[in] nmbSendingLoops: number of sending loops to be executed (0 means infinite)
    * @param[in] delayBetweenSending: delay between sending loops in ms
    */
   void openDevice(const ::std::string& device, const unsigned int nmbSendingLoops = 0, const unsigned int delayBetweenSending = 50);

   /**
    * Close device.
    */
   void closeDevice(void);

   /**
    * Enable/disable sending.
    *
    * @param[in] enable: enable flag
    */
   inline void enableSending(const bool enable) { _send = enable; }

   /**
    * Enable/disable raw mode.
    *
    * @param[in] enable: enable flag
    */
   inline void enableRawMode(const bool enable) { _rawMode = enable; }

private:
   /**
    * Default constructor. Private!!!
    */
   SppTestApp();

   /**
    * Destructor. Private!!!
    */
   virtual ~SppTestApp();

   /**
    * Device handler.
    *
    * @param[in] arg: argument
    *
    * @return = nothing
    */
   static void* deviceHandler(void* arg);

   /**
    * Print received data.
    *
    * @param[in] buffer: data
    * @param[in] length: length
    */
   void printData(unsigned char* buffer, size_t length);

   /**
    * Send data.
    */
   void sendData(void);

   /**
    * Send large data.
    */
   void sendLargeData(void);

   /**
    * Send message.
    *
    * @param[in] deviceFd: device descriptor
    * @param[in] msgLenTotal: message length
    * @param[in] msg: message
    */
   void sendMessage(int deviceFd, const size_t msgLenTotal, const unsigned char* msg);

   const int _defaultFd; /**< default value for file descriptor */
   int _fd; /**< file descriptor */
   unsigned int _defaultTimeout; /**< default value for poll timeout in ms */
   LockForeverAndNonReentrant _lock; /**< lock */
   bool _terminate; /**< terminate flag for thread */
   bool _send; /**< send flag */
   bool _rawMode; /**< flag for raw mode of character device */
   bool _toggle; /**< flag for switching between text and binary data */
   bool _dataMode; /**< flag for text or binary data */
   unsigned int _sendLoopCounter; /**< counter for send loops */
   unsigned int _maxSendLoopCounter; /**< maximum counter for send loops */
};

} //btstackif

#endif //_SPP_TEST_APP_H_
