/**
 * @file IStateMachineBase.h
 *
 * @par SW-Component
 * Interface
 *
 * @brief State machine interface.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Interface class for State machine.
 */

#ifndef _I_STATE_MACHINE_BASE_H_
#define _I_STATE_MACHINE_BASE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class Bts2App_BaseMessage;
class BtStackIfCallback;

/**
 *
 */
class IStateMachineBase
{
public:
   virtual ~IStateMachineBase() {}

   /**
    * Reset internal data.
    */
   virtual void reset(void) = 0;

   /**
    * Set request active flag.
    *
    * @param[in] enable: flag
    */
   virtual void setRequestActive(IN const bool enable) = 0;

   /**
    * Get request active flag.
    *
    * @return = request active flag
    */
   virtual bool getRequestActive(void) const = 0;

   /**
    * Sets App2Bts compare item.
    *
    * @param[in] item: compare item
    */
   virtual void setApp2BtsCompareItem(IN const BTSApp2BtsMessageCompareItem& item) = 0;

   /**
    * Gets App2Bts compare item.
    *
    * @param[out] item: compare item
    */
   virtual void getApp2BtsCompareItem(OUT BTSApp2BtsMessageCompareItem& item) const = 0;

   /**
    * Gets App2Bts compare item.
    *
    * @return = compare item
    */
   virtual const BTSApp2BtsMessageCompareItem& getApp2BtsCompareItem(void) const = 0;

   /**
    * Send status update message. Update is sent to all.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    */
   virtual void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList) = 0;

   /**
    * Send status update message. Update is sent to specific user. If specific user is NULL update is sent to all.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    */
   virtual void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle) = 0;

   /**
    * Send status update message. Use given status code if suitable. If specific user is NULL update is sent to all.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    * @param[in] statusCode: status code to be used for update (if suitable)
    */
   virtual void updateStatus(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle, IN const BTSCommonEnumClass statusCode) = 0;

   /**
    * Send status update as well as result message. If specific user is NULL status update is sent to all and no result message is sent.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    * @param[in] sendStatusToAll: flag indicating if status shall be send to all
    */
   virtual void updateStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle, IN const bool sendStatusToAll) = 0;

   /**
    * Send status update as well as result message. Use given result and status code if suitable. If specific user is NULL status update is sent to all and no result message is sent.
    *
    * @param[out] bts2AppMsgList: list of Bts2App messages to be sent
    * @param[in] ptrUser: user
    * @param[in] sessionHandle: session handle
    * @param[in] sendStatusToAll: flag indicating if status shall be send to all
    * @param[in] resultCode: result code to be used for update (if suitable)
    * @param[in] statusCode: status code to be used for update (if suitable)
    */
   virtual void updateStatusAndResult(OUT ::std::vector< Bts2App_BaseMessage* >& bts2AppMsgList, IN BtStackIfCallback* ptrUser, IN const BTSSessionHandle sessionHandle, IN const bool sendStatusToAll, IN const BTSCommonEnumClass resultCode, IN const BTSCommonEnumClass statusCode) = 0;
};

} //btstackif

#endif //_I_STATE_MACHINE_BASE_H_
