/**
 * @file BtsTimerPool.h
 *
 * @par SW-Component
 * Timer
 *
 * @brief TimerPool.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Timer pool handling.
 */

#ifndef _BTS_TIMER_POOL_H_
#define _BTS_TIMER_POOL_H_

#include "ITimerPool.h"
#include "TimerEntry.h"
#include "ExtendedTimerEntry.h"
#include "FwLock.h"

namespace btstackif {

/**
 *
 */
class TimerPool : public ITimerPool
{
public:
   /**
    * Get instance of timer pool (reference).
    *
    * @return = reference to timer pool object
    */
   static ITimerPool& getInstance(void);

   /**
    * Create timer pool.
    */
   virtual void create(void);

   /**
    * Destroy timer pool.
    */
   virtual void destroy(void);

   /**
    * Get timer.
    *
    * @return = timer
    */
   virtual Timer* getTimer(void);

   /**
    * Release timer.
    *
    * @param[in] timer: timer
    */
   virtual void releaseTimer(Timer* timer);

   /**
    * Get extended timer.
    *
    * @return = extended timer
    */
   virtual IExtendedTimer* getExtendedTimer(void);

   /**
    * Release extended timer.
    *
    * @param[in] timer: extended timer
    */
   virtual void releaseExtendedTimer(IExtendedTimer* timer);

   virtual void getTimerCount(unsigned int& nmbTimer, unsigned int& nmbUsedTimer, unsigned int& nmbExtendedTimer, unsigned int& nmbUsedExtendedTimer);

private:
   /**
    * Default constructor. Private!!!
    */
   TimerPool();

   /**
    * Destructor. Private!!!
    */
   virtual ~TimerPool();

   const unsigned int _count; /**< initial number of timer entries */
   const unsigned int _extendedCount; /**< initial number of extended timer entries */
   bool _created; /**< flag for created */
   ::std::vector< TimerEntry > _list; /**< list with timers */
   ::std::vector< ExtendedTimerEntry > _extendedList; /**< list with extended timers */
   LockForeverAndNonReentrant _lock; /**< lock */
};

} //btstackif

#endif //_BTS_TIMER_POOL_H_
