/**
 * @file BtsTimerBase.h
 *
 * @par SW-Component
 * Timer
 *
 * @brief Timer base.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for timer handling.
 */

#ifndef _BTS_TIMER_BASE_H_
#define _BTS_TIMER_BASE_H_

#include "BtStackInternalTypes.h"

namespace btstackif {

// class forward declarations
class ITimeoutHandler;

/**
 *
 */
class ITimerActiveHandler
{
public:
   virtual ~ITimerActiveHandler() {}

   /**
    * Sets/resets active flag.
    *
    * @param[in] enable: flag
    */
   virtual void setActiveFlag(IN const bool enable) = 0;
};

/**
 *
 */
class TimerItem
{
public:
   TimerItem();
   TimerItem(IN const TimerItem& ref);
   TimerItem& operator=(IN const TimerItem& ref);
   bool operator==(IN const TimerItem& ref) const;
   bool operator!=(IN const TimerItem& ref) const;
   virtual ~TimerItem();
   // member with public access
   BTSTimerCallBack callback;
   ITimeoutHandler* timeoutHandler;
   BTSTimerId id;
   BTSMonotonicTimeSeconds startSeconds;
   BTSMonotonicTimeNanoSeconds startNanoSeconds;
   BTSMonotonicTimeSeconds timeoutSeconds;
   BTSMonotonicTimeNanoSeconds timeoutNanoSeconds;
   BTSTimeValue setTimeoutValue;
   BTSTimeValue checkTimeoutValue;
   bool cyclic;
   ITimerActiveHandler* handler;
   inline void setCallback(IN BTSTimerCallBack input) { callback = input; }
   inline void setTimeoutHandler(IN ITimeoutHandler* input) { timeoutHandler = input; }
   inline void setTimerActiveHandler(IN ITimerActiveHandler* input) { handler = input; }
};

/**
 *
 */
class TimerBase : public ITimerActiveHandler
{
public:
   virtual void setActiveFlag(IN const bool enable);

   static void init(void);

   inline static void setTrace4CurrentTime(IN const bool enable) { _enableTrace4CurrentTime = enable; }
   inline static void setTrace4TimerStatistic(IN const bool enable) { _enableTrace4TimerStatistic = enable; }
   inline static void setTrace4TimoutCheck(IN const bool enable) { _enableTrace4TimoutCheck = enable; }

protected:
   /**
    * Default constructor.
    */
   TimerBase();

   /**
    * Destructor.
    */
   virtual ~TimerBase();

   static unsigned int _defaultContext; /**< default timer context */
   static unsigned int _maxNmbContexts; /**< maximum number of timer contexts */
   static unsigned int _usedNmbContexts; /**< used number of timer contexts */
   static bool _initDone; /**< initialization flag */
   static bool _defaultContextSet; /**< flag for set default context */
   static BTSMonotonicTimeSeconds _maxTimeSeconds; /**< maximum value for time in seconds */
   static BTSMonotonicTimeNanoSeconds _maxTimeNanoSeconds; /**< maximum value for time in nano seconds */
   static ::std::vector< BTSMonotonicTimeSeconds > _currentTimeSecondsList; /**< current time in seconds */
   static ::std::vector< BTSMonotonicTimeNanoSeconds > _currentTimeNanoSecondsList; /**< current time in nano seconds */
   static ::std::vector< ::std::list< TimerItem > > _timerList; /**< ordered list of timers */

   static bool _enableTrace4CurrentTime; /**< enable flag for trace */
   static bool _enableTrace4TimerStatistic; /**< enable flag for trace */
   static bool _enableTrace4TimoutCheck; /**< enable flag for trace */

   bool _active; /**< timer active flag */
   unsigned int _context; /**< timer context */

   /**
    * Gets current time.
    *
    * @param[out] timeSeconds: time in seconds
    * @param[out] timeNanoSeconds: time in nano seconds
    */
   void getCurrentTime(OUT BTSMonotonicTimeSeconds& timeSeconds, OUT BTSMonotonicTimeNanoSeconds& timeNanoSeconds);

   /**
    * Checks elapsed timer and calls callback functions.
    *
    * @param[in,out] timerList: list with timers
    * @param[in] context: timer context
    * @param[in] timeSeconds: current time in seconds
    * @param[in] timeNanoSeconds: current time in nano seconds
    */
   void checkForTimeout(INOUT std::list< TimerItem >& timerList, IN const unsigned int context, IN const BTSMonotonicTimeSeconds timeSeconds, IN const BTSMonotonicTimeNanoSeconds timeNanoSeconds);

   /**
    * Adds a timer to internal list.
    *
    * @param[in] context: timer context
    * @param[in] id: existing timer id
    * @param[in] timeout: timeout in ms
    * @param[in] callback: callback function
    * @param[in] timeoutHandler: timeout function
    * @param[in] cyclic: cyclic flag
    * @param[in] active: current active flag
    *
    * @return = timer id
    */
   BTSTimerId addTimer(IN const unsigned int context, IN const BTSTimerId id, IN const BTSTimeValue timeout, IN BTSTimerCallBack callback, IN ITimeoutHandler* timeoutHandler, IN const bool cyclic, IN const bool active);

   /**
    * Removes a timer from internal list.
    *
    * @param[in] context: timer context
    * @param[in] id: timer id
    */
   void removeTimer(IN const unsigned int context, IN const BTSTimerId id);

   /**
    * Prints timer statistic.
    *
    * @param[in] context: timer context
    */
   void printTimerStatistic(IN const unsigned int context) const;

private:
   static ::std::vector< BTSTimerId > _nextTimerIdList; /**< next timer id */

   /**
    * Adds a timer to internal list (internal function).
    *
    * @param[in] context: timer context
    * @param[in] id: existing timer id
    * @param[in] timeout: timeout in ms
    * @param[in] callback: callback function
    * @param[in] timeoutHandler: timeout function
    * @param[in] cyclic: cyclic flag
    * @param[in] active: current active flag
    * @param[in] restart: restart flag
    * @param[in] oldStartSeconds: previous start seconds time
    * @param[in] oldStartNanoSeconds: previous start nano seconds time
    *
    * @return = timer id
    */
   BTSTimerId addTimer(IN const unsigned int context, IN const BTSTimerId id, IN const BTSTimeValue timeout, IN BTSTimerCallBack callback, IN ITimeoutHandler* timeoutHandler, IN const bool cyclic, IN const bool active, IN const bool restart, IN const BTSMonotonicTimeSeconds oldStartSeconds, IN const BTSMonotonicTimeNanoSeconds oldStartNanoSeconds);

   /**
    * Adds time offset to a given time value.
    *
    * @param[inout] timeSeconds: given time in seconds
    * @param[inout] timeNanoSeconds: given time in nano seconds
    * @param[in] offSeconds: offset time in seconds
    * @param[in] offNanoSeconds: offset time in nano seconds
    */
   void addTimeOffset(INOUT BTSMonotonicTimeSeconds& timeSeconds, INOUT BTSMonotonicTimeNanoSeconds& timeNanoSeconds, IN const BTSMonotonicTimeSeconds offSeconds, IN const BTSMonotonicTimeNanoSeconds offNanoSeconds);

   /**
    * Adds time offset to a given time value (seconds only).
    *
    * @param[inout] timeSeconds: given time in seconds
    * @param[in] offSeconds: offset time in seconds
    */
   void increaseTimeSeconds(INOUT BTSMonotonicTimeSeconds& timeSeconds, IN const BTSMonotonicTimeSeconds offSeconds);

   /**
    * Adds time offset to a given time value.
    *
    * @param[in] startSeconds: start time in seconds
    * @param[in] startNanoSeconds: start time in nano seconds
    * @param[in] endSeconds: end time in seconds
    * @param[in] endNanoSeconds: end time in nano seconds
    * @param[in] line: line number the function was called from
    */
   BTSTimeValue calcDiff(IN const BTSMonotonicTimeSeconds startSeconds, IN const BTSMonotonicTimeNanoSeconds startNanoSeconds, IN const BTSMonotonicTimeSeconds endSeconds, IN const BTSMonotonicTimeNanoSeconds endNanoSeconds, IN const unsigned int line) const;
};

} //btstackif

#endif //_BTS_TIMER_BASE_H_
