/**
 * @file BtsMessageQueue.h
 *
 * @par SW-Component
 * Base
 *
 * @brief Message queue.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Base class for message queue handling.
 */

#ifndef _BTS_MESSAGE_QUEUE_H_
#define _BTS_MESSAGE_QUEUE_H_

#include "BtStackInternalTypes.h"
#include "FwAssert.h"
#include "FwLock.h"

#include <list>
#include <cstddef>

namespace btstackif {

/**
 *
 */
template < class T >
class MessageQueue
{
public:
   typedef typename ::std::list< T* >::iterator It;
   typedef typename ::std::list< T* >::const_iterator ConstIt;
   typedef typename ::std::list< T* >::reverse_iterator ReverseIt;

   MessageQueue()
   {
      // _messageList is empty after instantiation
      // _lock needs no explicit initialization
      _warningSize = 0; // must be set from outside before using first time
      _warningThresholdReached = false;
   }

   virtual ~MessageQueue()
   {
      T* ptrMessage;

      while(0 < _messageList.size())
      {
         ptrMessage = _messageList.front();
         _messageList.pop_front();
         if(NULL != ptrMessage)
         {
            delete ptrMessage;
         }
      }
   }

   void push(IN T* ptrMessage, bool withLock = true)
   {
      if(NULL != ptrMessage)
      {
         if(true == withLock)
         {
            _lock.lock();
         }

         _messageList.push_back(ptrMessage);

         if(_warningSize <= _messageList.size())
         {
            if(false == _warningThresholdReached)
            {
               _warningThresholdReached = true;

               // this can happen but we do not have any limit; therefore indicate once per power cycle
               // #error_indication
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }

         if(true == withLock)
         {
            _lock.unlock();
         }
      }
      else
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   void pushOnTop(IN T* ptrMessage, bool withLock = true)
   {
      if(NULL != ptrMessage)
      {
         if(true == withLock)
         {
            _lock.lock();
         }

         _messageList.push_front(ptrMessage);

         if(_warningSize <= _messageList.size())
         {
            if(false == _warningThresholdReached)
            {
               _warningThresholdReached = true;

               // this can happen but we do not have any limit; therefore indicate once per power cycle
               // #error_indication
               FW_NORMAL_ASSERT_ALWAYS();
            }
         }

         if(true == withLock)
         {
            _lock.unlock();
         }
      }
      else
      {
         // should never happen else you have programmed something wrong
         // #error_indication
         FW_NORMAL_ASSERT_ALWAYS();
      }
   }

   T* pop(bool withLock = true)
   {
      T* ptrMessage = NULL;

      if(true == withLock)
      {
         _lock.lock();
      }

      if(0 < _messageList.size())
      {
         ptrMessage = _messageList.front();
         _messageList.pop_front();
      }

      if(true == withLock)
      {
         _lock.unlock();
      }

      return ptrMessage;
   }

   void empty(bool withLock = true)
   {
      T* ptrMessage;

      if(true == withLock)
      {
         _lock.lock();
      }

      while(0 < _messageList.size())
      {
         ptrMessage = _messageList.front();
         _messageList.pop_front();
         if(NULL != ptrMessage)
         {
            delete ptrMessage;
         }
      }

      if(true == withLock)
      {
         _lock.unlock();
      }
   }

   void clear(bool withLock = true)
   {
      if(true == withLock)
      {
         _lock.lock();
      }

      _messageList.clear();

      if(true == withLock)
      {
         _lock.unlock();
      }
   }

   inline unsigned int getSize(void) const { return (unsigned int)_messageList.size(); }
   inline void lockAccess(void) { _lock.lock(); }
   inline void unlockAccess(void) { _lock.unlock(); }
   inline void setWarningSize(IN const unsigned int warningSize) { _warningSize = warningSize; }
   inline unsigned int getWarningSize(void) const { return _warningSize; }
   inline typename ::std::list< T* >::iterator getBegin(void) { return _messageList.begin(); }
   inline typename ::std::list< T* >::const_iterator getBegin(void) const { return _messageList.begin(); }
   inline typename ::std::list< T* >::iterator getEnd(void) { return _messageList.end(); }
   inline typename ::std::list< T* >::const_iterator getEnd(void) const { return _messageList.end(); }
   inline typename ::std::list< T* >::reverse_iterator getRBegin(void) { return _messageList.rbegin(); }
   inline typename ::std::list< T* >::const_reverse_iterator getRBegin(void) const { return _messageList.rbegin(); }
   inline typename ::std::list< T* >::reverse_iterator getREnd(void) { return _messageList.rend(); }
   inline typename ::std::list< T* >::const_reverse_iterator getREnd(void) const { return _messageList.rend(); }
   inline void doErase(typename ::std::list< T* >::iterator it) { _messageList.erase(it); }
   inline typename ::std::list< T* >::iterator doEraseAndNext(typename ::std::list< T* >::iterator it) { return _messageList.erase(it); }
   inline T* getFirst(void) { return _messageList.front(); }
   inline const T* getFirst(void) const { return _messageList.front(); }
   inline T* getLast(void) { return _messageList.back(); }
   inline const T* getLast(void) const { return _messageList.back(); }
   inline void swap(MessageQueue< T >& inputList) { _messageList.swap(inputList._messageList); }
   inline bool compare(const MessageQueue< T >& inputList) const { return (_messageList == inputList._messageList); }

private:
   ::std::list< T* > _messageList;
   LockForeverAndNonReentrant _lock;
   unsigned int _warningSize;
   bool _warningThresholdReached;
};

} //btstackif

#endif //_MESSAGE_QUEUE_H_
