/**
 * @file BtStackInternalTypes.h
 *
 * @par SW-Component
 * Types
 *
 * @brief Basic types.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Definitions of all internal types.
 */

#ifndef _BT_STACK_INTERNAL_TYPES_H_
#define _BT_STACK_INTERNAL_TYPES_H_

#include "BtStackIfTypes.h"
#include "BtStackInternalOpcodes.h"
#include "FwArray.h"

#include <list>
#include <stdint.h> // using <cstdint> compiler complains that std=c++0x or -std=gnu++0x compiler option must be enabled
#include <time.h>

#include "cc_dbus_if/CcDbusIfTypes.h"

namespace btstackif {

// class forward declarations
class BtStackIfCallback;
class App2Bts_BaseMessage;
class ICheckWaitingQueue;

/*
 * BIG HINT: If you want to add a new define/type/... please check if this type is already defined in EvolutionGeniviStackTypes.h or SimulationStackTypes.h.
 *           If yes then move it to here.
 *
 * RULE 1: use a class instead of declaring a struct (avoiding problems with copy actions), implement copy constructor, assignment operator and equality/inequality operator
 *
 * keep following order:
 * - class forward declarations
 * - defines (if really necessary)
 * - typedefs
 * - enums
 * - structs => classes
 */

#define BTS_DUMMY_BDADDRESS "000000000000" /**< dummy BT device address */

#define BTS_TTFIS_COMMAND "BTS_TTFIS_COMMAND" /**< TTFis command */

#define BTS_SIMULATE_STARTPAIRING_DURING_INCOMINGPAIRING "BTS_SIMULATE_STARTPAIRING_DURING_INCOMINGPAIRING" /**< connection control test command */

#define BTS_SIMULATE_STARTPAIRING_DURING_INCOMINGPAIRING_2 "BTS_SIMULATE_STARTPAIRING_DURING_INCOMINGPAIRING_2" /**< connection control test command */

#define BTS_SIMULATE_REMOTE_PAIRING_DURING_CONNECTING "BTS_SIMULATE_REMOTE_PAIRING_DURING_CONNECTING" /**< connection control test command */

#define BTS_TRIGGER_CANCEL_SEARCH_DURING_SEARCH_RESULT "BTS_TRIGGER_CANCEL_SEARCH_DURING_SEARCH_RESULT" /**< connection control test command */

#define BTS_TRIGGER_DISCONNECT_WHILE_DISCONNECTED "BTS_TRIGGER_DISCONNECT_WHILE_DISCONNECTED" /**< connection control test command */

#define BTS_TRIGGER_DISCONNECT_WHILE_DISCONNECTED_SET_STATE "BTS_TRIGGER_DISCONNECT_WHILE_DISCONNECTED_SET_STATE" /**< connection control test command */

#define BTS_TRIGGER_CANCEL_DURING_CONNECT_RESULT "BTS_TRIGGER_CANCEL_DURING_CONNECT_RESULT" /**< connection control test command */

#define BTS_TRIGGER_DISABLE_SEARCHING "BTS_TRIGGER_DISABLE_SEARCHING" /**< connection control test command */

#define BTS_TRIGGER_DISABLE_SETTING "BTS_TRIGGER_DISABLE_SETTING" /**< connection control test command */

#define BTS_TRIGGER_BLOCK_SENDING_DISCONNECT "BTS_TRIGGER_BLOCK_SENDING_DISCONNECT" /**< connection control test command */

#define BTS_TRIGGER_LOCAL_CONNECT_WHILE_REMOTE_CONNECT "BTS_TRIGGER_LOCAL_CONNECT_WHILE_REMOTE_CONNECT" /**< connection control test command */

#define BTS_TRIGGER_LOCAL_DISCONNECT_WHILE_REMOTE_CONNECT "BTS_TRIGGER_LOCAL_DISCONNECT_WHILE_REMOTE_CONNECT" /**< connection control test command */

#define BTS_TRIGGER_LOCAL_CONNECT_BEFORE_REMOTE_CONNECT "BTS_TRIGGER_LOCAL_CONNECT_BEFORE_REMOTE_CONNECT" /**< connection control test command */

#define BTS_TRIGGER_LOCAL_CONNECTED_BEFORE_REMOTE_CONNECT "BTS_TRIGGER_LOCAL_CONNECTED_BEFORE_REMOTE_CONNECT" /**< connection control test command */

#define BTS_TRIGGER_LOCAL_CONNECT_SAME_DEVICE "BTS_TRIGGER_LOCAL_CONNECT_SAME_DEVICE" /**< connection control test command */

#define BTS_TRIGGER_PROTOCOL_MANAGER_TEST_ACTION "BTS_TRIGGER_PROTOCOL_MANAGER_TEST_ACTION" /**< connection control test command */

#define BTS_SET_RETRY_MAX_COUNTER "BTS_SET_RETRY_MAX_COUNTER" /**< connection control test command */

#define BTS_SET_RETRY_TIMEOUT "BTS_SET_RETRY_TIMEOUT" /**< connection control test command */

#define BTS_TRIGGER_CANCEL_DURING_ACCEPT_REJECT_REMOTE_CONNECT "BTS_TRIGGER_CANCEL_DURING_ACCEPT_REJECT_REMOTE_CONNECT" /**< connection control test command */

#define BTS_TRIGGER_INTERNAL_SDP_SEARCH "BTS_TRIGGER_INTERNAL_SDP_SEARCH" /**< connection control test command */

#define BTS_TRIGGER_SDP_SEARCH_AFTER_PAIRING "BTS_TRIGGER_SDP_SEARCH_AFTER_PAIRING" /**< connection control test command */

#define BTS_BASIC_CONF_TRIGGER_IGNORE_UPDATE "BTS_BASIC_CONF_TRIGGER_IGNORE_UPDATE" /**< connection control test command */

#define BTS_TRIGGER_IGNORE_REPORTED_LINK_KEY "BTS_TRIGGER_IGNORE_REPORTED_LINK_KEY" /**< connection control test command */

#define BTS_TRIGGER_BLOCK_SET_DISCOVERABLE "BTS_TRIGGER_BLOCK_SET_DISCOVERABLE" /**< connection control test command */

#define BTS_TRIGGER_IGNORE_DEVICE_CONNECTED_UPDATE "BTS_TRIGGER_IGNORE_DEVICE_CONNECTED_UPDATE" /**< connection control test command */

#define BTS_TRIGGER_INVALID_FW_DISPATCHER_CALL "BTS_TRIGGER_INVALID_FW_DISPATCHER_CALL" /**< connection control test command */

/**
 * Forward declaration of class BTSApp2BtsMessageCompareItem.
 */
class BTSApp2BtsMessageCompareItem;

/**
 * Forward declaration of class BTSApp2BtsMessageMasterCompareItem.
 */
class BTSApp2BtsMessageMasterCompareItem;

/**
 * Forward declaration of class BTSApp2BtsMessageRequestItem.
 */
class BTSApp2BtsMessageRequestItem;

/**
 * Forward declaration of class BTSHandleIpc2BtsMessageItem.
 */
class BTSHandleIpc2BtsMessageItem;

/**
 * Forward declaration of class BTSProtocolBaseEntry.
 */
class BTSProtocolBaseEntry;

/**
 * Forward declaration of class BTSDbusInterfaceItem.
 */
class BTSDbusInterfaceItem;

/**
 * Type definition for BT device address.
 * Address value will be given as binary data.
 */
typedef ::fw::Array<unsigned char, 6> BTSBinaryBDAddress;

/**
 * Type definition for magic number.
 */
typedef unsigned int BTSMagicNumber;

/**
 * Type definition for event mask.
 */
typedef unsigned int BTSEventMask;

/**
 * Type definition for unspecific function block or opcode.
 */
typedef int BTSCommonEnumClass;

/**
 * Type definition for binary trace value of opcode.
 */
typedef unsigned short int BTSTraceOpCode;

/**
 * Type definition for object path.
 */
typedef ::std::string BTSObjectPath;

/**
 * Type definition for list of object paths.
 */
typedef ::std::vector< BTSObjectPath > BTSObjectPathList;

/**
 * Type definition for bus name.
 */
typedef ::std::string BTSBusName;

/**
 * Type definition for DBUS interface.
 */
typedef ::std::string BTSDbusInterface;

/**
 * Type definition for monotonic time (seconds).
 */
typedef __time_t BTSMonotonicTimeSeconds;

/**
 * Type definition for monotonic time (nanoseconds, range: 0...999999999).
 */
typedef long int BTSMonotonicTimeNanoSeconds;

/**
 * Type definition for timer callback.
 */
typedef void (*BTSTimerCallBack)(void);

/**
 * Type definition for timer id.
 */
typedef unsigned int BTSTimerId;

/**
 * Type definition for file name.
 */
typedef ::std::string BTSFilename;

/**
 * Type definition for SPP instance id (used for SPP connections).
 */
typedef unsigned char BTSSppInstanceId;

/**
 * Type definition for DBUS property list.
 */
typedef ::std::vector< ::ccdbusif::DbusVariantProperty > BTSDbusPropertyList;

/**
 * Default value for common enum class.
 */
enum /* c++11 feature enum-base : BTSCommonEnumClass */
{
   BTS_COMMON_ENUM_CLASS_DEFAULT_VALUE = 0
};

/**
 * Type definition for factory creator.
 */
enum BTSFactoryCreator
{
   BTS_LOCAL_CREATE = 0, /**< instance created locally */
   BTS_EXTERN_CREATE /**< instance created by external source (unit test) */
};

/**
 * Type definition for message type.
 */
enum BTSMessageType
{
   BTS_APP_TO_BTS = 0, /**< application to BtStackIf */
   BTS_BTS_TO_APP, /**< BtStackIf to application */
   BTS_BTS_TO_IPC, /**< BtStackIf to DBUS (IPC) */
   BTS_IPC_TO_BTS, /**< DBUS (IPC) to BtStackIf */
   // -------------------------------------------------------------------------
   BTS_X_TO_Y_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for common error codes for IPC received messages.
 */
enum BTSIpcCommonErrorCode
{
   BTS_IPC_SUCCESS = 0, /**< success */
   BTS_IPC_METHOD_RETURN_TIMEOUT, /**< timeout occurred for method return (confirmation) */
   BTS_IPC_SENDING_FAILED, /**< sending of message failed */
   BTS_IPC_ERROR, /**< request failed, general error */
   BTS_IPC_UPDATE_TIMEOUT, /**< timeout occurred e.g. for property update */
   BTS_IPC_BUSY, /**< Bluetooth stack is busy (blocked due to other activities); retry is necessary */
   BTS_IPC_RETRY_ABORTED, /**< any internal retry action was aborted */
   BTS_IPC_ALREADY_CONNECTED, /**< device/protocol is already connected */
   BTS_IPC_NOT_CONNECTED, /**< device/protocol is not connected (already disconnected) */
   BTS_IPC_ALREADY_CREATED, /**< object path is already created, pairing already done (link key available) */
   BTS_IPC_PAIRING_CONN_LOST, /**< failed due to connection lost */
   BTS_IPC_PAIRING_AUTHENTICATION_ERROR, /**< failed due to authentication error */
   BTS_IPC_PAIRING_PAIRING_CANCELLED, /**< failed because cancelled */
   BTS_IPC_PAIRING_MISSING_PIN, /**< failed because PIN is missing */
   BTS_IPC_PAIRING_UNKNOWN, /**< failed due to unknown reason */
   BTS_IPC_DBUS_ERROR_NO_SERVER, /**< org.freedesktop.DBus.Error.NoServer */
   BTS_IPC_CANCELLED, /**< action on BT stack side was cancelled / aborted */
   BTS_IPC_CONNECTION_CONFLICT_SAME_DEVICE, /**< conflict happened during connect, same device */
   // -------------------------------------------------------------------------
   BTS_IPC_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for semaphore sections.
 */
enum BTSSemaphoreSection
{
   BTS_SEM_COMMON = 0, /**< semaphore for common section */
   BTS_SEM_CONNECTION, /**< semaphore for connection section */
   BTS_SEM_PHONECALL, /**< semaphore for phonecall section */
   BTS_SEM_PHONEBOOK, /**< semaphore for phonebook section */
   BTS_SEM_MESSAGING, /**< semaphore for messaing section */
   BTS_SEM_MEDIA_PLAYER, /**< semaphore for media player section */
   // -------------------------------------------------------------------------
   BTS_SEM_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for DBUS service availability.
 */
enum BTSDbusServiceAvailability
{
   BTS_DBUS_SERVICE_NOT_AVAILABLE = 0, /**< DBUS service is not available */
   BTS_DBUS_SERVICE_WAITING, /**< waiting for DBUS service availability */
   BTS_DBUS_SERVICE_AVAILABLE, /**< DBUS service is available */
   // -------------------------------------------------------------------------
   BTS_DBUS_SERVICE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for test command group.
 */
enum BTSTestCommandGroup
{
   BTS_TEST_COMMAND_GROUP_GENERIC = 0, /**< generic part */
   BTS_TEST_COMMAND_GROUP_MAIN, /**< main control */
   BTS_TEST_COMMAND_GROUP_CONNECTION, /**< connection */
   // -------------------------------------------------------------------------
   BTS_TEST_COMMAND_GROUP_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Bluetooth device role.
 */
enum BTSBtRole
{
   BTS_BT_MASTER = 0, /**< master */
   BTS_BT_SLAVE, /**< slave */
   // -------------------------------------------------------------------------
   BTS_BT_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for internal disconnect reason.
 */
enum BTSInternalDisconnectReason
{
   BTS_INT_DISCONNECT_REASON_LOCAL_DEVICE = 0, /**< disconnected by local device */
   BTS_INT_DISCONNECT_REASON_REMOTE_DEVICE, /**< disconnected by remote device */
   BTS_INT_DISCONNECT_REASON_EXCEPTION, /**< disconnected by exception */
   BTS_INT_DISCONNECT_REASON_LINK_LOSS, /**< disconnected by link loss */
   // -------------------------------------------------------------------------
   BTS_INT_DISCONNECT_REASON_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for status transition.
 */
enum BTSStatusTransition
{
   BTS_STATUS_TRANSITION_NO_CHANGE = 0, /**< no status change */
   BTS_STATUS_TRANSITION_ENABLED, /**< status changed to enabled (on) */
   BTS_STATUS_TRANSITION_DISABLED, /**< status changed to disabled (off) */
   // -------------------------------------------------------------------------
   BTS_STATUS_TRANSITION_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Genivi trace level bits.
 */
enum BTSGeniviTraceLevelBit
{
   BTS_GENIVI_TRACE_LEVEL_FATAL = 0, /**< fatal */
   BTS_GENIVI_TRACE_LEVEL_ERROR, /**< error */
   BTS_GENIVI_TRACE_LEVEL_WARNING, /**< warning */
   BTS_GENIVI_TRACE_LEVEL_INFO, /**< info */
   BTS_GENIVI_TRACE_LEVEL_DEBUG, /**< debug */
   BTS_GENIVI_TRACE_LEVEL_LOG, /**< log */
   // -------------------------------------------------------------------------
   BTS_GENIVI_TRACE_LEVEL_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Genivi trace category bitfield.
 */
typedef BTSBitfield<unsigned int, BTSGeniviTraceLevelBit, BTS_GENIVI_TRACE_LEVEL_LAST> BTSGeniviTraceLevel;

/**
 * Type definition for Genivi trace category bits.
 */
enum BTSGeniviTraceCategoryBit
{
   BTS_GENIVI_TRACE_CATEGORY_OFD_INTROSPECTABLE = 0,
   BTS_GENIVI_TRACE_CATEGORY_OFD_PROPERTIES,
   BTS_GENIVI_TRACE_CATEGORY_OFD_OBJECT_MANAGER,
   BTS_GENIVI_TRACE_CATEGORY_ADAPTER,
   BTS_GENIVI_TRACE_CATEGORY_AGENT,
   BTS_GENIVI_TRACE_CATEGORY_AGENT_MANAGER,
   BTS_GENIVI_TRACE_CATEGORY_DEVICE,
   BTS_GENIVI_TRACE_CATEGORY_SERVICE,
   BTS_GENIVI_TRACE_CATEGORY_TRACE,
   BTS_GENIVI_TRACE_CATEGORY_DM,
   BTS_GENIVI_TRACE_CATEGORY_HANDSFREE,
   BTS_GENIVI_TRACE_CATEGORY_MANAGER,
   BTS_GENIVI_TRACE_CATEGORY_MODEM,
   BTS_GENIVI_TRACE_CATEGORY_VOICE_CALL,
   BTS_GENIVI_TRACE_CATEGORY_VOICE_CALL_MANAGER,
   BTS_GENIVI_TRACE_CATEGORY_NETWORK_REGISTRATION,
   BTS_GENIVI_TRACE_CATEGORY_NETWORK_OPERATOR,
   BTS_GENIVI_TRACE_CATEGORY_CALL_VOLUME,
   BTS_GENIVI_TRACE_CATEGORY_SIRI,
   BTS_GENIVI_TRACE_CATEGORY_CLIENT,
   BTS_GENIVI_TRACE_CATEGORY_PHONEBOOK_ACCESS,
   BTS_GENIVI_TRACE_CATEGORY_SESSION,
   BTS_GENIVI_TRACE_CATEGORY_TRANSFER,
   BTS_GENIVI_TRACE_CATEGORY_AVP,
   BTS_GENIVI_TRACE_CATEGORY_MEDIA_FOLDER,
   BTS_GENIVI_TRACE_CATEGORY_MEDIA_ITEM,
   BTS_GENIVI_TRACE_CATEGORY_MEDIA_PLAYER,
   BTS_GENIVI_TRACE_CATEGORY_MEDIA_TRANSPORT,
   BTS_GENIVI_TRACE_CATEGORY_MESSAGE,
   BTS_GENIVI_TRACE_CATEGORY_MESSAGE_ACCESS,
   BTS_GENIVI_TRACE_CATEGORY_CTN,
   BTS_GENIVI_TRACE_CATEGORY_OPPC,
   BTS_GENIVI_TRACE_CATEGORY_OPPS,
   BTS_GENIVI_TRACE_CATEGORY_SPP,
   BTS_GENIVI_TRACE_CATEGORY_DID,
   BTS_GENIVI_TRACE_CATEGORY_DUN,
   BTS_GENIVI_TRACE_CATEGORY_PAN,
   BTS_GENIVI_TRACE_CATEGORY_NETWORK,
   // -------------------------------------------------------------------------
   BTS_GENIVI_TRACE_CATEGORY_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for Genivi trace category bitfield.
 */
typedef BTSBitfield<uint64_t, BTSGeniviTraceCategoryBit, BTS_GENIVI_TRACE_CATEGORY_LAST> BTSGeniviTraceCategory;

/**
 * Type definition for Bluetooth power mode.
 */
enum BTSBtPowerMode
{
   BTS_BT_POWER_MODE_ACTIVE = 0, /**< active mode */
   BTS_BT_POWER_MODE_HOLD, /**< hold mode */
   BTS_BT_POWER_MODE_SNIFF, /**< sniff mode */
   BTS_BT_POWER_MODE_PARK, /**< park mode */
   // -------------------------------------------------------------------------
   BTS_BT_POWER_MODE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for configuration parameter.
 */
enum BTSConfigurationParameter
{
   BTS_CONFIG_PARAM_LOCAL_SERVICES = 0, /**< local services */
   BTS_CONFIG_PARAM_PAGE_TIMEOUT, /**< page timeout */
   BTS_CONFIG_PARAM_PAGE_TIMEOUT_SECOND, /**< page timeout for second device */
   // -------------------------------------------------------------------------
   BTS_CONFIG_PARAM_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for handling type of App2Bts message.
 */
enum BTSApp2BtsMessageHandlingType
{
   BTS_APP2BTS_PUSH_TO_WORKING = 0, /**< push to working queue */
   BTS_APP2BTS_PUSH_TO_WAITING, /**< push to waiting queue */
   BTS_APP2BTS_DELETE, /**< delete */
   // -------------------------------------------------------------------------
   BTS_APP2BTS_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for hands-free supported features on AG side (bits).
 */
enum BTSHandsfreeAgFeaturesBit
{
   BTS_HFP_AG_FEATURE_VOICE_RECOGNITION = 0, /**< Voice-recognition */
   BTS_HFP_AG_FEATURE_ATTACH_VOICE_TAG, /**< Attach-voice-tag */
   BTS_HFP_AG_FEATURE_3WAY, /**< 3way */
   BTS_HFP_AG_FEATURE_ECNR, /**< ECNR */
   BTS_HFP_AG_FEATURE_INBAND_RING, /**< Inband_ring */
   BTS_HFP_AG_FEATURE_REJECT_INCOMING_CALL, /**< Reject_incoming_call */
   BTS_HFP_AG_FEATURE_ENHANCED_CALL_STATUS, /**< Enhanced_call_status */
   BTS_HFP_AG_FEATURE_ENHANCED_CALL_CONTROL, /**< Enhanced_call_control */
   BTS_HFP_AG_FEATURE_EXTENDED_ERROR_CODES, /**< Extended_error_codes */
   BTS_HFP_AG_FEATURE_CODEC_NEGOTIATION, /**< Codec_negotiation */
   BTS_HFP_AG_FEATURE_HF_INDICATORS, /**< HF indicators */
   BTS_HFP_AG_FEATURE_ESCO_S4_AND_T2_SETTING_SUPPORTED, /**< eSCO S4 and T2 settings supported */
   // -------------------------------------------------------------------------
   BTS_HFP_AG_FEATURE_LAST /**< end marker, must be the last one */
};

/**
 * Type definition for hands-free supported features on AG side.
 */
typedef BTSBitfield<unsigned int, BTSHandsfreeAgFeaturesBit, BTS_HFP_AG_FEATURE_LAST> BTSHandsfreeAgFeatures;

/**
 * Type definition for App2Bts message compare item.
 * To be used to check if different messages providing the same request.
 * In normal case the message opcode is enough to compare.
 * But in case of connect the BDAddress, protocol, ... have to be compared too.
 */
class BTSApp2BtsMessageCompareItem
{
public:
   BTSApp2BtsMessageCompareItem();
   BTSApp2BtsMessageCompareItem(IN const BTSApp2BtsMessageCompareItem& ref);
   BTSApp2BtsMessageCompareItem& operator=(IN const BTSApp2BtsMessageCompareItem& ref);
   bool operator==(IN const BTSApp2BtsMessageCompareItem& ref) const;
   bool operator!=(IN const BTSApp2BtsMessageCompareItem& ref) const;
   virtual ~BTSApp2BtsMessageCompareItem();
   // member with public access
   BTSApp2BtsOpcode opCode;
   BTSBDAddress deviceAddress;
   BTSProtocolId protocolId;
   BTSUuid sppUuid;
   BTSMasInstanceName masInstance;
   BTSSearchType searchType;
   virtual void reset(void);
};

/**
 * Type definition for App2Bts message master compare item.
 * To be used to check if different messages providing the same request.
 * In normal case the message opcode is enough to compare.
 * But in case of connect the BDAddress, protocol, ... have to be compared too.
 */
class BTSApp2BtsMessageMasterCompareItem : public BTSApp2BtsMessageCompareItem
{
public:
   BTSApp2BtsMessageMasterCompareItem();
   BTSApp2BtsMessageMasterCompareItem(IN const BTSApp2BtsMessageMasterCompareItem& ref);
   BTSApp2BtsMessageMasterCompareItem& operator=(IN const BTSApp2BtsMessageMasterCompareItem& ref);
   bool operator==(IN const BTSApp2BtsMessageMasterCompareItem& ref) const;
   bool operator!=(IN const BTSApp2BtsMessageMasterCompareItem& ref) const;
   virtual ~BTSApp2BtsMessageMasterCompareItem();
   bool isMatch(IN const BTSApp2BtsMessageCompareItem& ref) const;
   // member with public access
   bool compareOpCode;
   bool compareDeviceAddress;
   bool compareProtocolId;
   bool compareSppUuid;
   bool compareMasInstance;
   bool compareSearchType;
};

/**
 * Type definition for App2Bts message request item.
 * Consists of App2Bts message compare item and user/client control data.
 */
class BTSApp2BtsMessageRequestItem
{
public:
   BTSApp2BtsMessageRequestItem();
   BTSApp2BtsMessageRequestItem(IN const BTSApp2BtsMessageRequestItem& ref);
   BTSApp2BtsMessageRequestItem& operator=(IN const BTSApp2BtsMessageRequestItem& ref);
   bool operator==(IN const BTSApp2BtsMessageRequestItem& ref) const;
   bool operator!=(IN const BTSApp2BtsMessageRequestItem& ref) const;
   virtual ~BTSApp2BtsMessageRequestItem();
   // member with public access
   BTSApp2BtsMessageCompareItem item;
   BtStackIfCallback* user;
   BTSSessionHandle handle;
   inline void setUser(IN BtStackIfCallback* ptrUser) { user = ptrUser; }
   void reset(void);
};

/**
 * Type definition for item to handle Ipc2Bts message.
 */
class BTSHandleIpc2BtsMessageItem
{
public:
   BTSHandleIpc2BtsMessageItem();
   BTSHandleIpc2BtsMessageItem(IN const BTSHandleIpc2BtsMessageItem& ref);
   BTSHandleIpc2BtsMessageItem& operator=(IN const BTSHandleIpc2BtsMessageItem& ref);
   bool operator==(IN const BTSHandleIpc2BtsMessageItem& ref) const;
   bool operator!=(IN const BTSHandleIpc2BtsMessageItem& ref) const;
   virtual ~BTSHandleIpc2BtsMessageItem();
   // member with public access
   BTSApp2BtsMessageCompareItem item;
   bool deleteMessage;
   App2Bts_BaseMessage* message;
   ICheckWaitingQueue* checkIf;
   inline void setMessage(IN App2Bts_BaseMessage* ptrMessage) { message = ptrMessage; }
   inline void setCheckIf(IN ICheckWaitingQueue* interface) { checkIf = interface; }
   void reset(void);
};

/**
 * Type definition for a protocol base entry.
 * A protocol is identified by address + protocol id: HFP, AVP, PBAP (=> PIM), PAN, DUN
 * A protocol is identified by address + protocol id + SPP uuid: SPP
 * A protocol is identified by address + protocol id + MAS instance id: MAP (=> MSG)
 */
class BTSProtocolBaseEntry
{
public:
   BTSProtocolBaseEntry();
   BTSProtocolBaseEntry(IN const BTSProtocolBaseEntry& ref);
   BTSProtocolBaseEntry& operator=(IN const BTSProtocolBaseEntry& ref);
   bool operator==(IN const BTSProtocolBaseEntry& ref) const;
   bool operator!=(IN const BTSProtocolBaseEntry& ref) const;
   bool operator<(const BTSProtocolBaseEntry& ref) const;
   bool operator>(const BTSProtocolBaseEntry& ref) const;
   virtual ~BTSProtocolBaseEntry();
   // member with public access
   BTSBDAddress deviceAddress; /**< device address */
   BTSProtocolId protocolId; /**< protocol id */
   BTSUuid sppUuid; /**< SPP uuid */
   BTSMasInstanceId masInstanceId; /**< MAS instance id */

protected:
   int compare(const BTSProtocolBaseEntry& ref) const;
};

/**
 * Type definition for DBUS interface item.
 */
class BTSDbusInterfaceItem
{
public:
   BTSDbusInterfaceItem();
   BTSDbusInterfaceItem(IN const BTSDbusInterfaceItem& ref);
   BTSDbusInterfaceItem& operator=(IN const BTSDbusInterfaceItem& ref);
   bool operator==(IN const BTSDbusInterfaceItem& ref) const;
   bool operator!=(IN const BTSDbusInterfaceItem& ref) const;
   virtual ~BTSDbusInterfaceItem();
   // member with public access
   BTSCommonEnumClass dbusInterface; /**< DBUS service interface */
   BTSCommonEnumClass busType; /**< bus type */
   BTSBusName busName; /**< bus name */
   BTSObjectPath objPath; /**< object path */
};

/**
 * Type definition for BDAddress trace.
 */
class BTSBDAddressTrace
{
public:
   BTSBDAddressTrace();
   BTSBDAddressTrace(IN const BTSBDAddress& address, IN const bool allowEmpty = false);
   BTSBDAddressTrace(IN const BTSBDAddressTrace& ref);
   BTSBDAddressTrace& operator=(IN const BTSBDAddressTrace& ref);
   bool operator==(IN const BTSBDAddressTrace& ref) const;
   bool operator!=(IN const BTSBDAddressTrace& ref) const;
   virtual ~BTSBDAddressTrace();
   uint32_t getUpper(void) const;
   uint16_t getLower(void) const;

private:
   BTSBDAddress _stringAddress; /**< device address (string) */
   BTSBinaryBDAddress _binaryAddress; /**< device address (binary) */
   bool _allowEmpty; /**< flag indicating that empty address is allowed */
};

/**
 * Type definition for user callback entry.
 */
class BTSUserCallbackEntry
{
public:
   BTSUserCallbackEntry();
   BTSUserCallbackEntry(IN BtStackIfCallback* cb, IN const BTSInterfaceType interface, IN const BTSFunctionBlock component);
   BTSUserCallbackEntry(IN const BTSUserCallbackEntry& ref);
   BTSUserCallbackEntry& operator=(IN const BTSUserCallbackEntry& ref);
   bool operator==(IN const BTSUserCallbackEntry& ref) const;
   bool operator!=(IN const BTSUserCallbackEntry& ref) const;
   virtual ~BTSUserCallbackEntry();
   // member with public access
   BtStackIfCallback* callback; /**< callback */
   BTSInterfaceType stackInterface; /**< stack interface */
   BTSFunctionBlock subComponent; /**< sub component */
};

/**
 * Get component key to access DBUS instance list.
 *
 * @param[in] entry: component key
 * @param[in] component: component
 * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
 * @param[in] subComponent: sub component
 *
 * @return = component key
 */
const BTSComponentEntry& getDbusComponentKey(IN BTSComponentEntry& entry, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

/**
 * Get component key to access control instance list.
 *
 * @param[in] entry: component key
 * @param[in] component: component
 * @param[in] stackInterface: stack interface to be used (Bluetooth stack)
 * @param[in] subComponent: sub component
 *
 * @return = component key
 */
const BTSComponentEntry& getControlComponentKey(IN BTSComponentEntry& entry, IN const BTSFunctionBlock component, IN const BTSInterfaceType stackInterface, IN const BTSFunctionBlock subComponent);

} //btstackif

#endif //_BT_STACK_INTERNAL_TYPES_H_
