/**
 * @file TrcBts2App.cpp
 *
 * @par SW-Component
 * Messages
 *
 * @brief Traces for Bts2App messages.
 *
 * @copyright (C) 2016 Robert Bosch GmbH.
 *
 * @par
 * The reproduction, distribution and utilization of this file as
 * well as the communication of its contents to others without express
 * authorization is prohibited. Offenders will be held liable for the
 * payment of damages. All rights reserved in the event of the grant
 * of a patent, utility model or design.
 *
 * @details Implementation of traces for Bts2App messages.
 */

#include "Bts2App_MessageWrapper.h"
#include "TraceClasses.h"
#include "FwTrace.h"
#include "BtsUtils.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_BTS_BTS2APP
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE
#include "trcGenProj/Header/TrcBts2App.cpp.trc.h"
#else
#include "BtStackIfTypesTrace.h"
#include "BtStackInternalTypesTrace.h"
#include "BtStackInternalOpcodesTrace.h"
#endif
#endif

namespace btstackif {

void Bts2App_SetDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   // TODO: second deviceAddress;
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   // TODO: second deviceAddress;
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result)));

#endif
}

void Bts2App_FbConnectionInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_LocalVersionInfo::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u stackType=%d supportedServices=0x%04X stackVersion=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSInterfaceType, _versionInfo.stackType),
            _versionInfo.supportedServices.getData(),
            _versionInfo.stackVersion.c_str() /*DONE: _versionInfo format*/));
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): btVersion=%d moduleName=%50s fwVersion=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            ETG_ENUM(TRC_BTSBtCoreVersion, _versionInfo.btVersion),
            _versionInfo.moduleName.c_str(),
            _versionInfo.fwVersion.c_str() /*DONE: _versionInfo format*/));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u stackType=%s supportedServices=0x%04X stackVersion=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getInterfaceType2String(_versionInfo.stackType),
            _versionInfo.supportedServices.getData(),
            _versionInfo.stackVersion.c_str() /*DONE: _versionInfo format*/));
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): btVersion=%s moduleName=%50s fwVersion=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            getBtCoreVersion2String(_versionInfo.btVersion),
            _versionInfo.moduleName.c_str(),
            _versionInfo.fwVersion.c_str() /*DONE: _versionInfo format*/));

#endif
}

void Bts2App_SwitchBluetoothOnOffResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_CurrentBluetoothOnOffStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress, true);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X bluetoothMode=%d offReason=%d adapterMode=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBluetoothMode, _bluetoothMode),
            ETG_ENUM(TRC_BTSBluetoothOffReason, _offReason),
            ETG_ENUM(TRC_BTSAdapterMode, _adapterMode)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress, true);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X bluetoothMode=%s offReason=%s adapterMode=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getBluetoothMode2String(_bluetoothMode),
            getBluetoothOffReason2String(_offReason),
            getAdapterMode2String(_adapterMode)));

#endif
}

void Bts2App_SwitchBluetoothHardwareOnOffResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetLocalBtNameResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_CurrentLocalBtName::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u adapterName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            _adapterName.c_str()));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u adapterName=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            _adapterName.c_str()));

#endif
}

void Bts2App_CurrentDeviceForRestrictedPairingConnecting::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress, true);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u expectedBtAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress, true);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u expectedBtAddr=%08X%04X",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2App_SetLocalAdapterModesResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_CurrentDiscoverableMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u discoverableMode=%d offReason=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSLocalMode, _discoverableMode),
            ETG_ENUM(TRC_BTSLocalModeOffReason, _offReason)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u discoverableMode=%s offReason=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getLocalMode2String(_discoverableMode),
            getLocalModeOffReason2String(_offReason)));

#endif
}

void Bts2App_CurrentConnectableMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u connectableMode=%d offReason=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSLocalMode, _connectableMode),
            ETG_ENUM(TRC_BTSLocalModeOffReason, _offReason)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u connectableMode=%s offReason=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getLocalMode2String(_connectableMode),
            getLocalModeOffReason2String(_offReason)));

#endif
}

void Bts2App_StartDiscoveryResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_StopDiscoveryResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_DiscoveringStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u discoveringStatus=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSDiscoveringStatus, _discoveringStatus)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u discoveringStatus=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getDiscoveringStatus2String(_discoveringStatus)));

#endif
}

void Bts2App_DiscoveredDeviceFound::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_discoveredDevice.address);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X rssi=%d majorServiceClass=0x%04X majorDeviceClass=%d minorDeviceClass=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            _discoveredDevice.rssi,
            _discoveredDevice.majorServiceClass.getData(),
            ETG_ENUM(TRC_BTSMajorDeviceClass, _discoveredDevice.majorDeviceClass),
            ETG_ENUM(TRC_BTSMinorDeviceClass, _discoveredDevice.minorDeviceClass)));
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): services=0x%04X sppCapabilities.size()=%u name=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _discoveredDevice.services.getData(),
            _discoveredDevice.sppCapabilities.size(),
            _discoveredDevice.name.c_str()));
   for(size_t i = 0; i < _discoveredDevice.sppCapabilities.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): sppCapabilities[%u]: uuidList.size()=%u serviceName=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _discoveredDevice.sppCapabilities[i].uuidList.size(),
               _discoveredDevice.sppCapabilities[i].serviceName.c_str() /*DONE: _sppCapabilities format*/));
      for(size_t j = 0; j < _discoveredDevice.sppCapabilities[i].uuidList.size(); j++)
      {
         ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): sppCapabilities[%u]: uuidList[%u]: uuid=%s",
                  getMessageTraceId(),
                  ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
                  i,
                  j,
                  _discoveredDevice.sppCapabilities[i].uuidList[j].c_str() /*DONE: _sppCapabilities format*/));
      }
   }

#else

   BTSBDAddressTrace traceAddr(_discoveredDevice.address);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X rssi=%d majorServiceClass=0x%04X majorDeviceClass=%s minorDeviceClass=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            _discoveredDevice.rssi,
            _discoveredDevice.majorServiceClass.getData(),
            getMajorDeviceClass2String(_discoveredDevice.majorDeviceClass),
            getMinorDeviceClass2String(_discoveredDevice.minorDeviceClass)));
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): services=0x%04X sppCapabilities.size()=%u name=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _discoveredDevice.services.getData(),
            _discoveredDevice.sppCapabilities.size(),
            _discoveredDevice.name.c_str()));
   for(size_t i = 0; i < _discoveredDevice.sppCapabilities.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): sppCapabilities[%u]: uuidList.size()=%u serviceName=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _discoveredDevice.sppCapabilities[i].uuidList.size(),
               _discoveredDevice.sppCapabilities[i].serviceName.c_str() /*DONE: _sppCapabilities format*/));
      for(size_t j = 0; j < _discoveredDevice.sppCapabilities[i].uuidList.size(); j++)
      {
         BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): sppCapabilities[%u]: uuidList[%u]: uuid=%s",
                  getMessageTraceId(),
                  getBts2AppOpcode2String(_opCode),
                  i,
                  j,
                  _discoveredDevice.sppCapabilities[i].uuidList[j].c_str() /*DONE: _sppCapabilities format*/));
      }
   }

#endif
}

void Bts2App_StartPairingResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result)));

#endif
}

void Bts2App_PinCodeRequested::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X pairingOriginator=%d deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSPairConnectOriginator, _pairingOriginator),
            _deviceName.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X pairingOriginator=%s deviceName=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getPairConnectOriginator2String(_pairingOriginator),
            _deviceName.c_str()));

#endif
}

void Bts2App_SecureSimplePairingRequested::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            _deviceName.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): sspMode=%d pairingOriginator=%d numericValue=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            ETG_ENUM(TRC_BTSSspMode, _sspMode),
            ETG_ENUM(TRC_BTSPairConnectOriginator, _pairingOriginator),
            _numericValue.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X deviceName=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            _deviceName.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): sspMode=%s pairingOriginator=%s numericValue=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            getSspMode2String(_sspMode),
            getPairConnectOriginator2String(_pairingOriginator),
            _numericValue.c_str() /*DONE: format*/));

#endif
}

void Bts2App_PairingFinished::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d linkKeyType=%d linkKey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            ETG_ENUM(TRC_BTSLinkKeyType, _linkKeyType),
            _linkKey.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): majorServiceClass=0x%04X majorDeviceClass=%d minorDeviceClass=%d dLinkKey=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _majorServiceClass.getData() /*DONE: format*/,
            ETG_ENUM(TRC_BTSMajorDeviceClass, _majorDeviceClass),
            ETG_ENUM(TRC_BTSMinorDeviceClass, _minorDeviceClass),
            _dLinkKey.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _deviceName.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s linkKeyType=%s linkKey=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            getLinkKeyType2String(_linkKeyType),
            _linkKey.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): majorServiceClass=0x%04X majorDeviceClass=%s minorDeviceClass=%s dLinkKey=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _majorServiceClass.getData() /*DONE: format*/,
            getMajorDeviceClass2String(_majorDeviceClass),
            getMinorDeviceClass2String(_minorDeviceClass),
            _dLinkKey.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): deviceName=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _deviceName.c_str() /*DONE: format*/));

#endif
}

void Bts2App_LinkKeyRequested::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower()));

#endif
}

void Bts2App_ConnectProtocolResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d protocolId=%d sppUuid=%32s masInstance=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _sppUuid.c_str() /*DONE: format*/,
            _masInstance.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s protocolId=%s sppUuid=%32s masInstance=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            getProtocolId2String(_protocolId),
            _sppUuid.c_str() /*DONE: format*/,
            _masInstance.c_str() /*DONE: format*/));

#endif
}

void Bts2App_RemoteProtocolConnectRequest::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X protocolId=%d sppUuid=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _sppUuid.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X protocolId=%s sppUuid=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getProtocolId2String(_protocolId),
            _sppUuid.c_str()));

#endif
}

void Bts2App_DisconnectProtocolResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d protocolId=%d sppUuid=%32s masInstance=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _sppUuid.c_str() /*DONE: format*/,
            _masInstance.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s protocolId=%s sppUuid=%32s masInstance=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            getProtocolId2String(_protocolId),
            _sppUuid.c_str() /*DONE: format*/,
            _masInstance.c_str() /*DONE: format*/));

#endif
}

void Bts2App_DisconnectDeviceResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result)));

#endif
}

void Bts2App_ProtocolConnectionStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X connectionStatus=%d disconnectReason=%d protocolId=%d sppUuid=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSConnectionStatus, _connectionStatus),
            ETG_ENUM(TRC_BTSDisconnectReason, _disconnectReason),
            ETG_ENUM(TRC_BTSProtocolId, _protocolId),
            _sppUuid.c_str() /*DONE: format*/));
   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): masInstance=%100s rfCommDevice=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _masInstance.c_str() /*DONE: format*/,
            _rfCommDevice.c_str() /*DONE: format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X connectionStatus=%s disconnectReason=%s protocolId=%s sppUuid=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getConnectionStatus2String(_connectionStatus),
            getDisconnectReason2String(_disconnectReason),
            getProtocolId2String(_protocolId),
            _sppUuid.c_str() /*DONE: format*/));
   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): masInstance=%100s rfCommDevice=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _masInstance.c_str() /*DONE: format*/,
            _rfCommDevice.c_str() /*DONE: format*/));

#endif
}

void Bts2App_DeviceConnectionStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X connectionStatus=%d disconnectReason=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSConnectionStatus, _connectionStatus),
            ETG_ENUM(TRC_BTSDisconnectReason, _disconnectReason)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X connectionStatus=%s disconnectReason=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getConnectionStatus2String(_connectionStatus),
            getDisconnectReason2String(_disconnectReason)));

#endif
}

void Bts2App_DeviceCapabilities::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X inbandRingtoneSupported=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSBool, _deviceCapabilities.inbandRingtoneSupported) /*DONE: _deviceCapabilities format*/));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X inbandRingtoneSupported=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getBool2String(_deviceCapabilities.inbandRingtoneSupported) /*DONE: _deviceCapabilities format*/));

#endif
}

void Bts2App_RemoteSupportedServices::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d supportedServices=0x%04X serviceInfoList.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            _supportedServices.getData() /*DONE: format*/,
            _serviceInfoList.size() /*DONE: format*/));
   for(size_t i = 0; i < _serviceInfoList.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): serviceInfoList[%u]: service=%d version=0x%04X",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               ETG_ENUM(TRC_BTSSupportedServicesBit, _serviceInfoList[i].service),
               _serviceInfoList[i].version /*DONE: _serviceInfoList format*/));
   }

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s supportedServices=0x%04X serviceInfoList.size()=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            _supportedServices.getData() /*DONE: format*/,
            _serviceInfoList.size() /*DONE: format*/));
   for(size_t i = 0; i < _serviceInfoList.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): serviceInfoList[%u]: service=%s version=0x%04X",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               getSupportedServicesBit2String(_serviceInfoList[i].service),
               _serviceInfoList[i].version /*DONE: _serviceInfoList format*/));
   }

#endif
}

void Bts2App_RemoteSppCapabilities::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d sppCapabilities.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            _sspCapabilities.size() /*DONE: format*/));
   for(size_t i = 0; i < _sspCapabilities.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): sppCapabilities[%u]: uuidList.size()=%u serviceName=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _sspCapabilities[i].uuidList.size(),
               _sspCapabilities[i].serviceName.c_str() /*DONE: _sppCapabilities format*/));
      for(size_t j = 0; j < _sspCapabilities[i].uuidList.size(); j++)
      {
         ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): sppCapabilities[%u]: uuidList[%u]: uuid=%s",
                  getMessageTraceId(),
                  ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
                  i,
                  j,
                  _sspCapabilities[i].uuidList[j].c_str() /*DONE: _sppCapabilities format*/));
      }
   }

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s sppCapabilities.size()=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            _sspCapabilities.size() /*DONE: format*/));
   for(size_t i = 0; i < _sspCapabilities.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): sppCapabilities[%u]: uuidList.size()=%u serviceName=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _sspCapabilities[i].uuidList.size(),
               _sspCapabilities[i].serviceName.c_str() /*DONE: _sppCapabilities format*/));
      for(size_t j = 0; j < _sspCapabilities[i].uuidList.size(); j++)
      {
         BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): sppCapabilities[%u]: uuidList[%u]: uuid=%s",
                  getMessageTraceId(),
                  getBts2AppOpcode2String(_opCode),
                  i,
                  j,
                  _sspCapabilities[i].uuidList[j].c_str() /*DONE: _sppCapabilities format*/));
      }
   }

#endif
}

void Bts2App_RemoteDeviceIdServiceRecords::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d deviceIdServiceRecords.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            _deviceIdServiceRecords.size() /*DONE: format*/));
   for(size_t i = 0; i < _deviceIdServiceRecords.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): deviceIdServiceRecords[%u]: specificationID=0x%04X vendorID=0x%04X productID=0x%04X version=0x%04X primaryRecord=%d vendorIDSource=0x%04X clientExecutableURL=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _deviceIdServiceRecords[i].specificationID,
               _deviceIdServiceRecords[i].vendorID,
               _deviceIdServiceRecords[i].productID,
               _deviceIdServiceRecords[i].version,
               ETG_ENUM(TRC_BTSBool, _deviceIdServiceRecords[i].primaryRecord),
               _deviceIdServiceRecords[i].vendorIDSource,
               _deviceIdServiceRecords[i].clientExecutableURL.c_str() /*DONE: _deviceIdServiceRecords format*/));
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): deviceIdServiceRecords[%u]: serviceDescription=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _deviceIdServiceRecords[i].serviceDescription.c_str() /*DONE: _deviceIdServiceRecords format*/));
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): deviceIdServiceRecords[%u]: clientDocumentationURL=%s",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _deviceIdServiceRecords[i].clientDocumentationURL.c_str() /*DONE: _deviceIdServiceRecords format*/));
   }

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s deviceIdServiceRecords.size()=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            _deviceIdServiceRecords.size() /*DONE: format*/));
   for(size_t i = 0; i < _deviceIdServiceRecords.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): deviceIdServiceRecords[%u]: specificationID=0x%04X vendorID=0x%04X productID=0x%04X version=0x%04X primaryRecord=%s vendorIDSource=0x%04X clientExecutableURL=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _deviceIdServiceRecords[i].specificationID,
               _deviceIdServiceRecords[i].vendorID,
               _deviceIdServiceRecords[i].productID,
               _deviceIdServiceRecords[i].version,
               getBool2String(_deviceIdServiceRecords[i].primaryRecord),
               _deviceIdServiceRecords[i].vendorIDSource,
               _deviceIdServiceRecords[i].clientExecutableURL.c_str() /*DONE: _deviceIdServiceRecords format*/));
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): deviceIdServiceRecords[%u]: serviceDescription=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _deviceIdServiceRecords[i].serviceDescription.c_str() /*DONE: _deviceIdServiceRecords format*/));
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): deviceIdServiceRecords[%u]: clientDocumentationURL=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _deviceIdServiceRecords[i].clientDocumentationURL.c_str() /*DONE: _deviceIdServiceRecords format*/));
   }

#endif
}

void Bts2App_RemoteServiceSearchResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X supportedServicesResult=%d sppCapabilitiesResult=%d deviceIdServiceRecordsResult=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _supportedServicesResult),
            ETG_ENUM(TRC_BTSRequestResult, _sppCapabilitiesResult),
            ETG_ENUM(TRC_BTSRequestResult, _deviceIdServiceRecordsResult)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X supportedServicesResult=%s sppCapabilitiesResult=%s deviceIdServiceRecordsResult=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_supportedServicesResult),
            getRequestResult2String(_sppCapabilitiesResult),
            getRequestResult2String(_deviceIdServiceRecordsResult)));

#endif
}

void Bts2App_RemoteNameResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            _deviceName.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s deviceName=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            _deviceName.c_str()));

#endif
}

void Bts2App_RemoteNameStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X deviceName=%s",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            _deviceName.c_str()));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X deviceName=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            _deviceName.c_str()));

#endif
}

void Bts2App_LinkQualityResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X result=%d rssi=%d linkQuality=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSRequestResult, _result),
            _rssi,
            _linkQuality));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X result=%s rssi=%d linkQuality=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getRequestResult2String(_result),
            _rssi,
            _linkQuality));

#endif
}

void Bts2App_StartTestModeResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_StopTestModeResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_TestModeStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   BTSBDAddressTrace traceAddr(_deviceAddress);

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u btAddr=%08X%04X connectionStatus=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            ETG_ENUM(TRC_BTSConnectionStatus, _connectionStatus)));

#else

   BTSBDAddressTrace traceAddr(_deviceAddress);

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u btAddr=%08X%04X connectionStatus=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            traceAddr.getUpper(), traceAddr.getLower(),
            getConnectionStatus2String(_connectionStatus)));

#endif
}

void Bts2App_TestModeLinkQualityResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d rssi=%d linkQuality=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result),
            _rssi,
            _linkQuality));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s rssi=%d linkQuality=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result),
            _rssi,
            _linkQuality));

#endif
}

void Bts2App_FbTelephonyInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetTelephonyDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_FbPhonebookInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetPhonebookDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_FbMessagingInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetMessagingDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_FbMediaPlayerInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetMediaPlayerDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_FbWblInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetWblDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_AllowWblCommunicationResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle));

#endif
}

void Bts2App_BlockWblCommunicationResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle));

#endif
}

void Bts2App_WblCommunicationStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u enabled=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSBool, _enabled)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u enabled=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getBool2String(_enabled)));

#endif
}

void Bts2App_WblServiceAvailability::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u serviceAvailable=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSBool, _serviceAvailable)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u serviceAvailable=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getBool2String(_serviceAvailable)));

#endif
}

void Bts2App_WblLastIntendedMode::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u lastIntendedModeList.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            _lastIntendedModeList.size() /*DONE: format*/));
   for(size_t i = 0; i < _lastIntendedModeList.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): lastIntendedModeList[%u]: id=%u mode=%d powerState=%d frequency=%d type=%d",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _lastIntendedModeList[i].id,
               ETG_ENUM(TRC_BTSWblWifiMode, _lastIntendedModeList[i].status.mode),
               ETG_ENUM(TRC_BTSWblWifiPowerState, _lastIntendedModeList[i].status.powerState),
               ETG_ENUM(TRC_BTSWblWifiFrequency, _lastIntendedModeList[i].status.frequency),
               ETG_ENUM(TRC_BTSWblAccessPointConfigType, _lastIntendedModeList[i].status.type) /*DONE: _lastIntendedModeList format*/));
   }

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u lastIntendedModeList.size()=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            _lastIntendedModeList.size() /*DONE: format*/));
   for(size_t i = 0; i < _lastIntendedModeList.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): lastIntendedModeList[%u]: id=%u mode=%s powerState=%s frequency=%s type=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _lastIntendedModeList[i].id,
               getWblWifiMode2String(_lastIntendedModeList[i].status.mode),
               getWblWifiPowerState2String(_lastIntendedModeList[i].status.powerState),
               getWblWifiFrequency2String(_lastIntendedModeList[i].status.frequency),
               getWblAccessPointConfigType2String(_lastIntendedModeList[i].status.type) /*DONE: _lastIntendedModeList format*/));
   }

#endif
}

void Bts2App_WblSupportedRestrictions::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u supportedRestrictionList.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            _supportedRestrictionList.size() /*DONE: format*/));
   for(size_t i = 0; i < _supportedRestrictionList.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): supportedRestrictionList[%u]: id=%u supportedRestrictions.size()=%u",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _supportedRestrictionList[i].id,
               _supportedRestrictionList[i].supportedRestrictions.size() /*DONE: _supportedRestrictionList format*/));
      for(size_t j = 0; j < _supportedRestrictionList[i].supportedRestrictions.size(); j++)
      {
         BTSWblSupportedRestriction& restriction = _supportedRestrictionList[i].supportedRestrictions[j];
         ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): supportedRestrictionList[%u][%u]: frequency=%d technologies.size()=%u",
                  getMessageTraceId(),
                  ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
                  i,
                  j,
                  ETG_ENUM(TRC_BTSWblWifiFrequency, restriction.frequency),
                  restriction.technologies.size() /*DONE: _supportedRestrictionList format*/));
         for(size_t k = 0; k < restriction.technologies.size(); k++)
         {
            BTSWblSupportedTechnology& technology = restriction.technologies[k];
            ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): supportedRestrictionList[%u][%u][%u]: frequency=%d type=%d setting=%d",
                     getMessageTraceId(),
                     ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
                     i,
                     j,
                     k,
                     ETG_ENUM(TRC_BTSWblWifiFrequency, restriction.frequency),
                     ETG_ENUM(TRC_BTSWblAccessPointConfigType, technology.type),
                     ETG_ENUM(TRC_BTSWblSupportedSetting, technology.setting) /*DONE: _supportedRestrictionList format*/));
         }
      }
   }

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u supportedRestrictionList.size()=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            _supportedRestrictionList.size() /*DONE: format*/));
   for(size_t i = 0; i < _supportedRestrictionList.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): supportedRestrictionList[%u]: id=%u supportedRestrictions.size()=%u",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _supportedRestrictionList[i].id,
               _supportedRestrictionList[i].supportedRestrictions.size() /*DONE: _supportedRestrictionList format*/));
      for(size_t j = 0; j < _supportedRestrictionList[i].supportedRestrictions.size(); j++)
      {
         BTSWblSupportedRestriction& restriction = _supportedRestrictionList[i].supportedRestrictions[j];
         BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): supportedRestrictionList[%u][%u]: frequency=%s technologies.size()=%u",
                  getMessageTraceId(),
                  getBts2AppOpcode2String(_opCode),
                  i,
                  j,
                  getWblWifiFrequency2String(restriction.frequency),
                  restriction.technologies.size() /*DONE: _supportedRestrictionList format*/));
         for(size_t k = 0; k < restriction.technologies.size(); k++)
         {
            BTSWblSupportedTechnology& technology = restriction.technologies[k];
            BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): supportedRestrictionList[%u][%u][%u]: frequency=%s type=%s setting=%s",
                     getMessageTraceId(),
                     getBts2AppOpcode2String(_opCode),
                     i,
                     j,
                     k,
                     getWblWifiFrequency2String(restriction.frequency),
                     getWblAccessPointConfigType2String(technology.type),
                     getWblSupportedSetting2String(technology.setting) /*DONE: _supportedRestrictionList format*/));
         }
      }
   }

#endif
}

void Bts2App_WblWiFiSetupStatus::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u wifiSetupStatusList.size()=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            _wifiSetupStatusList.size() /*DONE: format*/));
   for(size_t i = 0; i < _wifiSetupStatusList.size(); i++)
   {
      ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): wifiSetupStatusList[%u]: id=%u mode=%d powerState=%d frequency=%d type=%d",
               getMessageTraceId(),
               ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
               i,
               _wifiSetupStatusList[i].id,
               ETG_ENUM(TRC_BTSWblWifiMode, _wifiSetupStatusList[i].status.mode),
               ETG_ENUM(TRC_BTSWblWifiPowerState, _wifiSetupStatusList[i].status.powerState),
               ETG_ENUM(TRC_BTSWblWifiFrequency, _wifiSetupStatusList[i].status.frequency),
               ETG_ENUM(TRC_BTSWblAccessPointConfigType, _wifiSetupStatusList[i].status.type) /*DONE: _wifiSetupStatusList format*/));
   }

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u wifiSetupStatusList.size()=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            _wifiSetupStatusList.size() /*DONE: format*/));
   for(size_t i = 0; i < _wifiSetupStatusList.size(); i++)
   {
      BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): wifiSetupStatusList[%u]: id=%u mode=%s powerState=%s frequency=%s type=%s",
               getMessageTraceId(),
               getBts2AppOpcode2String(_opCode),
               i,
               _wifiSetupStatusList[i].id,
               getWblWifiMode2String(_wifiSetupStatusList[i].status.mode),
               getWblWifiPowerState2String(_wifiSetupStatusList[i].status.powerState),
               getWblWifiFrequency2String(_wifiSetupStatusList[i].status.frequency),
               getWblAccessPointConfigType2String(_wifiSetupStatusList[i].status.type) /*DONE: _wifiSetupStatusList format*/));
   }

#endif
}

void Bts2App_WblHealthinessIndicator::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u value=%u",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            _value));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u value=%u",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            _value));

#endif
}

void Bts2App_FbEcnrInitialized::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

void Bts2App_SetEcnrDummyResult::doOutputTrace(void)
{
#ifdef VARIANT_S_FTR_ENABLE_FW_ETG_USAGE

   ETG_TRACE_USR1((" Bts2App: 0x%08X (%d): SH=%u result=%d",
            getMessageTraceId(),
            ETG_ENUM(TRC_BTSBts2AppOpcode, _opCode),
            _sessionHandle,
            ETG_ENUM(TRC_BTSRequestResult, _result)));

#else

   BTS_TRACE_USR1((" Bts2App: 0x%08X (%s): SH=%u result=%s",
            getMessageTraceId(),
            getBts2AppOpcode2String(_opCode),
            _sessionHandle,
            getRequestResult2String(_result)));

#endif
}

} //btstackif
